package com.biz.crm.common.form.local.init;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.biz.crm.common.form.sdk.module.ModuleRegister;
import com.biz.crm.common.form.sdk.widget.WidgetKey;
import com.bizunited.nebula.init.service.InitProcessEnvironmentService;

import lombok.extern.slf4j.Slf4j;

/**
 * 动态表单功能在启动时需要进行验证，主要是对OperationStrategy接口各个实现类进行验证
 * @author yinwenjie
 */
@Slf4j
@Component
public class DynamicFormInitProcess implements InitProcessEnvironmentService {

  @Autowired(required = false)
  private List<ModuleRegister> moduleRegisters;
  
  @Autowired(required = false)
  private List<OperationStrategy<? extends DynamicForm>> operationStrateis;
  
  @Autowired
  private List<WidgetKey> widgetKeys;
  
  @Override
  public boolean doInitForAppCode(String appCode) {
    // 只有default性质的顶级租户才需要进行验证
    return StringUtils.equals(appCode, "default");
  }
  
  @Override
  public void initForAppCode(String appCode) {
    /*
     * 验证原则是：
     * 1、ModuleRegister各个实现中的moduleCode必须有值且是唯一的
     * 2、OperationStrategy各个实现中的dynamicFormCode必须有值且是唯一的
     * 3、OperationStrategy各个实现中的dynamicFormClass必须有值且是唯一的
     * 4、OperationStrategy各个实现中的moduleCode必须匹配已有的ModuleRegister注册内容
     * 5、WidgetKey各个实现中的code必须有值，且是唯一的
     * */
    if(CollectionUtils.isEmpty(moduleRegisters)) {
      log.error("动态表单模块没有发现有任何模块注册信息(ModuleRegister)!!");
      return;
    }
    if(CollectionUtils.isEmpty(operationStrateis)) {
      log.error("动态表单模块没有发现有任何操作策略注册信息(OperationStrategy)!!");
      return;
    }
    
    // 1、=====
    List<String> moduleCodes = this.moduleRegisters.stream().filter(item -> StringUtils.isNotBlank(item.moduleCode())).map(ModuleRegister::moduleCode).distinct().collect(Collectors.toList());
    Validate.isTrue(moduleCodes.size() == this.moduleRegisters.size() , "系统启动时，发现至少有一个ModuleRegister接口的实现不正确，请保证ModuleRegister的各个实现中moduleCode不为空，且唯一");
    // 2、======
    List<String> dynamicFormCodes = this.operationStrateis.stream().filter(item -> StringUtils.isNotBlank(item.dynamicFormCode())).map(OperationStrategy::dynamicFormCode).distinct().collect(Collectors.toList());
    Validate.isTrue(dynamicFormCodes.size() == this.operationStrateis.size() , "系统启动时，发现至少有一个ModuleRegister接口的实现不正确，请保证OperationStrategy的各个实现中dynamicFormCode不为空，且唯一");
    Map<String , OperationStrategy<? extends DynamicForm>> operationStrategyMapping = this.operationStrateis.stream().collect(Collectors.toMap(OperationStrategy::dynamicFormCode, item -> item));
    // 3、======
    Set<Class<? extends DynamicForm>> operationStrategyClassSet = this.operationStrateis.stream().filter(item -> item.dynamicFormClass() != null).map(OperationStrategy::dynamicFormClass).distinct().collect(Collectors.toSet());
    Validate.isTrue(operationStrategyClassSet.size() == this.operationStrateis.size() , "系统启动时，发现至少有一个ModuleRegister接口的实现不正确，请保证OperationStrategy的各个实现中dynamicFormClass不为空，且唯一");
    // 4、======
    for (Map.Entry<String,OperationStrategy<? extends DynamicForm>> entryItem : operationStrategyMapping.entrySet()) {
      String dynamicFormCode = entryItem.getKey();
      OperationStrategy<? extends DynamicForm> operationStrategy = entryItem.getValue();
      String moduleCode = operationStrategy.moduleCode();
      Validate.isTrue(moduleCodes.contains(moduleCode) , "OperationStrategy接口的一个具体实现：%s 没有正确设定moduleCode，请检查!!" , dynamicFormCode);
    }
    // 5、=====
    List<String> exsitWidgetKeys = this.widgetKeys.stream().filter(item -> StringUtils.isNotBlank(item.widgetCode())).map(WidgetKey::widgetCode).collect(Collectors.toList());
    Validate.isTrue(exsitWidgetKeys.size() == this.widgetKeys.size() , "系统启动时，发现至少有一个WidgetKey接口的实现不正确，请保证WidgetKey的各个实现中widgetCode不为空，且唯一");
  }
}
