package com.biz.crm.common.form.sdk;

import java.util.Collection;

import com.alibaba.fastjson.JSONObject;

/**
 * 动态表单内容操作的接口定义。其主要意义在于对固定业务下需要做动态表单（结构）部分的数据进行增、删、改和数据完善操作
 * @author yinwenjie
 * @param <T> 任意业务模型，例如：预算活动、订单、合同等业务的VO、DTO模型
 */
public interface DynamicFormService<T> {
  
  // ======== 以下方法定义和转换有关
  
  /**
   * 动态转换，将一个给定的jsonObject信息，试图转换为指定的模型结构；</br>
   * DynamicFormService中的dynamic将根据设定，在转换过程中完善模型结构中的动态表单结构
   * @param formJson 
   */
  public T dynamic(JSONObject formJson);
  
  // ======== 以下方法定义和表单模型（动态结构部分）的增删改操作有关
  
  /**
   * 在完成主模型T的添加操作中，操作者可以不对主模型T中的动态表单结构部分进行任何的添加操作；
   * 待主模型T中的不变部分完成添加后，再直接调用该方法完成主模型T中动态表单结构部分的添加操作。
   * @param formObject 主业务模型对象
   * @param parentCode 每一条将要保存的动态表单结构中的数据和主模型的业务对应关系（注意：是业务对应关系）
   */
  public void createDynamicDetails(T formObject , String parentCode);
  
  /**
   * 在完成主模型T的修改操作中，操作者可以不对主模型T中的动态表单结构部分进行任何的修改操作；
   * 待主模型T中的不变部分完成修改后，再直接调用该方法完成主模型T中动态表单结构部分的修改操作。
   * @param formObject 主业务模型对象
   * @param parentCode 每一条将要保存的动态表单结构中的数据和主模型的业务对应关系（注意：是业务对应关系）
   */
  public void modifyDynamicDetails(T formObject , String parentCode);
  
  /**
   * 在完成主模型T的删除操作中，操作者可以不对主模型T中的动态表单结构部分进行任何的删除操作；
   * 待主模型T中的不变部分完成删除后，再直接调用该方法完成主模型T中动态表单结构部分的删除操作。</p>
   * 
   * 请注意，该删除方式，将删除和这个主模型关联的（关联的值为parentCode），所有动态表单数据信息。
   * 如果操作者只要求删除和主模型关联的特定一部分信息（例如指定的某条动态表单数据），则可以使用另一个删除方法
   * 
   * @param parentCode 
   */
  public void deleteDynamicDetails(String parentCode);

  /**
   * TODO 注释未写
   * @param parentCode
   * @param dynamicFieldCode
   * @param dynamicKey
   * @param detailCodes
   */
  public void deleteDynamicDetails(String parentCode , String dynamicFieldCode, String dynamicKey, String[] detailCodes);
  
  // ======== 以下方法定义和表单模型（动态结构部分）的查询操作有关
  
  /**
   * 该方法用于进行指定动态表单信息的完善，常用于“查询指定主表单详细信息”的使用场景。
   * @param formObject
   * @param parentCode
   */
  public void perfectDynamicDetails(T formObject , String parentCode);
  
  /**
   * 按照主业务表父级code、主业务表字段名、动态表单key值，查询具体的业务数据——以单个对象的方式；</br>
   * 即适用于主业务表上，某个结构为Map< String , D >形式的映射动态表单的结构
   * @param parentCode 每一条将要保存的动态表单结构中的数据和主模型的业务对应关系（注意：是业务对应关系）
   * @param dynamicFieldCode 主业务表上的字段名
   * @param dynamicKey 动态表单对应的key值
   * @param dclass 动态表单对应的具体class
   */
  public <D> D findDetailByParentCodeAndDynamicFieldAndDynamicKey(String parentCode , String dynamicFieldCode, String dynamicKey , Class<D> dclass);
  
  /**
   * 按照主业务表父级code、主业务表字段名、动态表单key值，查询具体的业务数据——以集合对象的方式；</br>
   * 即适用于主业务表上，某个结构为Map< String , Collection<D> >形式的映射动态表单的结构
   * @param parentCode 每一条将要保存的动态表单结构中的数据和主模型的业务对应关系（注意：是业务对应关系）
   * @param dynamicFieldCode 主业务表上的字段名
   * @param dynamicKey 动态表单对应的key值
   * @param dclass 动态表单对应的具体class
   */
  public <D> Collection<D> findDetailsByParentCodeAndDynamicFieldAndDynamicKey(String parentCode , String dynamicFieldCode, String dynamicKey , Class<D> dclass);
}
