package com.biz.crm.common.ie.local.bean;

import cn.hutool.core.text.CharSequenceUtil;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ImportProcessEnum;
import com.biz.crm.common.ie.sdk.enums.TypeEnum;
import com.biz.crm.common.ie.sdk.vo.ImportProcessMsgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.gateway.websocket.client.sdk.service.ChannelMsgService;

import java.util.Date;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * 导入消息推送
 *
 * @author sunx
 * @date 2022/5/30
 */
@Component
public class ImportSendProcessMsgBean {

  @Value("${spring.application.name}")
  private String applicationName;

  @Autowired(required = false)
  private ChannelMsgService channelMsgService;
  /**
   * 导入导出模块中的导入信息
   */
  private static final String IE_IMPORT_MODEL_CODE = "ie_import_model";

  private static final Logger LOGGER = LoggerFactory.getLogger(ImportSendProcessMsgBean.class);

  /**
   * 发送消息（START，GET_PROCESS，DOWNLOAD，START_EXEC，ERROR_DATA，UPDATE_TASK）
   *
   * @param taskCode    任务编码
   * @param processEnum 任务执行阶段
   */
  public void sendProcessMsg(String taskCode, ImportProcessEnum processEnum, String account) {
    this.sendProcessMsg(taskCode, processEnum, ExecStatusEnum.RUNNING, account);
  }
  /**
   * 发送消息
   *
   * @param taskCode    任务编码
   * @param processEnum 任务执行阶段枚举对象
   * @param execStatus  执行状态枚举对象
   * @param account     账号
   */
  public void sendProcessMsg(String taskCode, ImportProcessEnum processEnum, ExecStatusEnum execStatusEnum, String account) {
    this.sendProcessMsg(taskCode, processEnum, execStatusEnum.getDictCode(), processEnum.getDesc(), account);
  }
  /**
   * 发送消息(END)
   *
   * @param taskCode    任务编码
   * @param processEnum 任务执行阶段
   * @param execStatus  执行状态
   * @param remark      备注信息
   */
  public void sendProcessMsg(
          String taskCode, ImportProcessEnum processEnum, String execStatus, String remark, String account) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(execStatus);
    vo.setProcessType(processEnum.getCode());
    vo.setRemark(remark);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendProcessMsg(vo);
  }

  /**
   * 发送消息(END)
   *
   * @param taskCode    任务编码
   * @param processEnum 任务执行阶段
   * @param execStatus  执行状态
   * @param remark      备注信息
   * @param importTask  导入文件信息
   */
  public void sendProcessMsg(
          String taskCode, ImportProcessEnum processEnum, String execStatus, String remark, String account, ImportTask importTask) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(execStatus);
    vo.setProcessType(processEnum.getCode());
    vo.setRemark(remark);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    //当导入文件失败时 ws携带失败文件
    vo.setFileCode(importTask.getFailedFileCode());
    this.sendProcessMsg(vo);
  }

  /**
   * 发送消息(RUNNING)
   *
   * @param taskCode   任务编码
   * @param total      任务总条数
   * @param cursor     进度
   * @param successNum 成功数量
   * @param failedNum  失败数量
   */
  public void sendProcessMsg(
          String taskCode, Integer total, Integer cursor, Integer successNum, Integer failedNum, String account) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    vo.setProcessType(ImportProcessEnum.RUNNING.getCode());
    vo.setPageSize(total);
    vo.setCursor(cursor);
    vo.setRemark(
            CharSequenceUtil.format(
                    ImportProcessEnum.RUNNING.getFormat(), total, cursor, successNum, failedNum));
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendProcessMsg(vo);
  }

  /**
   * 发送消息(END_EXEC)
   *
   * @param taskCode   任务编码
   * @param successNum 成功数量
   * @param failedNum  失败数量
   * @param total      总量
   */
  public void sendProcessMsg(
          String taskCode, Integer successNum, Integer failedNum, Integer total, String account) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    vo.setProcessType(ImportProcessEnum.END_EXEC.getCode());
    vo.setRemark(
            CharSequenceUtil.format(
                    ImportProcessEnum.END_EXEC.getFormat(), total, successNum, failedNum));
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendProcessMsg(vo);
  }

  /**
   * 校验及保存时消息推送
   * @param taskCode    任务编码
   * @param total       任务总条数
   * @param cursor      进度
   * @param successNum  成功数量
   * @param failedNum   失败数量
   * @param account     当前账户
   * @param process     执行情况
   */
  public void sendProcessMsg(
          String taskCode, Integer total, Integer cursor, Integer current,Integer successNum, Integer failedNum, String account, ImportProcessEnum process) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    vo.setProcessType(process.getCode());
    vo.setPageSize(total);
    vo.setCursor(current);
    vo.setRemark(
            CharSequenceUtil.format(
                    process.getFormat(), total, cursor, successNum, failedNum));
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendProcessMsg(vo);
  }

  /**
   * 发送失败信息消息(FAILED)
   *
   * @param taskCode 任务编码
   * @param msg      进度消息
   */
  public void sendFailedProcessMsg(String taskCode, String msg, String account) {
    ImportProcessMsgVo vo = new ImportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setIeType(TypeEnum.IMPORT.getDictCode());
    vo.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    vo.setProcessType(ImportProcessEnum.END.getCode());
    vo.setRemark(msg);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendProcessMsg(vo);
  }

  /**
   * 发送进度消息
   *
   * @param vo 进度消息实体
   */
  private void sendProcessMsg(ImportProcessMsgVo vo) {
    vo.setTime(new Date());
    byte[] jsonBytes = JSONObject.toJSONBytes(vo);
    String tenantCode = TenantUtils.getTenantCode();
    //导入导出公用通道
    String ieChannelTaskCode = ImportExportConstant.IE_CHANNEL_TASK_CODE;
    //所有业务所属通道标识
    String exportAndImportModelCode = ImportExportConstant.IE_EXPORT_IMPORT_MODEL_CODE;
    LOGGER.info(" import model send msg: tenantCode = {} , applicationName = {} , modelCode = {} ", tenantCode, applicationName, IE_IMPORT_MODEL_CODE);
    LOGGER.info(" import model send msg2: tenantCode = {} , applicationName = {} , modelCode = {} ", tenantCode, ieChannelTaskCode, exportAndImportModelCode);
    if (StringUtils.isNotBlank(vo.getAccount())) {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, applicationName, IE_IMPORT_MODEL_CODE, vo.getAccount(), jsonBytes);
      //公共导入通道
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, ieChannelTaskCode, exportAndImportModelCode, vo.getAccount(), jsonBytes);
    } else {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, applicationName, IE_IMPORT_MODEL_CODE, jsonBytes);
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, ieChannelTaskCode, exportAndImportModelCode, jsonBytes);
    }
  }

  /**
   * 发送进度消息
   *
   * @param vo 进度消息实体
   */
  public void sendMsgByAll(ImportProcessMsgVo vo) {
    vo.setTime(new Date());
    byte[] jsonBytes = JSONObject.toJSONBytes(vo);
    String tenantCode = TenantUtils.getTenantCode();
    //导入导出公用通道
    String ieChannelTaskCode = ImportExportConstant.IE_CHANNEL_TASK_CODE;
    //所有业务所属通道标识
    String exportAndImportModelCode = ImportExportConstant.IE_EXPORT_IMPORT_MODEL_CODE;
    LOGGER.info(" import model send msg2: tenantCode = {} , applicationName = {} , modelCode = {} ", tenantCode, ieChannelTaskCode, exportAndImportModelCode);
    if (StringUtils.isNotBlank(vo.getAccount())) {
      //公共导入通道
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, ieChannelTaskCode, exportAndImportModelCode, vo.getAccount(), jsonBytes);
    } else {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, ieChannelTaskCode, exportAndImportModelCode, jsonBytes);
    }
  }
}
