package com.biz.crm.common.ie.local.config;

import cn.hutool.core.io.unit.DataSize;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.convert.DataSizeUnit;
import org.springframework.util.unit.DataUnit;

/**
 * 导入导出配置信息
 *
 * @author sunx
 * @date 2022/5/10
 */
@Getter
@Setter
@ToString
@ConfigurationProperties(prefix = "crm.common.ie")
public class ImportExportProperties {

    // ##################################################
    /**
     * 是否执行导入导出任务
     */
    private Boolean execTask = Boolean.FALSE;
    /**
     * 导入导出临时文件存放根路径
     */
    private String root = "";
    /**
     * 任务组名称
     */
    private String taskGroup = "";

    /**
     * 任务扫描间隔,单位毫秒
     */
    private long taskInterval = 5000;

    /**
     * 导入任务等候时间，单位毫秒
     **/
    private long importSleep = 0;

    /**
     * 任务执行后休眠时长,默认500毫秒
     */
    private long taskExecutionIntervalSleep = 500;

    /**
     * spring boot应用程序名
     */
    @Value("${spring.application.name}")
    private String applicationName;

    /**
     * 当前JVM可以使用的处理器内核数
     */
    public static final int NCPU = Runtime.getRuntime().availableProcessors();

    /**
     * 获取任务名称
     */
    public String getIeTaskGroup() {
        StringBuilder sb = new StringBuilder();
        sb.append(applicationName).append(":ie");
        if (StringUtils.isNotBlank(taskGroup)) {
            if (taskGroup.startsWith("ie-")) {
                taskGroup = taskGroup.substring("ie-".length());
            }
            sb.append("-").append(taskGroup);
        }
        return sb.toString();
    }

    /**
     * 导入导出任务线程池隔离级别，true-应用级，false-顶级租户级
     */
    private Boolean appFlag = true;

    /**
     * 导入配置参数
     */
    private ImportProperties importProperties = new ImportProperties();

    /**
     * 导处入配置参数
     */
    private ExportProperties exportProperties = new ExportProperties();

    /**
     * 导处异步获取数据参数
     */
    private ExportAsyncGetTotalProperties exportAsyncGetDataProperties = new ExportAsyncGetTotalProperties();

    /**
     * 导出ES分段配置参数
     */
    private ExportEsParagraphProperties exportEsParagraphProperties = new ExportEsParagraphProperties();

    /**
     * 导出配置参数
     */
    @Data
    public static class ImportProperties {
        /**
         * 导入文件最大限制 xx M
         */
        @DataSizeUnit(DataUnit.MEGABYTES)
        private DataSize excelMaxSize = DataSize.ofMegabytes(40);
        /**
         * 导入并行任务数量，默认5个，请根据服务器CPU内核内存数进行调整，避免执行过程中频繁FullGC和内核切换
         */
        private int taskCount = 5;
        /**
         * 大任务的记录行数,默认20000
         */
        private int bigTaskRowNum = 20000;
    }

    /**
     * 导出配置参数
     */
    @Data
    public static class ExportProperties {
        /**
         * 同一个文件导出数量最大限制 xx 条
         */
        private int excelMaxCount = 500000;
        /**
         * 导入并行任务数量，默认5个，请根据服务器CPU内核内存数进行调整，避免执行过程中频繁FullGC和内核切换
         */
        private int taskCount = 5;
        /**
         * 大任务的记录行数,默认20000
         */
        private int bigTaskRowNum = 20000;
        /**
         * 数据查询分页大小,默认20000
         */
        private int queryPageSize = 1000;
        /**
         * 获取合并文件最大条数(超过该数值将不会合并文件,如果是小于等于0则不会限制无论多少条都会合并)
         **/
        private int mergeFileMaxCount = -1;
    }

    /**
     * 导出异步获取数据配置
     */
    @Data
    public static class ExportAsyncGetTotalProperties {
        private int taskCount = 3;
    }

    /**
     * 导出异步获取数据配置
     */
    @Data
    public static class ExportEsParagraphProperties {
        private int taskCount = 3;
    }
}
