package com.biz.crm.common.ie.local.repository;

import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.conditions.update.LambdaUpdateChainWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.mapper.ImportTaskMapper;
import com.biz.crm.common.ie.local.model.dto.ImportTaskModelDto;
import com.biz.crm.common.ie.local.model.vo.ImportTaskUpdateModelVo;
import com.biz.crm.common.ie.sdk.dto.ImportTaskPaginationDto;
import com.biz.crm.common.ie.sdk.enums.CallbackStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.datasource.configuration.MultiDataSourceProperties;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 导入任务信息(repository)
 *
 * @author sunx
 * @date 2022-05-12 09:25:17
 */
@Component
public class ImportTaskRepository extends ServiceImpl<ImportTaskMapper, ImportTask> {
  @Autowired
  private MultiDataSourceProperties multiDataSourceProperties;

  /**
   * 分页
   *
   * @param page 分页
   * @param dto 查询参数
   * @return
   */
  public Page<ImportTask> findByConditions(Page<ImportTask> page, ImportTaskPaginationDto dto) {
    if (multiDataSourceProperties.getSchemaForTenant()) {
      dto.setAppCode(TenantUtils.getTenantCode());
    }
    return this.baseMapper.findByConditions(page, dto);
  }

  /**
   * 根据id查询详情
   *
   * @param id 主键
   * @return
   */
  public ImportTask findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据任务编码获取任务信息
   *
   * @param taskCode 任务编码
   * @return
   */
  public ImportTask findByTaskCode(String taskCode) {
    LambdaQueryChainWrapper<ImportTask> lambdaQuery = this.lambdaQuery();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaQuery.eq(ImportTask::getAppCode, TenantUtils.getTenantCode());
    }
    return lambdaQuery.eq(ImportTask::getTaskCode, taskCode).one();
  }

  /**
   * 更新导入执行任务
   *
   * @param vo 导入任务
   */
  public void updateByImportTaskUpdateModelVo(ImportTaskUpdateModelVo vo) {
    if (vo == null || StringUtils.isBlank(vo.getTaskCode())) {
      return;
    }
    ImportTask importTask = this.findByTaskCode(vo.getTaskCode());
    importTask.setExecStatus(vo.getExecStatus());
    if (Objects.nonNull(vo.getTotal())) {
      importTask.setTotal(vo.getTotal());
    }
    if (Objects.nonNull(vo.getFailedNum())) {
      importTask.setFailedNum(vo.getFailedNum());
    }
    if (StringUtils.isNotBlank(vo.getFailedFileCode())) {
      importTask.setFailedFileCode(vo.getFailedFileCode());
    }
    if (StringUtils.isNotBlank(vo.getRemark())) {
      importTask.setRemark(vo.getRemark());
    }
    if (StringUtils.isNotBlank(vo.getLoadStatus())) {
      importTask.setLoadStatus(vo.getLoadStatus());
    }
    if (multiDataSourceProperties.getSchemaForTenant()) {
      importTask.setAppCode(TenantUtils.getTenantCode());
    }
    this.updateById(importTask);
  }

  /**
   * 更新导入任务加载状态
   *
   * @param taskCodes 任务编码集合
   * @param loadStatus 加载状态
   */
  public void updateLocalStatus(Set<String> taskCodes, String loadStatus) {
    LambdaUpdateChainWrapper<ImportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ImportTask::getAppCode, TenantUtils.getTenantCode());
    }
    lambdaUpdate
        .in(ImportTask::getTaskCode, taskCodes)
        .set(ImportTask::getLoadStatus, loadStatus)
        .update();
  }

  /**
   * 更新回调状态
   *
   * @param ids 主键集合
   * @param callbackStatusEnum 回调状态
   */
  public void updateCallBackStatusByIds(List<String> ids, CallbackStatusEnum callbackStatusEnum) {
    LambdaUpdateChainWrapper<ImportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ImportTask::getAppCode, TenantUtils.getTenantCode());
    }
    lambdaUpdate
        .in(ImportTask::getId, ids)
        .set(ImportTask::getCallBackStatus, callbackStatusEnum.getDictCode())
        .update();
  }

  /**
   * 根据ImportTaskModelDto获取任务信息
   *
   * @param dto 导入任务model实体
   * @return
   */
  public List<ImportTask> findByImportTaskModelDto(ImportTaskModelDto dto) {
    if (multiDataSourceProperties.getSchemaForTenant()) {
      dto.setAppCode(TenantUtils.getTenantCode());
    }
    return this.baseMapper.findByImportTaskModelDto(dto);
  }
}
