package com.biz.crm.common.ie.local.service.internal;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.MultipartFileUtil;
import com.biz.crm.common.ie.local.bean.ExportSendProcessMsgBean;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.entity.ExportTaskDetail;
import com.biz.crm.common.ie.local.model.dto.ExportTaskProcessModelDto;
import com.biz.crm.common.ie.local.model.vo.ExportTaskDetailUpdateModelVo;
import com.biz.crm.common.ie.local.repository.ExportTaskDetailRepository;
import com.biz.crm.common.ie.local.repository.ExportTaskRepository;
import com.biz.crm.common.ie.local.service.ExportTaskService;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.enums.CallbackStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExportDetailProcessEnum;
import com.biz.crm.common.ie.sdk.enums.ExportProcessEnum;
import com.biz.crm.common.ie.sdk.event.ExportNotifyEventListener;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelColumn;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelExport;
import com.biz.crm.common.ie.sdk.excel.process.AbstractEsParagraphExportProcess;
import com.biz.crm.common.ie.sdk.excel.process.ExportProcess;
import com.biz.crm.common.ie.sdk.excel.strategy.CrmExcelProcessStrategy;
import com.biz.crm.common.ie.sdk.excel.vo.FunctionPermissionVo;
import com.biz.crm.common.ie.sdk.exception.CrmExportException;
import com.biz.crm.common.ie.sdk.service.ExportProcessService;
import com.biz.crm.common.ie.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.mars.sdk.context.MarsAuthorityContext;
import com.bizunited.nebula.mars.sdk.context.MarsAuthorityContextHolder;
import com.bizunited.nebula.venus.sdk.dto.Base64UploadDto;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 导出操作实现
 *
 * @author sunx
 * @date 2022/5/19
 */
@Service
@Slf4j
public class ExportProcessServiceImpl implements ExportProcessService {
  @Autowired
  private ExportTaskService exportTaskService;
  @Autowired
  private ExportTaskDetailRepository exportTaskDetailRepository;
  @Autowired(required = false)
  private ExportTaskRepository exportTaskRepository;
  @Autowired
  private LoginUserService loginUserService;
  @Autowired
  private FileHandleService fileHandleService;
  @Autowired
  private ExportSendProcessMsgBean exportSendProcessMsgBean;
  @Autowired
  private CrmExcelProcessStrategy crmExcelProcessStrategy;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private ImportExportProperties importExportProperties;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<ExportNotifyEventListener> notifyEventListeners;
  @Value("${spring.application.name:}")
  private String subsystem;
  /** 上传导出子文件 */
  private static final Integer EXPORT_FILE_UPLOAD = 91;
  /** * 完全结束进度 */
  private static final Integer TOTAL = 100;
  private static final String ERR_MSG_FORMAT = "{}文件上传失败";

  /**
   * exportProcesses缓存
   */
  private static Map<String, ExportProcess<?>> exportProcessCacheMap = Maps.newHashMap();
  
  @Getter
  private boolean haveAsyncGetTotalProcess = false;
  
  @Autowired(required = true)
  public void setExportProcesses(List<ExportProcess<?>> exportProcesses) {
    for (ExportProcess<?> exportProcess : exportProcesses) {
      boolean asyncGetTotal = exportProcess.isAsyncGetTotal();
      if (asyncGetTotal == true) {
        haveAsyncGetTotalProcess = true;
        log.info("导出基座{}，异步获取任务总数", exportProcess.getBusinessCode());
      }
      exportProcessCacheMap.put(exportProcess.getBusinessCode(), exportProcess);
    }
  }

  /**
   * 获取对应的数据处理器
   *
   * @param businessCode
   * @return
   */
  @Override
  public ExportProcess<?> findExportProcess(String businessCode) {
    if (CollectionUtils.isEmpty(exportProcessCacheMap) || StringUtils.isBlank(businessCode)) {
      return null;
    }
    ExportProcess<?> exportProcess = exportProcessCacheMap.get(businessCode);
    return exportProcess;
  }
  
  @Override
  public void exportProcess(IeExecutorVoForExport vo) {
    try {
      // 推送进度信息
      this.exportSendProcessMsgBean.sendDetailTaskProcessMsg(vo.getExportTaskDetailCode(),
          ExecStatusEnum.RUNNING.getDictCode(), ExportDetailProcessEnum.START, vo.getAccount());
      //推送当前任务的排队位置 为0
//      if(ObjectUtils.isNotEmpty(exportTaskQueueDto)){
//        exportSendProcessMsgBean.sendTaskProcessQueueMsg(exportTaskQueueDto , 0);
//      }

      // step01:导出任务预处理
      ExportTaskProcessVo task = this.exportTaskPrevHandler(vo.getExportTaskDetailCode());
      // step02: 构建MarsAuthorityContext过滤器,对数据权限进行过滤
      // 不止要属性当前执行线程的用户身份上下文，还要根据可能的marscodelist，设置数据权限上下文
      // 数据权限：创建上下文，并存入本次请求请求需要处理的listCode信息
      MarsAuthorityContext marsAuthorityContext = MarsAuthorityContextHolder.getContext();
      String marsListCode = task.getMarsListCode();
      marsAuthorityContext.setListCode(marsListCode);
      // marsAuthorityContext.setBuilding(true);
      if (StringUtils.isNotBlank(task.getMarsListCode())) {
        log.info("========== 已设定数据导出子任务匹配的数据权限：{}", task.getMarsListCode());
      } else {
        log.info("========== 未设定数据导出子任务匹配的数据权限");
      }
      // step03: 处理导出
      task.setTaskSource(vo.getTaskSource());
      this.process(vo, task);
      // step05:推送进度
      this.exportSendProcessMsgBean.sendDetailTaskProcessMsg(vo.getExportTaskDetailCode(),
          ExecStatusEnum.FINISH.getDictCode(), ExportDetailProcessEnum.END, vo.getAccount());
    } catch (CrmExportException | IllegalArgumentException | NullPointerException | IndexOutOfBoundsException ex) {
      log.error("导出任务{}失败,err[{}]", vo.getExportTaskDetailCode(), Throwables.getStackTraceAsString(ex));
      this.handlerDetailTaskFailed(vo.getTaskCode(), vo.getExportTaskDetailCode(), ex.getMessage(), vo.getAccount());
    } catch (Exception ex) {
      log.error("导出任务{}失败,err[{}]", vo.getExportTaskDetailCode(), Throwables.getStackTraceAsString(ex));
      this.handlerDetailTaskFailed(vo.getTaskCode(), vo.getExportTaskDetailCode(), "导出失败", vo.getAccount());
    } finally {
      // step04:数据处理完成，清理MarsAuthorityContext现场
      MarsAuthorityContextHolder.clearContext();
    }
  }

  /**
   * 处理导出
   * 
   * @param vo 任务参数
   * @param task 任务初始化用户身份上下文、数据权限上下文等
   * @throws CrmExportException
   */
  private void process(IeExecutorVoForExport vo, ExportTaskProcessVo task) throws IllegalArgumentException, CrmExportException {
    // 2、获取任务处理器（process）
    String businessCode = task.getBusinessCode();
    ExportProcess<?> process = this.findExportProcess(businessCode);
    boolean isEuropaExport = ImportExportConstant.EXPORT_BIZ_CODE_MDM_EUROPA_CODE.equalsIgnoreCase(businessCode)
        || ImportExportConstant.EXPORT_BIZ_CODE_MDM_EUROPA_ASYNC_CODE.equalsIgnoreCase(businessCode);

    boolean isWebApiExport = ImportExportConstant.EXPORT_BIZ_CODE_EXPORT_WEB_API.equalsIgnoreCase(businessCode)
        || ImportExportConstant.EXPORT_BIZ_CODE_EXPORT_WEB_API_ASYNC.equalsIgnoreCase(businessCode);
    if (isEuropaExport) {
      log.debug("系统调用进入数据视图处理模块:{}", businessCode);
      // （2）获取导出字段
      String parentCode = vo.getParentCode();
      String functionCode = vo.getFunctionCode();
      Validate.isTrue(parentCode != null, "数据视图导出的字段属性未获取到parentCode");
      Validate.isTrue(functionCode != null, "数据视图导出的字段属性未获取到functionCode");
      // 处理数据权限
      FunctionPermissionVo funPermVo = new FunctionPermissionVo();
      funPermVo.setFunctionCode(functionCode);
      funPermVo.setParentCode(parentCode);
      // (3) 发送进展
      this.exportSendProcessMsgBean.sendStartProcessMsg(task, ExportDetailProcessEnum.RUNNING,
          ExportProcessEnum.RUNNING,
          vo.getAccount());

      // XZK 20230529 子任务处理[数据库读取、写入子任务文件]
      this.detailExcelHandlerForEuropa(task, process, funPermVo);
    }
    else if(isWebApiExport){
      log.debug("系统调用进入weiApi处理模块:{}",businessCode);
      // (2) 获取导出字段
      String parentCode = vo.getParentCode();
      String functionCode = vo.getFunctionCode();
      Validate.isTrue(parentCode != null, "webApi导出的字段属性未获取到parentCode");
      Validate.isTrue(functionCode != null,"webApi导出的字段属性未获取到functionCode");
      // 处理数据权限
      FunctionPermissionVo functionPermissionVo = new FunctionPermissionVo();
      functionPermissionVo.setFunctionCode(functionCode);
      functionPermissionVo.setParentCode(parentCode);
      // 发送进程
      this.exportSendProcessMsgBean.sendStartProcessMsg(task,ExportDetailProcessEnum.RUNNING,ExportProcessEnum.RUNNING,vo.getAccount());

      // 子任务处理[数据库读取，写入子任务文件]
      this.detailExcelHandlerForWebApi(task,process,functionPermissionVo);
    }
    else {
      String parametersJson = task.getParametersJson();
      JSONObject parseObj = JSONUtil.parseObj(parametersJson);
      Object excelField = parseObj.get("excelField");
      // （1）获取对象转换实体
      Class findCrmExcelVoClass = process.findCrmExcelVoClass();
      final CrmExcelExport excelExport = (CrmExcelExport) findCrmExcelVoClass.getAnnotation(CrmExcelExport.class);
      Validate.notNull(excelExport, "excel导入bean未配置CrmExcelExport注解");
      // （2）获取导出字段
      final Field[] fields = ReflectUtil.getFields(findCrmExcelVoClass);
      long count = Stream.of(fields)
          .filter(a -> Objects.nonNull(a.getAnnotation(CrmExcelColumn.class)))
          .count();
      Validate.isTrue(count > 0, "excel导出bean的字段属性未配置CrmExcelColumn注解");
      // (3) 发送进展
      // this.exportSendProcessMsgBean.sendStartProcessMsg(task, ExportDetailProcessEnum.RUNNING,
      // ExportProcessEnum.RUNNING, vo.getAccount());
      // 处理数据权限

      // XZK 20230529 子任务处理[数据库读取、写入子任务文件]
      // ES分段导出和普通自定义导出做出区分
      if (process instanceof AbstractEsParagraphExportProcess) {
        this.detailExcelHandlerForEsParagraph(task, process);
      } else {
        this.detailExcelHandler(task, process);
      }
    }
    // 处理主任务信息(合并文件，更新主任务状态)
    this.mainTaskHandler(task);

  }

  /**
   * webApi导出子任务处理
   * @param task
   * @param process
   * @param functionPermissionVo
   */
  private void detailExcelHandlerForWebApi(ExportTaskProcessVo task, ExportProcess<?> process,
      FunctionPermissionVo functionPermissionVo) throws CrmExportException {
    log.debug("开始处理webApi导出业务");
    // 文件名
    String fileName = getTaskFileName(task, process);
    // 以下方法和 detailExcelHandler 处理逻辑相同
    File tmpFile = null;
    String tmpFilePath = null;
    {
      String parentPath = this.findExportParentPath(task);
      File mkdir = FileUtil.mkdir(parentPath);
      tmpFile = FileUtil.file(mkdir, fileName);
      try {
        tmpFilePath = tmpFile.getCanonicalPath();
      } catch (IOException e) {
        log.error("子任务临时文件创建失败", e);
        Validate.isTrue(false, "子任务临时文件创建失败" + e.getMessage());
      }
    }

    final Map<String, Object> paramsMap = this.findParamsMap(task);
    if (StringUtils.isNotBlank(task.getMarsListCode())) {
      log.info("========== 已设定数据导出子任务匹配的数据权限：{}", task.getMarsListCode());
    } else {
      log.info("========== 未设定数据导出子任务匹配的数据权限");
    }
    StopWatch sw = new StopWatch();
    sw.start("子任务处理");
    this.crmExcelProcessStrategy.exportDetailForWebApi(task, process, paramsMap, tmpFile.getAbsolutePath(), functionPermissionVo);
    sw.stop();
    boolean f = FileUtil.exist(tmpFilePath) && FileUtil.size(tmpFile) > 0;
    Validate.isTrue(f, "子任务导出失败");
    // 上传子任务导出文件
    final ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(task.getDetailCode());
    msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    msgVo.setProcessType(ExportDetailProcessEnum.UPLOAD.getCode());
    msgVo.setRemark(ExportDetailProcessEnum.UPLOAD.getFormat());
    msgVo.setAccount(task.getCreateAccount());
    this.exportSendProcessMsgBean.sendMsg(msgVo);

    final String fileCode = this.uploadFile(tmpFilePath, fileName);
    FileUtil.del(tmpFilePath);
    Validate.notBlank(fileCode, CharSequenceUtil.format(ERR_MSG_FORMAT, fileName));
    // 更新子任务执行状态
    final ExportTaskDetailUpdateModelVo vo = new ExportTaskDetailUpdateModelVo();
    vo.setDetailCode(task.getDetailCode());
    vo.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    vo.setFileCode(fileCode);
    // 设置执行时长
    Long lastTaskTimeMillis = sw.getLastTaskTimeMillis();
    vo.setExecuteDuration(lastTaskTimeMillis.intValue());
    this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(vo);
  }

  /**
   * 子任务处理
   *
   * @param task
   * @param process
   * @throws CrmExportException
   * @throws Exception
   */
  private void detailExcelHandlerForEuropa(ExportTaskProcessVo task, ExportProcess<?> process,
      FunctionPermissionVo funPermVo) throws CrmExportException {
    log.debug("开始处理Europa导出业务2");
    // 文件名
    String fileName = getTaskFileName(task, process);
    // 以下方法和 detailExcelHandler 处理逻辑相同
    File tmpFile = null;
    String tmpFilePath = null;
    {
      String parentPath = this.findExportParentPath(task);
      File mkdir = FileUtil.mkdir(parentPath);
      tmpFile = FileUtil.file(mkdir, fileName);
      try {
        tmpFilePath = tmpFile.getCanonicalPath();
      } catch (IOException e) {
        log.error("子任务临时文件创建失败", e);
        Validate.isTrue(false, "子任务临时文件创建失败" + e.getMessage());
      }
    }
    
    final Map<String, Object> paramsMap = this.findParamsMap(task);
    if (StringUtils.isNotBlank(task.getMarsListCode())) {
      log.info("========== 已设定数据导出子任务匹配的数据权限：{}", task.getMarsListCode());
    } else {
      log.info("========== 未设定数据导出子任务匹配的数据权限");
    }
    StopWatch sw = new StopWatch();
    sw.start("子任务处理");
    this.crmExcelProcessStrategy.exportDetailForEuropa(task, process, paramsMap, tmpFile.getAbsolutePath(), funPermVo);
    sw.stop();
    boolean f = FileUtil.exist(tmpFilePath) && FileUtil.size(tmpFile) > 0;
    Validate.isTrue(f, "子任务导出失败");
    // 上传子任务导出文件
    final ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(task.getDetailCode());
    msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    msgVo.setProcessType(ExportDetailProcessEnum.UPLOAD.getCode());
    msgVo.setRemark(ExportDetailProcessEnum.UPLOAD.getFormat());
    msgVo.setAccount(task.getCreateAccount());
    this.exportSendProcessMsgBean.sendMsg(msgVo);

    final String fileCode = this.uploadFile(tmpFilePath, fileName);
    FileUtil.del(tmpFilePath);
    Validate.notBlank(fileCode, CharSequenceUtil.format(ERR_MSG_FORMAT, fileName));
    // 更新子任务执行状态
    final ExportTaskDetailUpdateModelVo vo = new ExportTaskDetailUpdateModelVo();
    vo.setDetailCode(task.getDetailCode());
    vo.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    vo.setFileCode(fileCode);
    // 设置执行时长
    Long lastTaskTimeMillis = sw.getLastTaskTimeMillis();
    vo.setExecuteDuration(lastTaskTimeMillis.intValue());
    this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(vo);
  }

  /**
   * 上传任务的文件名
   * 
   * @param task 任务
   * @param process
   * @return
   */
  private String getTaskFileName(ExportTaskProcessVo task, ExportProcess process) {
    // 从导出基座中获取上传任务文件名,如果不为空则返回获取到的名字,否则以默认方式生成名字
    String fileTempName = process.getTaskFileName(task);
    // 当数据量小 一个子任务处理掉时 生成的文件名称修改为主任务
    String fileNameType = (task.getTotal().equals(task.getPageSize())) ? "主任务" : "子任务";
    if (StringUtils.isNotBlank(fileTempName)) {
      // 拼接子任务编码,保持子文件excel名称的唯一性
      //当只存在一个子任务时 调取创建任务开始设置的文件名称并修改子文件
      ExportTask exportTaskTemp = exportTaskRepository.findByTaskCode(task.getTaskCode());
      if(ObjectUtils.isNotEmpty(exportTaskTemp)){
        return exportTaskTemp.getFileName();
      }
      return CharSequenceUtil.format("{}_{}_{}{}", fileNameType, fileTempName, task.getDetailCode(),
          ImportExportConstant.IE_FILE_SUFFIX);
    }
    // 文件名
    String fileName = null;
    if (task.getTotal().equals(task.getPageSize())) {
      ExportTask exportTaskTemp = exportTaskRepository.findByTaskCode(task.getTaskCode());
      if (ObjectUtils.isNotEmpty(exportTaskTemp)) {
        log.info("断点-数据3: {}", com.alibaba.fastjson.JSONObject.toJSONString(exportTaskTemp));
        fileName = exportTaskTemp.getFileName();
      }
    }

    if (fileName == null) {
      fileName = CharSequenceUtil.format("{}_{}_{}_{}{}", fileNameType, task.getTaskSource(),
          task.getCreateAccountName(), task.getDetailCode(), ImportExportConstant.IE_FILE_SUFFIX);
    }
    return fileName;
  }

  /**
   * 子任务处理
   *
   * @param task
   * @param process
   */
  private void detailExcelHandler(ExportTaskProcessVo task, ExportProcess process) {
    log.debug("自定义导出");
    String parentPath = this.findExportParentPath(task);
    String fileName = getTaskFileName(task, process);
    String tmpFilePath = parentPath + fileName;
    final Map<String, Object> paramsMap = this.findParamsMap(task);
    // 文件名
    this.crmExcelProcessStrategy.exportDetail(task, process, paramsMap, tmpFilePath);
    boolean f = FileUtil.exist(tmpFilePath) && FileUtil.size(FileUtil.file(tmpFilePath)) > 0;
    Validate.isTrue(f, "子任务导出失败");
    // 上传子任务导出文件
    final ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(task.getDetailCode());
    msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    msgVo.setProcessType(ExportDetailProcessEnum.UPLOAD.getCode());
    msgVo.setRemark(ExportDetailProcessEnum.UPLOAD.getFormat());
    msgVo.setAccount(task.getCreateAccount());
    this.exportSendProcessMsgBean.sendMsg(msgVo);

    final String fileCode = this.uploadFile(tmpFilePath, fileName);
    FileUtil.del(tmpFilePath);
    Validate.notBlank(fileCode, CharSequenceUtil.format(ERR_MSG_FORMAT, fileName));
    // 更新子任务执行状态
    final ExportTaskDetailUpdateModelVo vo = new ExportTaskDetailUpdateModelVo();
    vo.setDetailCode(task.getDetailCode());
    vo.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    vo.setFileCode(fileCode);
    this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(vo);
  }

  /**
   * ES导出子任务处理
   *
   * @param task
   * @param process
   */
  private void detailExcelHandlerForEsParagraph(ExportTaskProcessVo task, ExportProcess process) {
    log.debug("自定义ES分段导出");
    String parentPath = this.findExportParentPath(task);
    String fileName = getTaskFileName(task, process);
    String tmpFilePath = parentPath + fileName;
    final Map<String, Object> paramsMap = this.findParamsMap(task);
    // 文件名
    this.crmExcelProcessStrategy.exportDetailForEsParagraph(task, process, paramsMap, tmpFilePath);
    boolean f = FileUtil.exist(tmpFilePath) && FileUtil.size(FileUtil.file(tmpFilePath)) > 0;
    Validate.isTrue(f, "子任务导出失败");
    // 上传子任务导出文件
    final ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(task.getDetailCode());
    msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    msgVo.setProcessType(ExportDetailProcessEnum.UPLOAD.getCode());
    msgVo.setRemark(ExportDetailProcessEnum.UPLOAD.getFormat());
    msgVo.setAccount(task.getCreateAccount());
    this.exportSendProcessMsgBean.sendMsg(msgVo);

    final String fileCode = this.uploadFile(tmpFilePath, fileName);
    FileUtil.del(tmpFilePath);
    Validate.notBlank(fileCode, CharSequenceUtil.format(ERR_MSG_FORMAT, fileName));
    // 更新子任务执行状态
    final ExportTaskDetailUpdateModelVo vo = new ExportTaskDetailUpdateModelVo();
    vo.setDetailCode(task.getDetailCode());
    vo.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    vo.setFileCode(fileCode);
    this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(vo);
  }

  /**
   * 主任务处理
   *
   * @param task
   */
  private void mainTaskHandler(ExportTaskProcessVo task) {
    Validate.notNull(task, "任务处理信息不能为空");
    final String lockKey =
        CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_KEY_PREFIX_FORMAT, task.getTaskCode());

    while (true) {
      if (this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 10)) {
        try {
          this.taskHandler(task);
        } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
          log.error("导出任务{}处理失败,err[{}]", task.getTaskCode(), Throwables.getStackTraceAsString(e));
          this.handlerMainTaskFailed(task.getTaskCode(), e.getMessage(), task.getCreateAccount());
        } catch (Exception e) {
          log.error("导出任务{}处理失败,err[{}]", task.getTaskCode(), Throwables.getStackTraceAsString(e));
          this.handlerMainTaskFailed(task.getTaskCode(), "主任务处理失败", task.getCreateAccount());
        } finally {
          this.redisMutexService.unlock(lockKey);
        }
        break;
      }
      ThreadUtil.safeSleep(50);
    }
  }

  /**
   * 主任务处理
   *
   * <p>
   * 获取所有子任务信息（判断子任务是否执行完成）
   *
   * <p>
   * 下载所有的子任务的文件到本地
   *
   * <p>
   * 合并所有的Excel，并上传
   *
   * <p>
   * 更新主任务状态
   *
   * @param task
   */
  private void taskHandler(ExportTaskProcessVo task) {
    String parentPath = this.findExportParentPath(task);
    ExportTask exportTask = this.exportTaskService.findDetailByTaskCode(task.getTaskCode());
    String account = exportTask.getCreateAccount();
    Validate.notNull(exportTask, CharSequenceUtil.format("{}导出任务不存在", task.getTaskCode()));
    if (!(exportTask.getExecStatus().equals(ExecStatusEnum.RUNNING.getDictCode())
        || exportTask.getExecStatus().equals(ExecStatusEnum.DEFAULT.getDictCode()))) {
      // 主任务已完成
      return;
    }

    // 发消息 1、主任务开始处理
    final List<ExportTaskDetail> list =
        this.exportTaskDetailRepository.findByTaskCodes(Sets.newHashSet(task.getTaskCode()));
    Validate.isTrue(
        !CollectionUtils.isEmpty(list),
        CharSequenceUtil.format("{}导出任务子任务信息不存在", task.getTaskCode()));
    final Optional<ExportTaskDetail> first =
        list.stream()
            .filter(
                a -> ExecStatusEnum.RUNNING.getDictCode().equals(a.getExecStatus())
                    || ExecStatusEnum.DEFAULT.getDictCode().equals(a.getExecStatus()))
            .findFirst();
    if (first.isPresent()) {
      log.info(CharSequenceUtil.format("{}导出任务子任务还在执行中，无需执行该操作", task.getTaskCode()));
      // 子任务尚未执行完成（结束）
      return;
    }

    // 2、下载子任务导出文件
    // 判断是否只有一个子任务且任务已完成
    final List<ExportTaskDetail> finishDetailTaskList =
        list.stream()
            .filter(
                a -> ExecStatusEnum.FINISH.getDictCode().equals(a.getExecStatus())
                    && StringUtils.isNotBlank(a.getFileCode()))
            .collect(Collectors.toList());
    Validate.isTrue(!CollectionUtils.isEmpty(finishDetailTaskList), "未获取到子任务生成的文件信息");
    ExportProcess<?> process = this.findExportProcess(task.getBusinessCode());
    // 任务总数
    Integer taskTotal = list.size();
    // 已完成任务数量
    Integer finishTotal = finishDetailTaskList.size();
    Integer mergeFileMaxCount = Objects.nonNull(process.getMergeFileMaxCount()) ? process.getMergeFileMaxCount()
        : importExportProperties.getExportProperties().getMergeFileMaxCount();
    if (finishTotal == 1) {
      // 发消息 更新主任务状态
      final LinkedHashSet<String> set = Sets.newLinkedHashSet();
      set.add(finishDetailTaskList.get(0).getFileCode());
      this.finishTask(task, account, set);
      //当前任务结束后 把剩余排队任务重新推送一遍
//      List<ExportTaskQueueDto> linkedList = exportTaskQueueGlobalList.getLinkedList();
//      for (int i = 0; i < linkedList.size(); i++) {
//        exportSendProcessMsgBean.sendTaskProcessQueueMsg(linkedList.get(i), i);
//      }

      return;
    } else if (mergeFileMaxCount > 0 && mergeFileMaxCount < exportTask.getTotal()) {
      // 发消息 更新主任务状态
      this.finishTask(task, account, Sets.newLinkedHashSet());
      return;
    }  else {
      if (Objects.equals(finishTotal, taskTotal)) {
        // this.exportSendProcessMsgBean.sendTaskProcessMsg(
        // task, ExecStatusEnum.FINISH.getDictCode(), ExportProcessEnum.END, TOTAL, account);
      } else {
        int cursor = BigDecimal.valueOf(90).multiply(
            BigDecimal.valueOf(finishTotal).divide(BigDecimal.valueOf(taskTotal),4, RoundingMode.HALF_UP))
            .setScale(0, BigDecimal.ROUND_UP).intValue();
        this.exportSendProcessMsgBean.sendTaskProcessMsg(
            task, ExecStatusEnum.RUNNING.getDictCode(), ExportProcessEnum.RUNNING, cursor, account, true);
      }
    }

    // k-detailCode,v-localPath
    Map<String, String> detailTaskFilePathMap =
        this.createLocalFileByDetailTask(finishDetailTaskList, parentPath);

    final List<ExportTaskDetail> detailList1 =
        list.stream()
            .filter(a -> detailTaskFilePathMap.keySet().contains(a.getDetailCode()))
            .sorted(Comparator.comparing(ExportTaskDetail::getPageNo))
            .collect(Collectors.toList());
    List<ExportTaskDetailVo> detailList =
        (List<ExportTaskDetailVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            detailList1,
            ExportTaskDetail.class,
            ExportTaskDetailVo.class,
            HashSet.class,
            ArrayList.class);

    // 3、合并文件
    // k-fileName,v-path
    final LinkedHashMap<String, String> uploadFileMap =
        this.crmExcelProcessStrategy.merge(
            task, process, parentPath, detailList, detailTaskFilePathMap);
    Validate.isTrue(!uploadFileMap.isEmpty(), "未合并文件");
    for (String file : uploadFileMap.values()) {
      Validate.isTrue(FileUtil.exist(file) && FileUtil.size(FileUtil.file(file)) > 0, "导出任务未生成文件");
    }

    // 4、合并完成，上传主任务文件
    this.exportSendProcessMsgBean.sendTaskProcessMsg(
        task,
        ExecStatusEnum.RUNNING.getDictCode(),
        ExportProcessEnum.UPLOAD,
        EXPORT_FILE_UPLOAD, account, true);

    LinkedHashSet<String> fileCodeSet = Sets.newLinkedHashSet();
    try {
      for (Entry<String, String> item : uploadFileMap.entrySet()) {
        final String fileCode = this.uploadFile(item.getValue(), item.getKey());
        Validate.notBlank(fileCode, "文件上传失败");
        fileCodeSet.add(fileCode);
      }
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error(e.getMessage(), e);
      Validate.isTrue(false, e.getMessage());
    } catch (Exception e) {
      log.error(e.getMessage(), e);
      Validate.isTrue(false, "合并文件上传失败");
    } finally {
      FileUtil.del(parentPath);
    }
    this.finishTask(task, account, fileCodeSet);
    //当前任务结束后 把剩余排队任务重新推送一遍
//    List<ExportTaskQueueDto> linkedList = exportTaskQueueGlobalList.getLinkedList();
//    for (int i = 0; i < linkedList.size(); i++) {
//      exportSendProcessMsgBean.sendTaskProcessQueueMsg(linkedList.get(i), i);
//    }
  }

  /**
   * 完成主任务
   *
   * @param task 任务信息
   * @param account 账号
   * @param fileCodeSet 文件编码集合
   */
  private void finishTask(ExportTaskProcessVo task, String account, LinkedHashSet<String> fileCodeSet) {
    // 5、更新主任务状态
    final ExportTaskProcessModelDto dto = new ExportTaskProcessModelDto();
    dto.setTaskCode(task.getTaskCode());
    dto.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    dto.setFileCodeSet(fileCodeSet);
    this.exportTaskService.updateByExportTaskProcessModelDto(dto);
    this.exportSendProcessMsgBean.sendTaskProcessMsg(
        task, ExecStatusEnum.FINISH.getDictCode(), ExportProcessEnum.END, TOTAL, account, true, fileCodeSet.stream().findFirst().orElse(null));
    // 执行成功回调操作
    this.callBack(task.getTaskCode());
  }

  /**
   * 下载文件到本地
   *
   * @param list
   * @param parentPath
   * @return k-detailCode,v-filePath
   */
  private Map<String, String> createLocalFileByDetailTask(
      List<ExportTaskDetail> list, String parentPath) {
    final List<ExportTaskDetail> detailList =
        list.stream()
            .filter(
                a -> StringUtils.isNotBlank(a.getFileCode())
                    && ExecStatusEnum.FINISH.getDictCode().equals(a.getExecStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(detailList)) {
      return Maps.newHashMap();
    }
    // k-detailCode,v-filePath
    Map<String, String> map = Maps.newHashMap();
    // k-fileCode,v-detailCode
    Map<String, String> mapFileDetail =
        detailList.stream()
            .collect(
                Collectors.toMap(
                    ExportTaskDetail::getFileCode, ExportTaskDetail::getDetailCode, (a, b) -> a));
    for (List<ExportTaskDetail> item : Lists.partition(detailList, 10)) {
      Map<String, OrdinaryFileVo> downloadFileVoMap = Maps.newHashMap();
      for (ExportTaskDetail detail : item) {
        OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(detail.getFileCode());
        if (ordinaryFileVo == null) {
          continue;
        }
        byte[] bytes = this.fileHandleService.findContentByFilePathAndFileRename(ordinaryFileVo.getRelativeLocal(),
            ordinaryFileVo.getFileName());
        if (bytes == null) {
          continue;
        }
        // 附件文件下载
        String detailCode = mapFileDetail.get(detail.getFileCode());
        String tmpPath = parentPath + detailCode + ImportExportConstant.IE_FILE_SUFFIX;
        FileUtil.writeBytes(bytes, tmpPath);
        downloadFileVoMap.put(detail.getFileCode(), ordinaryFileVo);
      }
      if (downloadFileVoMap.isEmpty()) {
        continue;
      }
      for (Entry<String, OrdinaryFileVo> file : downloadFileVoMap.entrySet()) {
        String detailCode = mapFileDetail.get(file.getKey());
        String tmpPath = parentPath + detailCode + ImportExportConstant.IE_FILE_SUFFIX;
        map.put(detailCode, tmpPath);
      }
    }
    return map;
  }

  /**
   * 该私有方法为venus文件上传接口参数封装
   */
  private List<OrdinaryFileVo> venusFileUpload(MultipartFile multipartFile) throws IOException {
    Base64UploadDto base64UploadDto = new Base64UploadDto();
    base64UploadDto.setCreator(loginUserService.findCurrentAccount());
    base64UploadDto.setFileNanmes(new String[] {multipartFile.getOriginalFilename()});
    base64UploadDto.setBase64Contents(new String[] {Base64Utils.encodeToString(multipartFile.getBytes())});
    // subsystem不能有横杠只能用下划线分隔
    List<OrdinaryFileVo> ordinaryFileVoList = this.fileHandleService.fileUpload(subsystem.replace("-", "_"), base64UploadDto , null);
    Validate.notEmpty(ordinaryFileVoList, "文件上传失败！");
    return ordinaryFileVoList;
  }

  /**
   * 上传文件 返回 fileCode
   *
   * @param filePath
   * @param fileName
   * @return
   */
  private String uploadFile(String filePath, String fileName) {
    final boolean exist = FileUtil.exist(filePath);
    Validate.isTrue(exist, CharSequenceUtil.format("{}文件不存在", fileName));
    // 上传子任务导出文件
    String fileCode = null;
    try (BufferedInputStream is = FileUtil.getInputStream(filePath)) {
      final MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(is, fileName);
      OrdinaryFileVo ordinaryFileVo = this.venusFileUpload(multipartFile).get(0);
      fileCode = ordinaryFileVo.getId();
    } catch (Exception e) {
      log.error("导出文件上传失败,error[{}]", Throwables.getStackTraceAsString(e));
      Validate.isTrue(false, CharSequenceUtil.format(ERR_MSG_FORMAT, fileName));
    }
    return fileCode;
  }

  /**
   * 导出任务预处理（主要是为导出任务初始化用户身份上下文、数据权限上下文等操作）
   *
   * @param detailCode
   * @return
   */
  private ExportTaskProcessVo exportTaskPrevHandler(String detailCode) {
    ExportTaskProcessVo task = this.exportTaskService.findExportTaskModelVoByDetailCode(detailCode);
    Validate.notNull(task, CharSequenceUtil.format("导出子任务编码为{}的任务不存在", detailCode));
    Boolean f =
        task.getExecStatus().equals(ExecStatusEnum.CANCEL.getDictCode())
            || task.getExecStatus().equals(ExecStatusEnum.FINISH.getDictCode());
    Validate.isTrue(!f, CharSequenceUtil.format("导出子任务编码为{}的任务已完成或取消", detailCode));
    // 1、更新子任务信息为正在执行
    if (task.getExecStatus().equals(ExecStatusEnum.DEFAULT.getDictCode())) {
      ExportTaskDetailUpdateModelVo cur = new ExportTaskDetailUpdateModelVo();
      cur.setDetailCode(detailCode);
      cur.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
      this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(cur);
    }
    // 2、更新主任务信息状态为正在执行
    final ExportTaskProcessModelDto dto = new ExportTaskProcessModelDto();
    dto.setTaskCode(task.getTaskCode());
    dto.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    this.exportTaskService.updateByExportTaskProcessModelDto(dto);
    String[] str = detailCode.split("_");
    for (String item : str) {
      if (ImportExportConstant.ONE_MAIN_FLAG.equals(item)) {
        this.exportSendProcessMsgBean.sendTaskProcessMsg(task, ExecStatusEnum.RUNNING.getDictCode(),
            ExportProcessEnum.START, task.getCreateAccount(), true);
      }
    }
    return task;
  }

  /**
   * 获取任务参数信息
   *
   * @param task
   * @return
   */
  private Map<String, Object> findParamsMap(ExportTaskProcessVo task) {
    if (Objects.isNull(task) || StringUtils.isBlank(task.getParametersJson())) {
      return Maps.newHashMap();
    }
    Map<String, Object> map = Maps.newHashMap();
    JSONObject jsonObject = JSONUtil.parseObj(task.getParametersJson());
    Set<String> set = jsonObject.keySet();
    for (String item : set) {
      map.put(item, jsonObject.get(item));
    }
    return map;
  }

  /**
   * 获取导出文件本地路径
   *
   * @param task
   * @return
   */
  private String findExportParentPath(ExportTaskProcessVo task) {
    Date now = new Date();
    String root = this.importExportProperties.getRoot();
    if (StringUtils.isBlank(root)) {
      root = FileUtil.getTmpDirPath();
    }
    if (!root.endsWith("/")) {
      root = root + "/";
    }
    return CharSequenceUtil.format(
        "{}export/{}/{}/{}/{}/",
        root,
        DateUtil.year(now),
        DateUtil.month(now),
        DateUtil.dayOfMonth(now),
        task.getTaskCode());
  }

  /**
   * 子任务处理失败
   *
   * @param detailCode
   * @param msg
   */
  private void handlerDetailTaskFailed(String taskCode, String detailCode, String msg, String account) {
    final ExportTaskDetailUpdateModelVo vo = new ExportTaskDetailUpdateModelVo();
    vo.setDetailCode(detailCode);
    vo.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    vo.setRemark(msg);
    this.exportTaskDetailRepository.updateByExportTaskDetailUpdateModelVo(vo);
    ExportTaskProcessModelDto exportTaskProcessModelDto = new ExportTaskProcessModelDto();

    exportTaskProcessModelDto.setTaskCode(taskCode);
    exportTaskProcessModelDto.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    exportTaskProcessModelDto.setRemark(msg);
    this.exportTaskService.updateExecStatus(exportTaskProcessModelDto);
    // 发送子任务执行失败处理消息
    this.exportSendProcessMsgBean.sendFailedProcessMsg(detailCode, false, msg, account);
    // 发送主任务执行失败处理消息
    ThreadUtil.sleep(500);
    this.exportSendProcessMsgBean.sendFailedProcessMsg(taskCode, true, msg, account);
    // 执行失败回调
    this.callBack(taskCode);
  }

  /**
   * 主任务处理失败
   *
   * @param taskCode
   * @param msg
   */
  private void handlerMainTaskFailed(String taskCode, String msg, String account) {
    final ExportTaskProcessModelDto dto = new ExportTaskProcessModelDto();
    dto.setTaskCode(taskCode);
    dto.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    dto.setRemark(msg);
    this.exportTaskService.updateByExportTaskProcessModelDto(dto);
    // 主任务失败消息
    this.exportSendProcessMsgBean.sendFailedProcessMsg(taskCode, true, msg, account);
    // 执行失败回调
    this.callBack(taskCode);
  }

  /**
   * 执行回调操作
   *
   * @param taskCode 导出任务编码
   */
  private void callBack(String taskCode) {
    if (CollectionUtils.isEmpty(notifyEventListeners)) {
      return;
    }
    String lockKey = CharSequenceUtil.format(ImportExportConstant.IE_EXPORT_CALL_BACK_LOCK_PREFIX_FORMAT, taskCode);
    boolean locked = false;
    try {
      locked = this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
      if (locked) {
        long taskExecutionAfterSleep = importExportProperties.getTaskExecutionIntervalSleep();
        // 应对主从数据不同步问题
        if (taskExecutionAfterSleep > 0) {
          ThreadUtil.sleep(taskExecutionAfterSleep);
          log.debug("******休眠释放数据库压力*********");
        }
        ExportTask exportTask = this.exportTaskRepository.findByTaskCode(taskCode);
        if (Objects.isNull(exportTask)) {
          log.warn("执行失败回调未能找到对应导出任务信息[{}]", taskCode);
          return;
        }
        CallbackStatusEnum callbackStatusEnum = CallbackStatusEnum.getByDictCode(exportTask.getCallBackStatus());
        if (Objects.nonNull(callbackStatusEnum)) {
          log.warn("导出任务[{}]回调,跳过本次回调执行操作[{}]", callbackStatusEnum.getValue(), exportTask.getExecStatus());
          return;
        }
        ExportTaskEventVo exportTaskEventVo = this.nebulaToolkitService.copyObjectByWhiteList(
            exportTask, ExportTaskEventVo.class, HashSet.class, ArrayList.class);
        CallbackStatusEnum callbackStatus = null;
        try {
          if (ExecStatusEnum.FAILED.getDictCode().equals(exportTask.getExecStatus())) {
            log.info("导出任务[{}]当前任务状态[{}],执行失败回调操作", taskCode, exportTask.getExecStatus());
            notifyEventListeners.forEach(eventListener -> eventListener.onFail(exportTaskEventVo));
            callbackStatus = CallbackStatusEnum.HAVE_CALL_BACK_FAIL;
          } else if (ExecStatusEnum.FINISH.getDictCode().equals(exportTask.getExecStatus())) {
            log.info("导出任务[{}]当前任务状态[{}],执行成功回调操作", taskCode, exportTask.getExecStatus());
            notifyEventListeners.forEach(eventListener -> eventListener.onSuccess(exportTaskEventVo));
            callbackStatus = CallbackStatusEnum.HAVE_CALL_BACK_SUCCESS;
          } else {
            log.info("导出任务[{}]当前任务状态[{}],没有匹配的回调操作", taskCode, exportTask.getExecStatus());
          }
        } catch (Exception e) {
          log.error("导出任务执行回调发生异常", e);
        }
        if (Objects.nonNull(callbackStatus)) {
          this.exportTaskRepository.updateCallBackStatusByIds(Lists.newArrayList(exportTask.getId()), callbackStatus);
        }
      }
    } catch (Exception e) {
      log.error("导出任务执行回调发生异常", e);
    } finally {
      if (locked) {
        redisMutexService.unlock(lockKey);
      }
    }
  }
}
