package com.biz.crm.common.ie.local.service.internal;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.entity.ImportTemplate;
import com.biz.crm.common.ie.local.entity.ImportTemplateDetail;
import com.biz.crm.common.ie.local.model.dto.ImportTaskModelDto;
import com.biz.crm.common.ie.local.model.vo.FileModelVo;
import com.biz.crm.common.ie.local.repository.ImportTaskRepository;
import com.biz.crm.common.ie.local.repository.ImportTemplateDetailRepository;
import com.biz.crm.common.ie.local.repository.ImportTemplateRepository;
import com.biz.crm.common.ie.local.service.ImportTaskService;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.dto.ImportTaskPaginationDto;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.LoadStatusEnum;
import com.biz.crm.common.ie.sdk.event.ImportExportTaskEventListener;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.rulecode.sdk.service.GenerateCodeRuleVoService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * 导入任务信息(ImportTask)表服务实现类
 *
 * @author sunx
 * @date 2022-05-12 09:25:18
 */
@Service("importTaskService")
public class ImportTaskServiceImpl implements ImportTaskService {

    @Autowired(required = false)
    private ImportTaskRepository importTaskRepository;

    @Autowired(required = false)
    private GenerateCodeRuleVoService generateCodeRuleVoService;

    @Autowired(required = false)
    private ImportTemplateRepository importTemplateRepository;

    @Autowired(required = false)
    private ImportTemplateDetailRepository importTemplateDetailRepository;

    @Autowired(required = false)
    private FileHandleService fileHandleService;

    @Autowired(required = false)
    private ImportExportProperties importExportProperties;

    @Autowired(required = false)
    private List<ImportProcess> importProcesses;

    @Autowired
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private List<ImportExportTaskEventListener> importExportTaskAuthRecordListeners;

    @Override
    public Page<ImportTask> findByConditions(Pageable pageable, ImportTaskPaginationDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        dto = Optional.ofNullable(dto).orElse(new ImportTaskPaginationDto());
        dto.setApplicationName(this.importExportProperties.getIeTaskGroup());
        dto.setAppCode(TenantUtils.getTenantCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<ImportTask> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.importTaskRepository.findByConditions(page, dto);
    }

    @Override
    public ImportTask findDetailByTaskCode(String taskCode) {
        if (StringUtils.isBlank(taskCode)) {
            return null;
        }
        final ImportTask task = this.importTaskRepository.findByTaskCode(taskCode);
        Set<String> set = Sets.newHashSet();
        if (StringUtils.isNotBlank(task.getFileCode())) {
            set.add(task.getFileCode());
        }
        if (StringUtils.isNotBlank(task.getFailedFileCode())) {
            set.add(task.getFailedFileCode());
        }
        if (CollectionUtils.isEmpty(set)) {
            return task;
        }
        Map<String, OrdinaryFileVo> mapFile = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(set)) {
            for (String fileCode : set) {
                OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(fileCode);
                if (ordinaryFileVo == null) {
                    continue;
                }
                mapFile.put(fileCode, ordinaryFileVo);
            }
        }

        if (StringUtils.isNotBlank(task.getFileCode())) {
            final OrdinaryFileVo fileVo = mapFile.get(task.getFileCode());
            if (Objects.nonNull(fileVo)) {
                FileModelVo fileModelVo = new FileModelVo();
                fileModelVo.setFileName(fileVo.getOriginalFileName());
                fileModelVo.setFileCode(fileVo.getId());
                task.setFile(fileModelVo);
            }
        }
        if (StringUtils.isNotBlank(task.getFailedFileCode())) {
            final OrdinaryFileVo fileVo = mapFile.get(task.getFailedFileCode());
            if (Objects.nonNull(fileVo)) {
                FileModelVo fileModelVo = new FileModelVo();
                fileModelVo.setFileName(fileVo.getOriginalFileName());
                fileModelVo.setFileCode(fileVo.getId());
                task.setFailedFile(fileModelVo);
            }
        }
        return task;
    }

    @Override
    @Transactional
    public ImportTask create(ImportTask importTask) {
        this.validation(importTask);
        importTask.setTenantCode(TenantUtils.getTenantCode());
        importTask.setTaskCode(this.generateCodeRuleVoService.generateCode(ImportExportConstant.IMPORT_TASK_CODE_PREFIX));
        importTask.setExecStatus(ExecStatusEnum.DEFAULT.getDictCode());
        importTask.setLoadStatus(LoadStatusEnum.NO.getDictCode());
        final ImportProcess importProcess = this.findImportProcess(importTask.getTemplateCode());
        Validate.notNull(importProcess, "导入模板不存在");
        Map globalParams = importProcess.getGlobalParams();
        if (globalParams != null && !globalParams.isEmpty()) {
            if (StringUtils.isBlank(importTask.getParametersJson())) {
                importTask.setParametersJson(JSONUtil.toJsonStr(globalParams));
            } else {
                JSONObject parameters = JSON.parseObject(importTask.getParametersJson());
                parameters.putAll(globalParams);
                importTask.setParametersJson(JSONUtil.toJsonStr(parameters));
            }
        }
        this.importTaskRepository.saveOrUpdate(importTask);

        // ======== 触发监听
        String createAccount = this.loginUserService.getLoginAccountName();
        if (!CollectionUtils.isEmpty(this.importExportTaskAuthRecordListeners)) {
            for (ImportExportTaskEventListener listener : this.importExportTaskAuthRecordListeners) {
                listener.onTaskCreate(false, importTask.getTaskCode(), importTask.getAppCode(), importTask.getTenantCode(), importTask.getApplicationName(), createAccount);
            }
        }

        return importTask;
    }

    @Override
    public List<ImportTask> findByImportTaskModelDto(ImportTaskModelDto dto) {
        if (Objects.isNull(dto)) {
            return Lists.newLinkedList();
        }
        return this.importTaskRepository.findByImportTaskModelDto(dto);
    }


    @Override
    @Transactional
    public void updateLocalStatus(Set<String> taskCodes, String loadStatus) {
        Validate.isTrue(!CollectionUtils.isEmpty(taskCodes), "导入任务编码不能为空");
        Validate.notEmpty(loadStatus, "加载状态不能为空");
        this.importTaskRepository.updateLocalStatus(taskCodes, loadStatus);
    }

    private void validation(ImportTask importTask) {
        Validate.notNull(importTask, "导入任务信息不能为空");
        importTask.setAppCode(TenantUtils.getTenantCode());
        importTask.setApplicationName(this.importExportProperties.getIeTaskGroup());
        Validate.notEmpty(importTask.getTemplateCode(), "模板编码不能为空");
        Validate.notEmpty(importTask.getFileCode(), "文件编码不能为空");
        ImportTemplateDetail detail =
                this.importTemplateDetailRepository.findDetailByTemplateCode(importTask.getTemplateCode());
        Validate.notNull(detail, "模板信息不存在");
        ImportTemplate template =
                this.importTemplateRepository.findByBusinessCode(detail.getBusinessCode());
        Validate.notNull(template, "模板业务配置信息不存在");
        importTask.setTaskName(detail.getTemplateName());
        importTask.setBusinessCode(template.getBusinessCode());
        importTask.setBusinessName(template.getBusinessName());
    }

    /**
     * 获取对应的数据处理器
     *
     * @param code
     * @return
     */
    private ImportProcess findImportProcess(String code) {
        if (CollectionUtils.isEmpty(importProcesses) || StringUtils.isBlank(code)) {
            return null;
        }
        Optional<ImportProcess> op =
                importProcesses.stream().filter(item -> item.getTemplateCode().equals(code)).findFirst();
        return op.orElse(null);
    }
}
