package com.biz.crm.common.ie.local.service.internal;

import com.biz.crm.common.ie.local.entity.ImportTemplateDetail;
import com.biz.crm.common.ie.local.model.vo.FileModelVo;
import com.biz.crm.common.ie.local.repository.ImportTemplateDetailRepository;
import com.biz.crm.common.ie.local.service.ImportTemplateDetailService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 导入模板明细(ImportTemplateDetail)表服务实现类
 *
 * @author sunx
 * @date 2022-05-10 10:57:20
 */
@Service("importTemplateDetailService")
public class ImportTemplateDetailServiceImpl implements ImportTemplateDetailService {

  @Autowired(required = false) private ImportTemplateDetailRepository importTemplateDetailRepository;

  @Autowired(required = false) private FileHandleService fileHandleService;

  @Override
  public List<ImportTemplateDetail> findDetailByBusinessCode(String businessCode) {
    if (StringUtils.isBlank(businessCode)) {
      return Lists.newLinkedList();
    }
    List<ImportTemplateDetail> list =
        this.importTemplateDetailRepository.findDetailByBusinessCode(businessCode);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    Set<String> fileCodes =
        list.stream().map(ImportTemplateDetail::getFileCode).collect(Collectors.toSet());
    Map<String, FileModelVo> map = this.findFileModelVoMap(fileCodes);
    for (ImportTemplateDetail item : list) {
      item.setFile(map.get(item.getFileCode()));
    }
    return list;
  }

  @Override
  @Transactional
  public void createBatch(List<ImportTemplateDetail> list, String businessCode) {
    Validate.notEmpty(businessCode, "业务编码不能为空");
    this.deleteByBusinessCodes(Lists.newArrayList(businessCode));
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    int i = 1;
    Set<String> codeSet = Sets.newHashSet();
    Set<String> fileCodeSet = Sets.newHashSet();
    for (ImportTemplateDetail item : list) {
      item.setId(null);
      item.setBusinessCode(businessCode);
      item.setSort(i++);
      Validate.notEmpty(item.getTemplateCode(), "模板编码不能为空");
      Validate.notEmpty(item.getFileCode(), "模板文件不能为空");
      Validate.isTrue(codeSet.add(item.getTemplateCode()), "存在相同的模板编码");
      Validate.isTrue(fileCodeSet.add(item.getFileCode()), "存在相同的模板文件");
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    this.importTemplateDetailRepository.saveBatch(list);
  }

  @Override
  @Transactional
  public void deleteByBusinessCodes(List<String> businessCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(businessCodes), "业务编码集合不能为空");
    this.importTemplateDetailRepository.deleteByBusinessCodes(businessCodes);
  }

  /**
   * 获取文件信息
   *
   * @param fileCodeSet
   * @return
   */
  private Map<String, FileModelVo> findFileModelVoMap(Set<String> fileCodeSet) {
    if (CollectionUtils.isEmpty(fileCodeSet)) {
      return Maps.newHashMap();
    }


    List<OrdinaryFileVo> uploadFileVos = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(fileCodeSet)) {
      for (String fileCode:fileCodeSet) {
        OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(fileCode);
        if (ordinaryFileVo == null) {
          continue;
        }
        uploadFileVos.add(ordinaryFileVo);
      }
    }

    if (CollectionUtils.isEmpty(uploadFileVos)) {
      return Maps.newHashMap();
    }
    Map<String, FileModelVo> map = Maps.newHashMap();
    for (OrdinaryFileVo item : uploadFileVos) {
      FileModelVo modelVo = new FileModelVo();
      modelVo.setFileCode(item.getId());
      modelVo.setFileName(item.getOriginalFileName());
      map.put(item.getId(), modelVo);
    }
    return map;
  }
}
