package com.biz.crm.common.ie.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ImportTemplate;
import com.biz.crm.common.ie.local.entity.ImportTemplateDetail;
import com.biz.crm.common.ie.local.model.vo.ImportTreeModelVo;
import com.biz.crm.common.ie.local.repository.ImportTemplateRepository;
import com.biz.crm.common.ie.local.service.ImportTemplateDetailService;
import com.biz.crm.common.ie.local.service.ImportTemplateService;
import com.biz.crm.common.ie.sdk.dto.ImportTemplatePaginationDto;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 导入模板主表(ImportTemplate)表服务实现类
 *
 * @author sunx
 * @date 2022-05-10 10:27:59
 */
@Service("importTemplateService")
public class ImportTemplateServiceImpl implements ImportTemplateService {

  @Autowired private ImportTemplateRepository importTemplateRepository;

  @Autowired private ImportTemplateDetailService importTemplateDetailService;

  @Autowired private ImportExportProperties importExportProperties;

  @Autowired(required = false)
  private List<ImportProcess> importProcesses;

  @Override
  public Page<ImportTemplate> findByConditions(Pageable pageable, ImportTemplatePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    dto = Optional.ofNullable(dto).orElse(new ImportTemplatePaginationDto());
    dto.setApplicationName(this.getApplicationName());
    Page<ImportTemplate> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.importTemplateRepository.findByConditions(page, dto);
  }

  private String getApplicationName() {
    /*
     * 但是考虑代码已发布，仅影响开发环境和测试环境混用的时候，所以讲数据还原
     */
    return this.importExportProperties.getIeTaskGroup();
    // return this.importExportProperties.getApplicationName();
  }

  @Override
  public ImportTemplate findDetailByBusinessCode(String businessCode) {
    if (StringUtils.isBlank(businessCode)) {
      return null;
    }
    ImportTemplate template = this.importTemplateRepository.findByBusinessCode(businessCode);
    if (Objects.isNull(template)) {
      return null;
    }
    List<ImportTemplateDetail> list =
        this.importTemplateDetailService.findDetailByBusinessCode(template.getBusinessCode());
    template.setList(list);
    return template;
  }

  @Override
  @Transactional
  public ImportTemplate create(ImportTemplate importTemplate) {
    this.validation(importTemplate);
    importTemplate.setTenantCode(TenantUtils.getTenantCode());
    this.importTemplateRepository.saveOrUpdate(importTemplate);
    this.importTemplateDetailService.createBatch(
        importTemplate.getList(), importTemplate.getBusinessCode());
    return importTemplate;
  }

  @Override
  @Transactional
  public ImportTemplate update(ImportTemplate importTemplate) {
    this.validation(importTemplate);
    String currentId = importTemplate.getId();
    ImportTemplate current = importTemplateRepository.findById(currentId);
    Validate.notNull(current, "修改信息不存在");
    importTemplate.setTenantCode(current.getTenantCode());
    this.importTemplateRepository.saveOrUpdate(importTemplate);
    this.importTemplateDetailService.createBatch(
        importTemplate.getList(), importTemplate.getBusinessCode());
    return importTemplate;
  }

  @Override
  @Transactional
  public void deleteByBusinessCodes(List<String> businessCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(businessCodes), "业务编码集合不能为空");
    this.importTemplateRepository.deleteByBusinessCodes(businessCodes);
    this.importTemplateDetailService.deleteByBusinessCodes(businessCodes);
  }

  @Override
  public List<ImportTreeModelVo> findTemplateTree() {
    if (CollectionUtils.isEmpty(importProcesses)) {
      return Lists.newLinkedList();
    }

    List<ImportTemplate> existList =
        this.importTemplateRepository.findByAppName(this.getApplicationName());
    Set<String> set =
        existList.stream().map(ImportTemplate::getBusinessCode).collect(Collectors.toSet());

    List<ImportTreeModelVo> list = Lists.newArrayList();
    Map<String, List<ImportProcess>> mapList =
        this.importProcesses.stream()
            .collect(Collectors.groupingBy(ImportProcess::getBusinessCode));
    Map<String, String> map =
        this.importProcesses.stream()
            .collect(
                Collectors.toMap(
                    ImportProcess::getBusinessCode, ImportProcess::getBusinessName, (a, b) -> a));
    List<ImportProcess> importProcesses =
        mapList.getOrDefault(StringUtils.EMPTY, Lists.newLinkedList());

    for (ImportProcess item : importProcesses) {
      mapList.put(item.getTemplateCode(), Lists.newArrayList(item));
      map.put(item.getTemplateCode(), item.getTemplateName());
    }

    map.keySet().stream()
        .filter(StringUtils::isNotBlank)
        .sorted()
        .forEach(
            a -> {
              if (set.isEmpty() || !set.contains(a)) {
                ImportTreeModelVo cur = new ImportTreeModelVo();
                cur.setCode(a);
                cur.setName(map.get(a));
                List<ImportTreeModelVo> child = Lists.newArrayList();
                List<ImportProcess> list1 = mapList.get(a);
                list1.stream()
                    .sorted(Comparator.comparing(ImportProcess::order))
                    .forEach(
                        b -> {
                          ImportTreeModelVo cur1 = new ImportTreeModelVo();
                          cur1.setCode(b.getTemplateCode());
                          cur1.setName(b.getTemplateName());
                          child.add(cur1);
                        });
                cur.setChild(child);
                list.add(cur);
              }
            });
    return list;
  }

  @Override
  public List<ImportTreeModelVo> findTemplateSelectByBusinessCode(String businessCode) {
    if (CollectionUtils.isEmpty(importProcesses) || StringUtils.isBlank(businessCode)) {
      return Lists.newLinkedList();
    }
    Map<String, List<ImportTreeModelVo>> map = Maps.newHashMap();
    this.importProcesses.stream()
        .forEach(
            a -> {
              String code =
                  StringUtils.isBlank(a.getBusinessCode())
                      ? a.getTemplateCode()
                      : a.getBusinessCode();
              final List<ImportTreeModelVo> list = map.getOrDefault(code, Lists.newArrayList());
              final ImportTreeModelVo cur = new ImportTreeModelVo();
              cur.setCode(a.getTemplateCode());
              cur.setName(a.getTemplateName());
              cur.setOrder(a.order());
              list.add(cur);
              map.put(code, list);
            });
    final List<ImportTreeModelVo> list = map.get(businessCode);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return list.stream()
        .sorted(Comparator.comparing(ImportTreeModelVo::getOrder))
        .collect(Collectors.toList());
  }

  private void validation(ImportTemplate importTemplate) {
    importTemplate.setApplicationName(this.getApplicationName());
    Validate.notNull(importTemplate, "模板配置信息不能为空");
    Validate.notEmpty(importTemplate.getBusinessCode(), "业务编码不能为空");
    Validate.notEmpty(importTemplate.getBusinessName(), "业务名称不能为空");
    ImportTemplate entity =
        this.importTemplateRepository.findByBusinessCode(importTemplate.getBusinessCode());
    if (StringUtils.isBlank(importTemplate.getId()) && Objects.nonNull(entity)) {
      Validate.isTrue(false, "模板业务编码已存在");
    } else if (Objects.nonNull(entity)) {
      Validate.isTrue(importTemplate.getId().equals(entity.getId()), "模板业务编码已存在");
    }
    Validate.isTrue(CollectionUtils.isNotEmpty(importTemplate.getList()), "模板明细信息不能为空");
    for (ImportTemplateDetail item : importTemplate.getList()) {
      item.setApplicationName(this.getApplicationName());
    }
  }
}
