package com.biz.crm.common.ie.local.service.spring.impl;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.local.service.spring.rest.IMethodParameterTypeParse;
import com.biz.crm.common.ie.local.service.spring.SpringControllerApiService;
import com.biz.crm.common.ie.local.service.spring.rest.ISpringApiResultParse;
import com.biz.crm.common.ie.local.service.spring.rest.strategy.ParamsParseFactory;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.core.MethodParameter;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.condition.RequestMethodsRequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

/**
 * @author Kevin
 * @title SpringControllerApiServiceImpl
 * @date 2023/10/8 15:36
 * @description
 */
@Slf4j
@Component
public class SpringControllerApiServiceImpl implements SpringControllerApiService {

  @Value("${server.servlet.context-path}")
  private String contextPath;

  @Autowired
  private RequestMappingHandlerMapping requestMappingHandlerMapping;

  @Autowired
  private List<IMethodParameterTypeParse> iMethodParameterTypeParses;

  @Autowired
  private ParamsParseFactory paramsParseFactory;

  @Autowired
  private List<ISpringApiResultParse> iSpringApiResultParses;

  @Autowired
  private ApplicationContext applicationContext;

  @Override
  public Long getPageTotal(String url, String requestMapping, String requestParamJson) {
    int pageNum = 0;
    int pageSize = 1;
    final IPage<?> page = this.getPage(url,requestMapping,requestParamJson,pageNum,pageSize);
    return page.getTotal();
  }

  @Override
  public IPage<?> getPage(String url, String requestMapping, String requestParamJson, int pageNum,
      int pageSize) {
    //1.根据结果参数执行invoke方法
    final Object doInvoke = doInvokeByResult(url,requestMapping,requestParamJson,pageNum,pageSize);
    //2.分页参数转换
    IPage<?> page = null;
    for (ISpringApiResultParse iSpringApiResultParse : iSpringApiResultParses){
      if (iSpringApiResultParse.isApplicable(doInvoke)){
        page = iSpringApiResultParse.convertToPage(doInvoke);
        break;
      }
    }
    Validate.isTrue(page != null,"url地址返回数据结构无对应解析策略实现类(IDoInvokeResultParse) 无法解析 %s",url);
    return page;
  }

  /**
   * 1.根据结果参数执行invoke方法
   * @param url  调用的url地址
   * @param requestMapping 请求类型
   * @param requestParamJson 请求参数
   * @param pageNum 分页参数
   * @param pageSize 页面大小
   * @return
   */
  private Object doInvokeByResult(String url, String requestMapping, String requestParamJson, int pageNum, int pageSize) {
    Validate.isTrue(pageNum >= 0 ,"参数错误：pageNum 小于 0");
    Validate.isTrue(pageSize > 0,"参数错误：pageSize 小于等于 0");
    Map<String,Object> requestParamMap = new HashMap<>();
    try {
      requestParamMap = JSONObject.parseObject(requestParamJson,HashMap.class);
    }catch (JSONException jsonException){
      log.warn("url[{}] 的请求json无法解析：{}",url,requestParamJson);
    }

    //获取请求中第一页的分页数据 并根据数据进行修正获取分页的内容
    //对请求参数中的 page size  pageNum pageSize 按照页码进行纠偏
    final Object varPage = requestParamMap.get("page");
    {
      //page属性默认值为0
      int jsonPage = NumberUtils.toInt(varPage == null ? null : "" + varPage, 0);
      requestParamMap.put("page",jsonPage + pageNum);
      requestParamMap.put("size",pageSize);
    }
    final Object varPageNum =  requestParamMap.get("pageNum");
    {
      //pageNum属性默认值为1
      int jsonPageNum = NumberUtils.toInt(varPageNum == null ? null: "" + varPageNum, 1);
      requestParamMap.put("pageNum",jsonPageNum + pageNum);
      requestParamMap.put("pageSize",pageSize);
    }

    //执行invoke 方法后  返回值结果 按照标品Result结果集进行封装
    final Object doInvoke = this.doInvoke(url, requestMapping, requestParamMap);

    if (doInvoke instanceof Result){
      Result result = (Result) doInvoke;
      return result.getResult();
    }
    log.warn("接口返回对象非Result对象,{}",doInvoke);
    Validate.isTrue((doInvoke instanceof Result),"接口返回对象非Result对象,{}",doInvoke);
    return doInvoke;
  }

  @Override
  public Object doInvoke(String url, String requestMapping, Map<String, Object> requestParamMap) {
    //1.请求 url 地址获取handlerMethod 的方法 校验请求方法是否是服务器内部因存在的api方法
    HandlerMethod handlerMethod = this.getHandlerMethod(url,requestMapping);
    Validate.notNull(handlerMethod, "当前系统未匹配到对应的url: %s", url);
    //2.获取方法参数并校验
    Object [] args = this.getMethodArgumentValue(handlerMethod,requestParamMap);
    //3.执行invoke
    Object invoke = null;
    {
      final Method controllerMethod = handlerMethod.getMethod();
      final Object controllerBean = handlerMethod.getBean();
      try{
        //注：执行invoke时考虑参数取别名的情况
       invoke = controllerMethod.invoke(controllerBean,args);
      }catch (IllegalAccessException | InvocationTargetException exception){
        log.error("直接调用Api异常",exception);
        Validate.isTrue(false, "直接调用Api异常，URL%s，异常: %s", url, exception.getCause());
      }
      log.info("invoke:{}",invoke);
    }
    return invoke;
  }

  /**
   * 获取方法参数
   * @param handlerMethod
   * @param requestParamMap
   * @return
   */
  private Object[] getMethodArgumentValue(HandlerMethod handlerMethod, Map<String, Object> requestParamMap) {
   final MethodParameter[] methodParameters = handlerMethod.getMethodParameters();
   Object []  args = new Object[methodParameters.length];
   for (int index = 0; index < args.length; index++){
    final  MethodParameter methodParameter = methodParameters[index];
     Class<?> parameterType = methodParameter.getParameterType();
     Validate.isTrue(parameterType != HttpServletRequest.class,"不支持HttpServletRequest参数 [%s]",handlerMethod.getMethod());
     for (IMethodParameterTypeParse<T> iMethodParameterTypeParse : iMethodParameterTypeParses){
       //满足参数校验的参数处理
       if (iMethodParameterTypeParse.supportsParameter(methodParameter)){
         args[index] = iMethodParameterTypeParse.resolveArgument(methodParameter, requestParamMap);
         break;
       }else {
         continue;
       }
     }
     //取到值则跳出当前循环
     if (args[index] != null){
       continue;
     }
     // 使用普通对象创建获取
     //1.基本数据类型
     String parameterName = methodParameter.getParameterName();
     if (StringUtils.isNotBlank(parameterName)){
      final Object value = requestParamMap.get(parameterName);
      if (value != null){
        if (value.getClass() == parameterType){
          //类型相同 直接赋值
          args[index] = value;
        }else {
          //类型不同 调用验证
          args[index] = paramsParseFactory.resolveArgument(parameterType,"" + value);
        }
      }
     }
     //2.非基本数据类型
     if (args[index] == null){
       try{
         args[index] = BeanUtil.toBeanIgnoreCase(requestParamMap, parameterType, true);
         //处理日期String 转 Date 正则验证不成功问题
         args[index] = this.buildArgs(requestParamMap,parameterType,args[index]);
       }catch (Exception e){
         log.error("不支持%s参数类型 [%s]",parameterType,handlerMethod.getMethod());
         Validate.isTrue(false,"不支持%s参数 [%s]",parameterType,handlerMethod.getMethod());
       }
     }
     log.info("param[{}] = {}",index,args[index]);
   }
    if (log.isDebugEnabled()){
      log.debug("请求参数:{}", Arrays.toString(args));
    }
    return args;
  }


  /**
   * 构建对象参数
   * @param
   * @param requestParamMap
   * @param parameterType
   * @return
   * @throws IllegalAccessException 对象异常
   */
  private Object buildArgs(Map<String, Object> requestParamMap,
      Class<?> parameterType, Object obj){
    Field[] declaredFields = parameterType.getDeclaredFields();
    JSONObject jsonObject = JsonUtils.toJSONObject(obj);
    for (Field field : declaredFields) {
      String name = field.getName();
      Object o = requestParamMap.get(name);
      field.setAccessible(true);
      Class<?> type = field.getType();
      if (o == null) {
        continue;
      }
      //转换之前有值  转换之后没有值
      if (o.getClass() != type && jsonObject.get(name) == null) {
        if (o instanceof String) {
          if (type == Date.class) {
            DateTimeFormat annotation = field.getAnnotation(DateTimeFormat.class);
            if (annotation == null) {
              continue;
            }
            String pattern = annotation.pattern();
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
            Date parse = null;
            try {
              parse = simpleDateFormat.parse(String.valueOf(o));
            } catch (ParseException e) {
              log.error("日期格式转换异常:");
              log.error(e.getMessage(),e);
            }
            if (parse == null) {
              continue;
            }
            jsonObject.putIfAbsent(name,parse);
          }
        }
      }
    }
    return jsonObject.toJavaObject(parameterType);
  }

  /**
   * 请求 url 地址获取handlerMethod 的方法 校验请求方法是否是服务器内部因存在的api方法
   * @param url
   * @param requestMapping
   * @return
   */
  private HandlerMethod getHandlerMethod(String url, String requestMapping) {
    //1.判断路径是否有效
    final String resolveLookUpPath = this.getPath(url);
    //2.处理器映射器 获取处理方法
    Map<RequestMappingInfo, HandlerMethod> handlerMethods = requestMappingHandlerMapping.getHandlerMethods();
    HandlerMethod handlerMethod = null;
    for (Map.Entry<RequestMappingInfo,HandlerMethod> handlerMethodEntry : handlerMethods.entrySet()){
      //请求映射信息
      final RequestMappingInfo requestMappingInfo = handlerMethodEntry.getKey();
      //处理器方法
      final HandlerMethod tempHandlerMethod = handlerMethodEntry.getValue();
      //获取请求映射信息中 获取定向（指向） 路径
      final Set<String> directPaths = requestMappingInfo.getDirectPaths();
      if (directPaths.contains(resolveLookUpPath)){
        log.info("requestMappingInfo ===>{}",requestMappingInfo);
        //从请求映射信息中获取方法详情
        final RequestMethodsRequestCondition methodsCondition = requestMappingInfo.getMethodsCondition();
        //获取方法集
        Set<RequestMethod> methods = methodsCondition.getMethods();
        for (RequestMethod requestMethod : methods){
          if (requestMethod.name().equalsIgnoreCase(requestMapping)){
              handlerMethod = tempHandlerMethod;
              handlerMethod = handlerMethod.createWithResolvedBean();
              break;
          }
        }
      }
    }
    return handlerMethod;
  }

  /**
   * 判断请求url地址是否有效 并返回
   * @param url
   * @return
   */
  private String getPath(String url) {
    String tempUrl;
    //1.构建完整的url地址
    //将url地址转换为小写  判断是否以http开头
    if (url.toLowerCase().startsWith("http")){
      tempUrl = url;
    }else {
      if (url.startsWith("/")){
        tempUrl = "http://127.0.0.1" + url;
      }else {
        tempUrl = "http://127.0.0.1/" + url;
      }
    }
    //2.地址中涉及特殊字符，如 '|' '&' 等，所以不能直接用string 代替  URI 来访问
    String resolvedLookUpPath = null;
    try {
      URL urlObj = new URL(tempUrl);
      resolvedLookUpPath = urlObj.getPath();
    }catch (MalformedURLException e){
      Validate.isTrue(false,"URL转换失败，请检查，%s",url);
    }
    //判断访问路径中是否以服务路径开头 例如/crm-mdm/...
    if (resolvedLookUpPath.startsWith(contextPath)){
      //去掉服务路径
     resolvedLookUpPath = resolvedLookUpPath.substring(contextPath.length());
    }
    return resolvedLookUpPath;
  }
}
