package com.biz.crm.common.ie.local.service.strategy;

import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.biz.crm.common.ie.local.model.vo.ImportExcelReadModelVo;
import com.biz.crm.common.ie.sdk.enums.WriteErrorExcelModelEnums;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.excel.vo.CrmExcelVo;
import com.biz.crm.common.ie.sdk.strategy.ImportExcelStrategy;
import com.biz.crm.common.ie.sdk.vo.ExcelItemVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 导入任务读监听实现
 *
 * @author sunx
 * @date 2022/6/8
 */
@Slf4j
public class CrmImportExcelReadListener extends AnalysisEventListener<Map<Integer, Object>> {
  private ImportExcelReadModelVo readVo;
  private ExcelWriter excelWriter;
  private WriteSheet writeSheet;

  private int failedNum = 0;
  private int successNum = 0;
  private int cursor = 0;

  private List<List<Object>> cachedDataList = Lists.newLinkedList();
  private Map<Integer, String> cacheErr = Maps.newHashMap();
  private Map<Integer, String> headMaps  = Maps.newHashMap();
  private LinkedHashMap<Integer, CrmExcelVo> cacheData = Maps.newLinkedHashMap();
  private int cacheCursor = 0;
  private String account;
  private ImportExcelStrategy importExcelStrategy;

  public CrmImportExcelReadListener(
          ImportExcelReadModelVo readVo, ExcelWriter excelWriter, WriteSheet writeSheet, String account,ImportExcelStrategy importExcelStrategy) {
    this.readVo = readVo;
    this.excelWriter = excelWriter;
    this.writeSheet = writeSheet;
    this.account = account;
    this.importExcelStrategy =importExcelStrategy;
  }

  /**
   * 具体的导入业务 通过执行execute方法完成导入任务
   *
   * @param data    one row value. Is is same as {@link AnalysisContext#readRowHolder()}
   * @param context analysis context
   */
  @Override
  public void invoke(Map<Integer, Object> data, AnalysisContext context) {
    if (this.readVo.getProcess().getBatchCount().compareTo(0) <= 0) {
      return;
    }
    if (this.cacheCursor == this.readVo.getProcess().getBatchCount()) {
      this.execute();
    }
    List<Object> cur = this.findData(data);
    cachedDataList.add(cur);
    cursor++;
    try {
      CrmExcelVo o = (CrmExcelVo) this.readVo.getProcess().findCrmExcelVoClass().newInstance();
      importExcelStrategy.setCrmExcelVoValue(o,data,headMaps);
      //BzExcelUtil.setCrmExcelVoValue(o, data);
      cacheData.put(cacheCursor++, o);
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error(e.getMessage(),e);
      cur.add(e.getMessage());
      cacheErr.put(cacheCursor++, e.getMessage());
      failedNum += 1;
    } catch (Exception e) {
      log.error(e.getMessage(),e);
      cur.add("数据异常");
      cacheErr.put(cacheCursor++, "数据异常");
      failedNum += 1;
    }
    this.sendProcessMsg(true);
  }

  @Override
  public void doAfterAllAnalysed(AnalysisContext context) {
    this.execute();
    this.sendProcessMsg(false);
    this.readVo.setSuccessNum(this.successNum);
    this.readVo.setFailedNum(this.failedNum);
  }

  /**
   * 业务处理
   */
  private void execute() {
    if (cacheData == null || cacheData.isEmpty()) {
      return;
    }
    Map<Integer, String> map = null;
    ImportProcess<CrmExcelVo> process = this.readVo.getProcess();
    try {
      map = process.execute(cacheData, this.readVo.getParamsVo(), this.readVo.getParams());
      map = Optional.ofNullable(map).orElse(Maps.newHashMap());
      long count = 0;
      if (map.isEmpty()) {
        count = cacheData.size();
      } else {
        final Set<Integer> set = cacheData.keySet();
        Map<Integer, String> finalMap = map;
        count = set.parallelStream().filter(a -> !finalMap.keySet().contains(a)).count();
      }
      successNum += new Long(count).intValue();
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("[1]直接保存异常" ,e);
      for (Integer item : cacheData.keySet()) {
        cacheErr.put(item, "处理异常:" + e.getMessage());
        failedNum++;
      }
    } catch (Exception e) {
      log.error("[1]直接保存校验异常" ,e);
      for (Integer item : cacheData.keySet()) {
        cacheErr.put(item, "处理异常");
        failedNum++;
      }
    }
    if (map != null && !map.isEmpty()) {
      for (Entry<Integer, String> item : map.entrySet()) {
        cacheErr.put(item.getKey(), "处理异常:" + item.getValue());
        failedNum++;
      }
    }
    
    WriteErrorExcelModelEnums writeErrorExcelModel = process.getWriteErrorExcelModel();
    this.writeErrorExcel(writeErrorExcelModel);
    this.resetCache();
  }

  /**
   * 写错误文件
   * @param writeErrorExcelModel 
   */
  private void writeErrorExcel(WriteErrorExcelModelEnums writeErrorExcelModel) {
    int i = 0;
    List<List<Object>> errDataList = Lists.newLinkedList();
    for (List<Object> item : this.cachedDataList) {
      final String errMsg = cacheErr.get(i++);
      switch (writeErrorExcelModel) {
        case ALL_ROW:
          if (StringUtils.isNotBlank(errMsg)) {
            item.add(errMsg.trim()); // 追加错误信息
          } else {
            item.add("导入-成功");
          }
          errDataList.add(item);
          break;
        case ONLY_ERROR_ROW:
        default:
          if (StringUtils.isNotBlank(errMsg)) {
            item.add(errMsg.trim());// 追加错误信息
            errDataList.add(item);
          }
      }
    }
    if (CollectionUtils.isEmpty(errDataList)) {
      return;
    }
    this.excelWriter.write(errDataList, writeSheet);
  }

  /**
   * 根据解析的行记录获取业务处理对象数据
   *
   * @param data
   * @return
   */
  private List<Object> findData(Map<Integer, Object> data) {
    if (data == null || data.isEmpty()) {
      return Lists.newLinkedList();
    }
    List<ExcelItemVo> items = Lists.newLinkedList();
    for (Entry<Integer, Object> item : data.entrySet()) {
      final ExcelItemVo cur = new ExcelItemVo();
      cur.setColumn(item.getKey());
      cur.setData(item.getValue());
      items.add(cur);
    }
    return items.stream()
            .sorted(Comparator.comparing(ExcelItemVo::getColumn))
            .map(ExcelItemVo::getData)
            .collect(Collectors.toList());
  }

  /**
   * 重置缓存
   */
  private void resetCache() {
    this.cacheCursor = 0;
    this.cachedDataList = Lists.newLinkedList();
    this.cacheErr = Maps.newHashMap();
    this.cacheData = Maps.newLinkedHashMap();
    this.headMaps = Maps.newHashMap();
  }

  /**
   * 推送进度信息(过程只需要每10条推送一次)
   *
   * @param processFlag 是否是处理过程中
   */
  private void sendProcessMsg(Boolean processFlag) {
    if (Boolean.TRUE.equals(processFlag) && this.cursor % 10 != 0) {
      return;
    }
    log.info(
            "任务{},第1页,总计{},当前进度{},成功{},失败{}",
            this.readVo.getParamsVo().getTaskCode(),
            this.readVo.getTotal(),
            this.cursor,
            this.successNum,
            this.failedNum);
    // 推送进度信息
    this.readVo
            .getMsgBean()
            .sendProcessMsg(
                    this.readVo.getParamsVo().getTaskCode(),
                    this.readVo.getTotal(),
                    this.cursor,
                    this.successNum,
                    this.failedNum,
                    this.account);
  }

  /**
   * 获取表头信息
   *
   *
   * */
  public void invokeHeadMap(Map<Integer, String> headMap, AnalysisContext context) {
    this.headMaps.putAll(headMap);
  }
}
