package com.biz.crm.common.ie.local.service.strategy;

import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelReader;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.read.metadata.ReadSheet;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.alibaba.fastjson.JSONArray;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.ie.local.bean.ExportSendProcessMsgBean;
import com.biz.crm.common.ie.local.bean.ImportSendProcessMsgBean;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.listener.EasyExcelListener;
import com.biz.crm.common.ie.local.model.vo.ImportExcelReadModelVo;
import com.biz.crm.common.ie.local.repository.ExportTaskRepository;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.enums.ExcelFileTypeEnum;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExportProcessEnum;
import com.biz.crm.common.ie.sdk.enums.ExportTypeEnum;
import com.biz.crm.common.ie.sdk.enums.ImportDataStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ImportProcessEnum;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelExport;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelImport;
import com.biz.crm.common.ie.sdk.excel.process.ExportProcess;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.excel.strategy.CrmExcelProcessStrategy;
import com.biz.crm.common.ie.sdk.excel.strategy.CrmExportColumnStrategy;
import com.biz.crm.common.ie.sdk.excel.util.BzExcelUtil;
import com.biz.crm.common.ie.sdk.excel.util.BzExcelUtil.CrmExcelColumnField;
import com.biz.crm.common.ie.sdk.excel.vo.ColumnVo;
import com.biz.crm.common.ie.sdk.excel.vo.FunctionPermissionVo;
import com.biz.crm.common.ie.sdk.exception.CrmExportException;
import com.biz.crm.common.ie.sdk.strategy.ImportExcelStrategy;
import com.biz.crm.common.ie.sdk.vo.EsParagraphFieldRangeVo;
import com.biz.crm.common.ie.sdk.vo.ExcelFileVo;
import com.biz.crm.common.ie.sdk.vo.ExportProcessMsgVo;
import com.biz.crm.common.ie.sdk.vo.ExportTaskDetailVo;
import com.biz.crm.common.ie.sdk.vo.ExportTaskProcessVo;
import com.biz.crm.common.ie.sdk.vo.ImportExcelLocalFile;
import com.biz.crm.common.ie.sdk.vo.ImportTaskHandlerResultVo;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.bizunited.nebula.mars.sdk.service.MarsAuthorityExcludedFieldDetailService;
import com.google.common.base.CaseFormat;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.File;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import lombok.extern.slf4j.Slf4j;

/**
 * 导入导出excel处理策略实现-默认策略实现
 *
 * @author sunx
 * @date 2022/6/8
 */
@Slf4j
public class DefaultExcelProcessStrategyImpl implements CrmExcelProcessStrategy {

  @Autowired
  private ImportSendProcessMsgBean importMsgBean;
  @Autowired
  private ExportSendProcessMsgBean exportMsgBean;
  @Autowired
  private ImportExportProperties importExportProperties;
  @Autowired
  private ExportSendProcessMsgBean exportSendProcessMsgBean;
  @Autowired(required = false)
  private MarsAuthorityExcludedFieldDetailService marsAuthorityExcludedFieldDetailService;
  @Autowired(required = false)
  private CrmExportColumnStrategy crmExportColumnStrategy;
  @Autowired(required = false)
  private List<ImportExcelStrategy> importExcelStrategies;
  @Value("${import.type:#{'default'}}")
  private String importType;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private ExportTaskRepository exportTaskRepository;
  
  /**
   * 根据文件路径生成一个ExcelWriter对象
   * 
   * @param errorFilePath
   * @param process
   * @return
   */
  private ExcelWriter newErrorExcelWriter(final File errorFilePath, ImportProcess process) {
    ExcelWriter excelWriter =
        EasyExcelFactory.write(errorFilePath)
            .head(this.findHead(process.findCrmExcelVoClass()))
            .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
            .build();
    return excelWriter;
  }
  
  @Override
  public ImportTaskHandlerResultVo importExcel(
      ImportProcess process, TaskGlobalParamsVo paramsVo,
      Map<String, Object> params, Map<String, ExcelFileVo> fileMap) {
    Validate.isTrue(
        Objects.nonNull(paramsVo) && StringUtils.isNotBlank(paramsVo.getTaskCode()), "任务公参编码为必要参数");
    final ImportTaskHandlerResultVo resultVo = new ImportTaskHandlerResultVo();
    final ExcelFileVo originalFile = fileMap.get(ExcelFileTypeEnum.ORIGINAL.getDictCode());
    final ExcelFileVo errorFileVo = fileMap.get(ExcelFileTypeEnum.ERROR.getDictCode());
    File errorFilePath = new File(errorFileVo.getPath());
    CrmExcelImport crmExcelImport =
        (CrmExcelImport) process.findCrmExcelVoClass().getAnnotation(CrmExcelImport.class);

    final ImportExcelReadModelVo readVo = new ImportExcelReadModelVo();
    readVo.setMsgBean(importMsgBean);
    readVo.setProcess(process);
    readVo.setParamsVo(paramsVo);
    readVo.setParams(params);
    ExcelWriter excelWriter = null;
    int failedNum = 0;
    try {
      final int totalRow = findExcelTotalRowIgnoreBlankLines(originalFile.getPath(), crmExcelImport.sheetNo(), process);
      Validate.isTrue(totalRow > crmExcelImport.startRow(), "导入excel总记录数须大于开始处理行数");
      resultVo.setTotal(totalRow - crmExcelImport.startRow());
      excelWriter = this.newErrorExcelWriter(errorFilePath, process);
      WriteSheet writeSheet = EasyExcelFactory.writerSheet(ImportExportConstant.SHEET1).build();
      readVo.setTotal(resultVo.getTotal());
      List<ImportExcelStrategy> importExcelStrategyList = this.importExcelStrategies.stream()
          .filter(importExcelStrategy -> importExcelStrategy.importExcelType().equals(importType))
          .collect(Collectors.toList());
      Validate.isTrue(!CollectionUtils.isEmpty(importExcelStrategyList), "导入匹配策略不存在，请检查yml配置信息import.type是否配置");
      ImportExcelStrategy importExcelStrategy = importExcelStrategyList.get(0);

      // 保存数据监听器
      AnalysisEventListener<Map<Integer, Object>> saveDbListener;
      /*
       * 执行分2步,校验、保存
       * 
       * step-01、校验（可选）
       * 
       * step-02、保存：如果校验失败，则不进行保存
       * 
       */
      // step-01、校验（可选）
      if (readVo.getProcess().importBeforeValidationFlag()) {
        // 模式1：先校验、后导入
        CrmImportExcelReadVerifyListener verifyDbListener =
            new CrmImportExcelReadVerifyListener(readVo, excelWriter, writeSheet, paramsVo.getCreateAccount(),
                importExcelStrategy);
        EasyExcelFactory.read(originalFile.getPath(), verifyDbListener)
            .sheet().headRowNumber(crmExcelImport.startRow()).doRead();
        // 解决偶发性bug 校验任务结束强制性推送一条最终状态消息
        ThreadUtil.sleep(50, TimeUnit.MILLISECONDS);
        this.importMsgBean.sendProcessMsg(
            readVo.getParamsVo().getTaskCode(),
            readVo.getTotal(),
            readVo.getTotal(),
            0,
            readVo.getSuccessNum(),
            readVo.getFailedNum(),
            readVo.getParamsVo().getCreateAccount(),
            ImportProcessEnum.RUNNING_VERIFY);
        failedNum = Integer.max(readVo.getFailedNum(), 0);
        readVo.setFailedNum(failedNum);

        // 发送消息，清理现场
        if (failedNum > 0) {
          // 校验失败，推送一条校验失败消息
          this.importMsgBean.sendProcessMsg(readVo.getParamsVo().getTaskCode(), ImportProcessEnum.VERIFY_FAIL,
              readVo.getParamsVo().getCreateAccount());
          // == 保存数据库监听器=校验失败，不进行保存
          saveDbListener = null;
        } else {
          // 校验通过，推送一条校验成功消息
          this.importMsgBean.sendProcessMsg(readVo.getParamsVo().getTaskCode(), ImportProcessEnum.VERIFY_SUCCESS,
              readVo.getParamsVo().getCreateAccount());
          // 校验通过，删除已经生成的校验数据
          excelWriter.close();
          FileUtil.del(errorFilePath);
          excelWriter = this.newErrorExcelWriter(errorFilePath, process);
          // == 保存数据库监听器=校验通过时
          saveDbListener = new CrmImportExcelReadSaveListener(readVo, excelWriter, writeSheet,
              paramsVo.getCreateAccount(), importExcelStrategy);
        }
      } else {
        // == 保存数据库监听器=不验证时
        saveDbListener = new CrmImportExcelReadListener(readVo, excelWriter, writeSheet, paramsVo.getCreateAccount(),
            importExcelStrategy);
      }

      // step-02、保存：如果校验失败，则不进行保存
      if (saveDbListener != null) {
        // 模式2：直接导入
        EasyExcelFactory.read(originalFile.getPath(), saveDbListener)
            .sheet().headRowNumber(crmExcelImport.startRow()).doRead();
        // 解决偶发性bug 保存任务结束强制性推送一条最终状态消息
        ThreadUtil.sleep(50, TimeUnit.MILLISECONDS);
        this.importMsgBean.sendProcessMsg(
            readVo.getParamsVo().getTaskCode(),
            readVo.getTotal(),
            readVo.getTotal(),
            readVo.getTotal(),
            readVo.getSuccessNum(),
            readVo.getFailedNum(),
            readVo.getParamsVo().getCreateAccount(),
            ImportProcessEnum.RUNNING_SAVE);
        
        if (readVo.getFailedNum() == 0) {
          resultVo.setImportDataStatus(ImportDataStatusEnum.IMPORT_SUCCESS); // 导入成功
        } else {
          // 导入失败或者导入部分失败
          resultVo.setImportDataStatus(readVo.getTotal() == readVo.getFailedNum() ? ImportDataStatusEnum.IMPORT_FAILED
              : ImportDataStatusEnum.IMPORT_FAILED_SECTION);
        }
      } else {
        // 校验失败或者校验部分失败
        resultVo.setImportDataStatus(readVo.getTotal() == readVo.getFailedNum() ? ImportDataStatusEnum.VERIFY_FAILED
            : ImportDataStatusEnum.VERIFY_FAILED_SECTION);
      }
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("**** 导入出错了 ****", e);
      Validate.isTrue(false, e.getMessage());
    } catch (Exception e) {
      log.error("**** 导入出错了 ****", e);
      Validate.isTrue(false, String.format("导入出错了:%s", e.getMessage()));
    } finally {
      if (excelWriter != null) {
        excelWriter.close();
      }
      failedNum = Integer.max(readVo.getFailedNum(), 0);
      if (failedNum == 0) {
        FileUtil.del(errorFilePath);
      }
    }
    resultVo.setFailedNum(failedNum);
    return resultVo;
  }

  @Override
  public void exportDetail(
      ExportTaskProcessVo task, ExportProcess process,
      Map<String, Object> paramsMap, String filePath) {

    List<List<Object>> excelDataList = null;
    Integer count = null;
    {
      final JSONArray data = process.getData(task, paramsMap);
      Validate.isTrue(Objects.nonNull(data) && !data.isEmpty(), "未获取到需要导出的数据");
      count = BzExcelUtil.findMsgCount(task.getPageSize());
      excelDataList = this.findExportData(data, process, task);
    }
    Validate.isTrue(!CollectionUtils.isEmpty(excelDataList), "未获取到导出数据");
    ExcelWriter excelWriter = null;
    if (!FileUtil.exist(filePath)) {
      FileUtil.touch(filePath);
    }
    try {
      List<List<String>> excelHead = this.findHead(process.findCrmExcelVoClass());
      excelWriter =
          EasyExcelFactory.write(filePath)
              .head(excelHead)
              .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
              .build();
      WriteSheet writeSheet = EasyExcelFactory.writerSheet(ImportExportConstant.SHEET1).build();
      // 账号
      String account = task.getCreateAccount();
      if (count <= 10) {
        excelWriter.write(excelDataList, writeSheet);
        excelDataList = null; // 主动清理内存
      } else {
        List<List<List<Object>>> result = Lists.partition(excelDataList, count);
        excelDataList = null; // 主动清理内存
        for (int stepNum = 0, pageSize = result.size(); stepNum < pageSize; ++stepNum) {
          // 获取数据
          List<List<Object>> item = result.get(stepNum);
          excelWriter.write(item, writeSheet);
          // 进度百分比
          int cursor = (1 + stepNum) * 100 / pageSize;
          // 发进度提示
          this.postStepWebSocketMsg(task, account, cursor, false);
        }
      }
      // 等待1秒
      Thread.sleep(1000);
      this.postStepWebSocketMsg(task, account, 100, false);
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("**** 导出出错了 ****", e);
      Validate.isTrue(false, e.getMessage());
    } catch (Exception e) {
      log.error("**** 导出出错了 ****", e);
      Validate.isTrue(false, "excel导出出错了");
    } finally {
      if (excelWriter != null) {
        excelWriter.close();
      }
    }
  }

  @Override
  public void exportDetailForEsParagraph(ExportTaskProcessVo task, ExportProcess process,
      Map<String, Object> paramsMap, String filePath) {
    List<List<String>> excelHead = this.findHead(process.findCrmExcelVoClass());
    if (!FileUtil.exist(filePath)) {
      FileUtil.touch(filePath);
    }
    try (ExcelWriter excelWriter =
        EasyExcelFactory.write(filePath)
            .head(excelHead)
            .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
            .build()) {
      List<List<EsParagraphFieldRangeVo>> paragraphFieldRanges = task.getTotalParagraphFieldRanges();
      paragraphFieldRanges.forEach(ranges -> {
        task.setQueryParagraphFieldRanges(ranges);
        final JSONArray data = process.getData(task, paramsMap);
        Validate.isTrue(Objects.nonNull(data) && !data.isEmpty(), "未获取到需要导出的数据");
        List<List<Object>> excelDataList = this.findExportData(data, process, task);

        Validate.isTrue(!CollectionUtils.isEmpty(excelDataList), "未获取到导出数据");

        WriteSheet writeSheet = EasyExcelFactory.writerSheet(ImportExportConstant.SHEET1).build();
        excelWriter.write(excelDataList, writeSheet);
      });
      // 等待1秒
      Thread.sleep(1000);
      this.postStepWebSocketMsg(task, task.getCreateAccount(), 100, false);
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("**** 导出出错了 ****", e);
      Validate.isTrue(false, e.getMessage());
    } catch (Exception e) {
      log.error("**** 导出出错了 ****", e);
      Validate.isTrue(false, "excel导出出错了");
    }
  }

  @Override
  public void exportDetailForEuropa(
      ExportTaskProcessVo task, ExportProcess<?> process,
      Map<String, Object> paramsMap, String filePath,
      FunctionPermissionVo funPermVo) throws CrmExportException {
    // XZK 20230529 将数据库读取和文件写入分离
    StopWatch sw = null;
    // #################### 获取数据视图中允许展示的字段表头信息
    if (log.isDebugEnabled()) {
      sw = new StopWatch();
      sw.start("获取数据视图表头");
    }
    // 查找字段也需要任务的拓展参数
    funPermVo.setParamsJson(task.getParametersJson());
    List<ColumnVo> showDataviewColumn = this.crmExportColumnStrategy.getShowDataviewColumn(funPermVo);
    List<ColumnVo> columnList = Optional
        .of(showDataviewColumn)
        .orElseThrow(() -> new IllegalArgumentException("未获取到Europa视图的可查看对象"));
    Validate.isTrue(false == columnList.isEmpty(), "数据视图【%s】当前无可导出字段授权", task.getBusinessCode());
    if (log.isDebugEnabled()) {
      sw.stop();
      sw.start("创建文件");
    }
    if (!FileUtil.exist(filePath)) {
      FileUtil.touch(filePath);
    }
    if (log.isDebugEnabled()) {
      sw.stop();
    }
    try {
      if (log.isDebugEnabled()) {
        sw.start("读取数据库并写入文件");
      }
      this.exportDetailWithPage(filePath, this.findHeadByEuropa(columnList), task, process
          , paramsMap, ExportTypeEnum.EUROPA_EXPORT, columnList);
      if (log.isDebugEnabled()) {
        sw.stop();
      }
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("**** 数据视图-导出出错了 ****", e);
      // Validate.isTrue(false, e.getMessage());
      throw new CrmExportException(e.getMessage(), e.getCause());
    } catch (Exception e) {
      log.error("**** 数据视图-导出出错了 ****", e);
      throw new CrmExportException(e.getMessage(), e.getCause());
    } finally {
      if (log.isDebugEnabled()) {
        if (sw != null) {
          log.debug("{}", sw.prettyPrint());
        }
      }
    }
    if (log.isDebugEnabled()) {
      log.debug("任务{},数据量{},文件{},执行结果{}", task.getDetailCode(), task.getPageSize(), filePath, sw.prettyPrint());
    }
  }

  @Override
  public void exportDetailForWebApi(ExportTaskProcessVo task, ExportProcess<?> process,
      Map<String, Object> paramsMap, String absolutePath,
      FunctionPermissionVo functionPermissionVo) throws CrmExportException {
    StopWatch sw = null;
    // #################### 获取数据视图中允许展示的字段表头信息
    if (log.isDebugEnabled()) {
      sw = new StopWatch();
      sw.start("获取webApi导出表头");
    }
    // 查找字段也需要任务的拓展参数
    functionPermissionVo.setParamsJson(task.getParametersJson());
    List<ColumnVo> showDataviewColumn = this.crmExportColumnStrategy.getShowDataviewColumn(functionPermissionVo);
    List<ColumnVo> columnList = Optional
        .of(showDataviewColumn)
        .orElseThrow(() -> new IllegalArgumentException("未获取到页面配置的可查看对象"));
    Validate.isTrue(false == columnList.isEmpty(), "页面配置【%s】当前无可导出字段授权", task.getBusinessCode());
    if (log.isDebugEnabled()) {
      sw.stop();
    }
    // #################### 创建文件
    if (log.isDebugEnabled()) {
      sw.start("创建文件");
    }
    if (!FileUtil.exist(absolutePath)) {
      FileUtil.touch(absolutePath);
    }
    if (log.isDebugEnabled()) {
      sw.stop();
    }
    try {
      if (log.isDebugEnabled()) {
        sw.start("读取数据库并写入文件");
      }
      this.exportDetailWithPage(absolutePath, this.findHeadByEuropa(columnList), task, process
          , paramsMap, ExportTypeEnum.API_EXPORT, columnList);
      if (log.isDebugEnabled()) {
        sw.stop();
      }
    } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
      log.error("**** webApi-导出出错了 ****", e);
      // Validate.isTrue(false, e.getMessage());
      throw new CrmExportException(e.getMessage(), e.getCause());
    } catch (Exception e) {
      log.error("**** webApi-导出出错了 ****", e);
      throw new CrmExportException(e.getMessage(), e.getCause());
      // Validate.isTrue(false, "excel导出出错了");
    } finally {
      if (log.isDebugEnabled()) { // debug级别，加入日志
        if (sw != null) {
          log.debug("{}", sw.prettyPrint());
        }
      }
    }
    if (log.isDebugEnabled()) {
      log.debug("任务{},数据量{},文件{},执行结果{}", task.getDetailCode(), task.getPageSize(), absolutePath, sw.prettyPrint());
    }
  }

  /**
   * 以分页方式导出子任务文件
   *
   * @param filePath 文件路径
   * @param excelHead 导出文件标题行
   * @param task 子任务信息
   * @param process 导出任务实现类
   * @param paramsMap 导出参数
   * @param exportType 导出类型
   * @param columnList 导出字段
   * @throws Exception 异常
   */
  private void exportDetailWithPage(String filePath, List<List<String>> excelHead
      , ExportTaskProcessVo task, ExportProcess<?> process, Map<String, Object> paramsMap
      , ExportTypeEnum exportType, List<ColumnVo> columnList) throws Exception{
    try (ExcelWriter excelWriter =
        EasyExcelFactory.write(filePath)
            .head(excelHead)
            .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
            .build()) {
      WriteSheet writeSheet = EasyExcelFactory.writerSheet(ImportExportConstant.SHEET1).build();
      String createAccount = task.getCreateAccount();
      // 子任务分割文件的大小
      int splitPageSize = process.getPageSize();
      int queryPageSize = Objects.nonNull(process.getQueryPageSize()) && process.getQueryPageSize() > 0
          ? process.getQueryPageSize() : importExportProperties.getExportProperties().getQueryPageSize();
      Validate.isTrue(splitPageSize % queryPageSize == 0, "导出子任务拆分大小必须是查询数据分页大小的整数倍!");
      int startPageNo = (task.getPageNo()) * (splitPageSize / queryPageSize);
      int endPageNo = (task.getPageNo() + 1) * (splitPageSize / queryPageSize) - 1;
      int dealTotal = 0;
      for (int pageNo = startPageNo; pageNo <= endPageNo; pageNo++) {
        // 设置获取导出数据时入参中的分页查询大小, 同时process.getPageSize获取到的值也从文件分割大小变成了分页查询大小
        paramsMap.put(ImportExportConstant.IE_EXPORT_COMMON_PARAM_PAGE_SIZE, queryPageSize);
        paramsMap.put(ImportExportConstant.IE_EXPORT_COMMON_PARAM_PAGE_NO, pageNo);
        final JSONArray data = process.getData(task, paramsMap);
        if (Objects.isNull(data) || data.isEmpty()) {
          // 第一页数据没有获取到,则将会终止导出,并且抛出无数据异常信息
          Validate.isTrue(pageNo > startPageNo, "未获取到需要导出的数据");
          // 如果不是第一页则停止数据查询
          break;
        }
        dealTotal += data.size();
        // JSONArray 转 List
        List<JSONObject> dataList = data.stream().map(JSONUtil::parseObj).collect(Collectors.toList());
        List<List<Object>> excelDataList;
        switch (exportType) {
          case API_EXPORT:
          case EUROPA_EXPORT:
            //对即将写入excel的数据进行二次清洗，填充数据字典
            dataList = this.crmExportColumnStrategy.fullDictCode(dataList, columnList);
            excelDataList = this.findExportDataForEuropa(dataList, process, task, columnList);
            break;
          default:
            throw new IllegalArgumentException("不支持的导出类型");
        }
        // 获取导出任务数据
        excelWriter.write(excelDataList, writeSheet);
        // 发进度提示
        this.postStepWebSocketMsg(task, createAccount, dealTotal * 100 / task.getTotal(), false);
      }
      // 等待500毫秒再发送任务完成100%进度
      Thread.sleep(500);
      this.postStepWebSocketMsg(task, createAccount, 100, false);
    }
  }
  /**
   * 新-推送步进进度
   * @param task  导出任务实体
   * @param account 当前账户
   * @param cursor  当前进度
   * @param mainFlag  主任务标识
   */
  private void postStepWebSocketMsg(ExportTaskProcessVo task, String account, int cursor, Boolean mainFlag) {
    try {
      this.exportSendProcessMsgBean.sendTaskProcessMsg(task, ExecStatusEnum.RUNNING.getDictCode(),
          ExportProcessEnum.RUNNING, cursor, account, mainFlag);
    } catch (NullPointerException e) {
    } finally {
      ThreadUtil.sleep(50); // 每次休息50毫秒，避免太近导致步进消息到客户端顺序出问题
    }
  }

  @Override
  public LinkedHashMap<String, String> merge(
      ExportTaskProcessVo task, ExportProcess process,
      String parentPath, List<ExportTaskDetailVo> list, Map<String, String> tmpFileMap) {
    Validate.isTrue(
        Objects.nonNull(task) && StringUtils.isNotBlank(task.getTaskCode()), "导出主任务编码不能为空");
    String taskCode = task.getTaskCode();
    Validate.isTrue(!CollectionUtils.isEmpty(list), "子任务不存在");
    for (ExportTaskDetailVo item : list) {
      final String s = tmpFileMap.get(item.getDetailCode());
      Validate.notBlank(s, CharSequenceUtil.format("{}子任务文件加载失败", item.getDetailCode()));
    }
    LinkedHashMap<String, String> map = Maps.newLinkedHashMap();
    if (list.size() == 1) {
      map.put(
          taskCode + ImportExportConstant.IE_FILE_SUFFIX,
          tmpFileMap.get(list.get(0).getDetailCode()));
      return map;
    }
    int maxSize = this.importExportProperties.getExportProperties().getExcelMaxCount();
    int pageSize = list.get(0).getPageSize();
    Validate.isTrue(maxSize > pageSize, "最大导出数必须大于处理器页数");
    int splitSize = maxSize / pageSize;
    int i = 0;
    int cursor = 0;
    int size = list.size();

    int startRow = 1;
    List<List<String>> excelHead;
    String businessCode = task.getBusinessCode();
    boolean isEuropaExport = ImportExportConstant.EXPORT_BIZ_CODE_MDM_EUROPA_CODE.equalsIgnoreCase(businessCode)
        || ImportExportConstant.EXPORT_BIZ_CODE_MDM_EUROPA_ASYNC_CODE.equalsIgnoreCase(businessCode);

    boolean isWebApiExport = ImportExportConstant.EXPORT_BIZ_CODE_EXPORT_WEB_API.equalsIgnoreCase(businessCode)
        || ImportExportConstant.EXPORT_BIZ_CODE_EXPORT_WEB_API_ASYNC.equalsIgnoreCase(businessCode);
    if (isEuropaExport) {
      String parentCode = task.getParentCode();
      String functionCode = task.getFunctionCode();
      FunctionPermissionVo funPermVo = new FunctionPermissionVo();
      funPermVo.setFunctionCode(functionCode);
      funPermVo.setParentCode(parentCode);
      funPermVo.setParamsJson(task.getParametersJson());
      List<ColumnVo> columnList = this.crmExportColumnStrategy.getShowDataviewColumn(funPermVo);
      excelHead = this.findHeadByEuropa(columnList);
    }
    else if(isWebApiExport){
      String parentCode = task.getParentCode();
      String functionCode = task.getFunctionCode();
      FunctionPermissionVo funPermVo = new FunctionPermissionVo();
      funPermVo.setFunctionCode(functionCode);
      funPermVo.setParentCode(parentCode);
      funPermVo.setParamsJson(task.getParametersJson());
      List<ColumnVo> columnList = this.crmExportColumnStrategy.getShowDataviewColumn(funPermVo);
      excelHead = this.findHeadByEuropa(columnList);
    }
    else {
      Class crmExcelVoClass = process.findCrmExcelVoClass();
      CrmExcelExport crmExcelExport = (CrmExcelExport) crmExcelVoClass.getAnnotation(CrmExcelExport.class);
      Validate.notNull(crmExcelExport, "导出业务未配置@CrmExcelExport注解");
      startRow = crmExcelExport.startRow();
      excelHead = this.findHead(process.findCrmExcelVoClass());
    }
    final List<List<ExportTaskDetailVo>> partition = Lists.partition(list, splitSize);
    for (List<ExportTaskDetailVo> item : partition) {
      ExcelWriter excelWriter = null;
      // 各个合并的主excel名称
      String fileName = CharSequenceUtil.format("主文件_{}_{}_{}{}", task.getTaskSource(), taskCode, i,
          ImportExportConstant.IE_FILE_SUFFIX);
      String fileTempName = process.getTaskFileName(task);
      if (StringUtils.isNotBlank(fileTempName)) {
        fileName = CharSequenceUtil.format("{}_{}{}", fileTempName, i, ImportExportConstant.IE_FILE_SUFFIX);
      }
      ExportTask exportTaskTemp = exportTaskRepository.findByTaskCode(taskCode);
      // 如果此处文件分割超出了2个主文件,那么就以循环中的fileName为准,如果只有1个主文件以导出任务中的文件名为准
      if(ObjectUtils.isNotEmpty(exportTaskTemp) && partition.size() <= 1){
        fileName = exportTaskTemp.getFileName();
      }

      try {
        excelWriter =
            EasyExcelFactory.write(parentPath + fileName)
                .head(excelHead)
                .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                .build();
        WriteSheet writeSheet = EasyExcelFactory.writerSheet(ImportExportConstant.SHEET1).build();
        for (ExportTaskDetailVo detail : item) {
          final Integer count = BzExcelUtil.findMsgCount(detail.getPageSize());
          final String curPath = tmpFileMap.get(detail.getDetailCode());
          Validate.isTrue(
              FileUtil.exist(curPath) && FileUtil.size(FileUtil.file(curPath)) > 0,
              CharSequenceUtil.format("未加载到子任务{}对应的文件", detail.getDetailCode()));
          EasyExcelFactory.read(
              curPath, new CrmExportExcelReadListener(count, excelWriter, writeSheet))
              .sheet()
              .headRowNumber(startRow)
              .doRead();
          cursor += 1;
          this.sendExportMergeProcessMsg(cursor, size, task.getTaskCode());
        }
      } catch (IllegalArgumentException | NullPointerException | IndexOutOfBoundsException e) {
        log.error("**** 自定义-导出出错了 ****", e);
        Validate.isTrue(false, e.getMessage());
      } catch (Exception e) {
        log.error("**** 自定义-导出出错了 ****", e);
        Validate.isTrue(false, "导出出错了");
      } finally {
        if (excelWriter != null) {
          excelWriter.close();
        }
        map.put(fileName, parentPath + fileName);
      }
      i++;
    }
    return map;
  }

  /**
   * 根据自定义处理器获取写入文件head信息
   *
   * @param processClass
   * @return
   */
  private List<List<String>> findHead(Class<?> processClass) {
    return Optional.of(BzExcelUtil.findCrmExcelExportTitle(processClass))
        .orElseThrow(() -> new IllegalArgumentException("未获取到业务操作实体的标题信息"));
  }

  /**
   * 跟进europa视图获取可显示的文件头信息
   *
   * @param showDataviewColumn
   * @return
   */
  private List<List<String>> findHeadByEuropa(List<ColumnVo> showDataviewColumn) {
    // step1:查询可显示文件头信息
    List<List<String>> headerList = Lists.newLinkedList();
    // step2:填充数据结构返回
    showDataviewColumn.forEach(colum -> {
      List<String> header = Lists.newArrayList();
      header.add(colum.getTitle());
      headerList.add(header);
    });
    return headerList;
  }

  /**
   * 获取文件的总记录数（包含head） 该方法统计的时候无法过滤掉空行
   * 
   * @param path
   * @param sheetNo
   * @return
   */
  private Integer findExcelTotalRow(String path, Integer sheetNo) {
    final ExcelReader excelReader =
        EasyExcelFactory.read(path).build().read(new ReadSheet(sheetNo));
    Integer total = excelReader.analysisContext().readRowHolder().getRowIndex() + 1;
    excelReader.close();
    return total;
  }

  /**
   * 获取文件的总记录数（包含head）忽略空行数据
   * 
   * @return
   */
  private int findExcelTotalRowIgnoreBlankLines(String path, Integer sheetNo, ImportProcess process) {
    int total;
    EasyExcelListener easyExcelListener = new EasyExcelListener(true, process.getLimitRowNum());
    ExcelReader excelReader = EasyExcelFactory.read(path, easyExcelListener).build()
        .read(new ReadSheet(sheetNo));
    List<Object> list = easyExcelListener.getList(Object.class);
    if (CollectionUtils.isEmpty(list) || list.size() == 0) {
      total = excelReader.analysisContext().readRowHolder().getRowIndex() + 1;
    } else {
      total = list.size() + 1;
    }
    excelReader.close();
    // 加上head
    return total;
  }

  /**
   * 获取导出任务数据
   *
   * @param dataList
   * @param process
   * @return
   */
  private List<List<Object>> findExportDataForEuropa(List<JSONObject> dataList, ExportProcess<?> process,
      ExportTaskProcessVo task,
      List<ColumnVo> columnList) {
    // 获取当前可显示字段
    List<List<Object>> list = Lists.newLinkedList();
    List<String> allowExportColumn = Lists.newLinkedList();
    // 列过滤数据权限
    Set<String> allowFields = null;
    if (StringUtils.isNotBlank(task.getMarsListCode())) {
      allowFields = this.marsAuthorityExcludedFieldDetailService.findByListCode(task.getMarsListCode());
    }
    // 列导出权限与列个人性化设置
    if (crmExportColumnStrategy != null) {
      // 获取当前可展示字段
      List<String> exportColumn = columnList.stream()
          // .filter(row -> row.isColumnExport())
          .map(ColumnVo::getField).collect(Collectors.toList());
      if (false == CollectionUtils.isEmpty(exportColumn)) {
        // 配置了个性化以及页面引擎导出的字段
        if (false == CollectionUtils.isEmpty(allowFields)) {
          for (String column : exportColumn) {
            if (allowFields.contains(column)) {
              allowExportColumn.add(column);
            }
          }
        } else {
          allowExportColumn = exportColumn;
        }
      }
    }
    if (allowExportColumn.isEmpty()) {
      log.info("数据视图禁止导出数据，返回空表格");
      return list;
    }
    for (JSONObject obj : dataList) {
      List<Object> cur = Lists.newLinkedList();
      for (String fieldName : allowExportColumn) {
        Object orDefault = obj.getOrDefault(fieldName, null);
        cur.add(orDefault);
      }
      list.add(cur);
    }
    return list;
  }

  /**
   * 获取导出任务数据
   *
   * @param data
   * @param process
   * @return
   */
  private List<List<Object>> findExportData(JSONArray data, ExportProcess<?> process, ExportTaskProcessVo task) {
    final List<CrmExcelColumnField> fields =
        BzExcelUtil.findCrmExcelColumnField(process.findCrmExcelVoClass());
    Validate.isTrue(!CollectionUtils.isEmpty(fields), "导出实体属性未配置@CrmExcelColumn注解");
    List<List<Object>> list = Lists.newLinkedList();
    // 列过滤数据权限
    Set<String> allowFields = null;
    if (StringUtils.isNotBlank(task.getMarsListCode())) {
      allowFields = this.marsAuthorityExcludedFieldDetailService.findByListCode(task.getMarsListCode());
    }
    // 列导出权限与列个人性化设置
    if (crmExportColumnStrategy != null) {
      Set<String> exportColumn = this.crmExportColumnStrategy.exportColumn(task);
      if (CollectionUtils.isEmpty(exportColumn) == false) {
        // 配置了个性化以及页面引擎导出的字段
        if (!CollectionUtils.isEmpty(allowFields)) {
          allowFields = Sets.intersection(allowFields, exportColumn);
        } else {
          allowFields = exportColumn;
        }
      }
    }

    for (Object o : data) {
      JSONObject obj = JSONUtil.parseObj(o);
      List<Object> cur = Lists.newLinkedList();
      for (CrmExcelColumnField field : fields) {
        String fieldName = field.getField().getName();
        Object filedValue = obj.getOrDefault(fieldName, null);
        if (allowFields == null) {
          cur.add(filedValue);
        } else {
          if (filedValue == null) {
            String fieldNameByLowerCamel = CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, fieldName);
            filedValue = obj.getOrDefault(fieldNameByLowerCamel, null);
          }
          cur.add(filedValue);
        }
      }
      list.add(cur);
    }
    return list;
  }

  /**
   * 导出任务合并文件进度消息通知
   *
   * @param cursor
   * @param size
   * @param taskCode
   */
  private void sendExportMergeProcessMsg(Integer cursor, Integer size, String taskCode) {
    Integer n = 40 + (cursor * 40 / size);
    log.info("主任务{},第1页,总计{},当前进度{},{}", taskCode, size, cursor, n);
    final ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(taskCode);
    msgVo.setMainFlag(true);
    msgVo.setIsCombine(true);
    msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    msgVo.setProcessType(ExportProcessEnum.COMBINE.getCode());
    msgVo.setRemark(CharSequenceUtil.format("任务{},总计{}个子任务,已合并{}个", taskCode, size, cursor));
    msgVo.setPageSize(100);
    msgVo.setAccount(loginUserService.getLoginAccountName());
    msgVo.setCursor(n);
    // 推送进度信息
    this.exportMsgBean.sendMsg(msgVo);
  }
}
