package com.biz.crm.common.ie.local.service.task;

import com.biz.crm.common.ie.local.concurrent.threadpool.priority.FutureTaskComparable;
import com.biz.crm.common.ie.local.concurrent.threadpool.priority.ThreadPoolExecutorPriorityBlocking;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.entity.ExportTaskDetail;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.model.dto.ExportTaskDetailModelDto;
import com.biz.crm.common.ie.local.model.dto.ExportTaskModelDto;
import com.biz.crm.common.ie.local.model.dto.ImportTaskModelDto;
import com.biz.crm.common.ie.local.service.ExportTaskDetailService;
import com.biz.crm.common.ie.local.service.ExportTaskService;
import com.biz.crm.common.ie.local.service.ImportTaskService;
import com.biz.crm.common.ie.local.service.init.ImportExportInitProcessEnvServ;
import com.biz.crm.common.ie.local.service.task.cache.ExportTaskCache;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.LoadStatusEnum;
import com.biz.crm.common.ie.sdk.service.ExportProcessService;
import com.biz.crm.common.ie.sdk.vo.IeExecutorVoForExport;
import com.biz.crm.common.ie.sdk.vo.IeExecutorVoForExportEsParagraph;
import com.biz.crm.common.ie.sdk.vo.IeExecutorVoForExportGetTotal;
import com.biz.crm.common.ie.sdk.vo.IeExecutorVoForImport;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.common.vo.AbstractTenantInfo;
import com.bizunited.nebula.common.vo.SimpleTenantInfo;
import com.bizunited.nebula.datasource.configuration.MultiDataSourceProperties;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import cn.hutool.core.thread.ExecutorBuilder;
import lombok.Getter;
import lombok.val;
import lombok.extern.slf4j.Slf4j;

/**
 * 加载导入导出任务
 *
 * @author sunx
 * @date 2022/5/11
 */
@Component("_ImportExportLoadTask")
@Slf4j
public class ImportExportLoadTask {
  /** 防止导入导出任务加载任务过程中，程序二次重复加载 */
  private static final Lock runLock = new ReentrantLock(true);
  /** 在执行中的导入任务数量 */
  @Getter
  private static LongAdder taskNumForImport = new LongAdder();
  /** 在执行中的导出任务数量 */
  @Getter
  private static LongAdder taskNumForExport = new LongAdder();
  @Autowired
  @Qualifier("_importExportThreadPoolExecutor")
  private ImportExportThreadPoolExecutor importExportThreadPoolExecutor;
  @Autowired(required = false)
  private ExportProcessService exportProcessService;
  @Autowired
  private ApplicationContext applicationContext;

  @Autowired
  private ImportTaskService importTaskService;

  @Autowired
  private ExportTaskDetailService exportTaskDetailService;

  @Autowired
  private ExportTaskService exportTaskService;

  @Autowired
  private ImportExportProperties importExportProperties;

  @Autowired
  private ImportExportInitProcessEnvServ importExportInitProcessEnvServ;

  @Autowired
  private MultiDataSourceProperties multiDataSourceProperties;
  @Autowired
  private ExportTaskCache exportTaskCache;

  private static ThreadPoolExecutor poolExe;

  /**
   * 根据应用+顶级租户信息加载任务信息
   *
   * @param applicationName
   * @param loadFlag true:已加载任务/ false:未加载任务
   */
  @Transactional
  public void load(String applicationName, Boolean loadFlag) {
    log.debug(loadFlag ? "读取：已加载任务" : "读取：未加载任务");
    boolean tryLock = runLock.tryLock(); // 防止导入导出任务加入过程中，程序二次进入
    if (tryLock == false) {
      return;
    }
    // 对多数据源的判断，在ExportTaskRepository和ImportTaskRepository类里面也有判断
    // 如果修改此处业务逻辑，需要注意另外两个类是否也有类似业务逻辑
    Boolean schemaForTenant = multiDataSourceProperties.getSchemaForTenant();
    try {
      if (schemaForTenant == false) {
        log.debug("加载导入导出[单数据源]");
        // 1、加载导入任务
        loadTask4Ie(applicationName, loadFlag);
      } else {
        List<String> appCodeList = importExportInitProcessEnvServ.getAppCodeList();
        log.debug("加载导入导出[多数据源]:{}", String.join(",", appCodeList));

        int appCodeSize = appCodeList.size();
        if (appCodeSize == 0) {
          return;
        }
        // 判断是否有null，判断创建新的线程池
        if (poolExe == null) {
          synchronized (ImportExportLoadTask.class) {
            if (poolExe == null) {
              poolExe = ExecutorBuilder.create()
                  .setCorePoolSize(appCodeSize)
                  .setMaxPoolSize(100)
                  .setWorkQueue(new ArrayBlockingQueue<>(100))
                  .build();
            }
          }
        }
        // 对多数据源的数据进行加载到CompletableFuture进行执行启动
        CompletableFuture<?>[] runCfArray = new CompletableFuture<?>[appCodeSize];        
        for (int idx = appCodeSize - 1; idx >= 0; idx--) {
          String appCode = appCodeList.get(idx);
          runCfArray[idx] = CompletableFuture.runAsync(() -> {
            log.debug("切换租户：{}", appCode);
            AbstractTenantInfo tenantInfo = new SimpleTenantInfo(appCode);
            TenantContextHolder.setTenantInfo(tenantInfo);
            // 1、加载导入任务
            loadTask4Ie(applicationName, loadFlag);
          }, poolExe);
        }
        CompletableFuture.allOf(runCfArray).join();
        log.debug("任务加载OK");
      }
    } finally {
      runLock.unlock();
    }
  }
  
  /**
   * 注意，该方法调用，必须在一个新的线程中执行
   * 
   * @param appCode
   * @param applicationName
   * @param loadFlag
   */
  private void loadTask4Ie(String applicationName, Boolean loadFlag) {
    final boolean haveAsyncGetTotalProcess = exportProcessService.isHaveAsyncGetTotalProcess();
    // 1、加载导入任务
    try {
      loadTask4Import(applicationName, loadFlag);
    } catch (Exception e) {
      log.error("ERROR! ERROR! ERROR! 加载导入任务异常", e);
    }
    // 2、加载导出任务
    try {
      loadTask4Export(applicationName, loadFlag);
    } catch (Exception e) {
      log.error("ERROR! ERROR! ERROR! 加载导出任务异常", e);
    }
    // 3、加载导出获取任务总数任务[根据当前系统是否有异步获取任务确定是否加载此线程池]
    if (haveAsyncGetTotalProcess == true) {
      try {
        loadTask4ExportAsyncGetTotal(applicationName, loadFlag);
      } catch (Exception e) {
        log.error("ERROR! ERROR! ERROR! 加载导出获取任务总数任务", e);
      }
    }

    try {
      loadTask4ExportEsParagraph(applicationName, loadFlag);
    } catch (Exception e) {
      log.error("ERROR! ERROR! ERROR! 加载导出获取任务总数任务", e);
    }
  }

  /**
   * 加载导出任务，将主任务拆分为子任务
   * 
   * @param applicationName
   * @param loadFlag
   */
  private void loadTask4ExportAsyncGetTotal(String applicationName, Boolean loadFlag) {
    log.debug("加载异步获取任务总数线程池");
    log.debug(loadFlag ? "读取：已加载【执行中、待执行】导出任务" : "读取：未加载导出任务");
    Map<String, List<ExportTask>> exportTaskMap = this.findExportTaskNoGetTotal(applicationName, loadFlag);
    if (exportTaskMap.isEmpty()) {
      return;
    }
    AbstractTenantInfo tenantInfo = TenantContextHolder.getTenantInfo();
    for (Entry<String, List<ExportTask>> item : exportTaskMap.entrySet()) {
      // step01: 获取一个租户的全部待执行任务List对象
      String appCode = item.getKey();
      List<ExportTask> exportTasklList = item.getValue();
      if (CollectionUtils.isEmpty(exportTasklList)) {
        continue;
      }
      int idx = 0;
      ThreadPoolExecutor threadPoolExecutor =this.importExportThreadPoolExecutor.findExportGetTotalThreadPoolExecutorByKey(appCode);
      for (ExportTask exportTask : exportTasklList) {
        final String taskCode = exportTask.getTaskCode();
        if (this.exportTaskCache.checkExecAsyncGetTotalTask(exportTask.getAppCode(), exportTask.getApplicationName(), exportTask.getTaskCode())) {
          log.info("导出任务{}已存在正在执行的异步获取导出总数线程,跳过执行", taskCode);
          continue;
        }
        // 将租户数据传入[刷新登陆使用]
        IeExecutorVoForExportGetTotal vo = new IeExecutorVoForExportGetTotal();
        vo.setFunctionCode(exportTask.getFunctionCode());
        vo.setParentCode(exportTask.getParentCode());
        vo.setTaskCode(taskCode);
        vo.setAccount(exportTask.getCreateAccount());
        vo.setAccountName(exportTask.getCreateName());
        vo.setTenantCode(exportTask.getTenantCode());
        vo.setAppCode(exportTask.getAppCode());
        vo.setApplicationName(exportTask.getApplicationName());
        vo.setLoadStatus(LoadStatusEnum.YES.getDictCode());
        
        IeTaskExecutorForExportGetTotal taskExe =
            this.applicationContext.getBean(IeTaskExecutorForExportGetTotal.class, vo, exportTask, tenantInfo);
        threadPoolExecutor.submit(taskExe);
        log.debug("{}任务进入异步拆子任务：{}", idx++, taskCode);
        // 加入任务到缓存中防止重复执行
        this.exportTaskCache.addExecAsyncGetTotalTask(exportTask.getAppCode(), exportTask.getApplicationName(), exportTask.getTaskCode());
      }
    }
  }

  /**
   * es导出任务分段子任务
   *
   * @param applicationName
   * @param loadFlag
   */
  private void loadTask4ExportEsParagraph(String applicationName, Boolean loadFlag) {
    log.debug("加载ES子任务分段线程池");
    log.debug(loadFlag ? "读取：已加载【执行中、待执行】导出任务" : "读取：未加载导出任务");
    Map<String, List<ExportTask>> exportTaskMap = this.findExportTaskNoEsParagraph(applicationName, loadFlag);
    if (exportTaskMap.isEmpty()) {
      return;
    }
    AbstractTenantInfo tenantInfo = TenantContextHolder.getTenantInfo();
    for (Entry<String, List<ExportTask>> item : exportTaskMap.entrySet()) {
      // step01: 获取一个租户的全部待执行任务List对象
      String appCode = item.getKey();
      List<ExportTask> exportTasklList = item.getValue();
      if (CollectionUtils.isEmpty(exportTasklList)) {
        continue;
      }
      int idx = 0;
      ThreadPoolExecutor threadPoolExecutor =this.importExportThreadPoolExecutor.findExportEsParagraphThreadPoolExecutorByKey(appCode);
      for (ExportTask exportTask : exportTasklList) {
        final String taskCode = exportTask.getTaskCode();
        if (this.exportTaskCache.checkExecEsParagraphTask(exportTask.getAppCode(), exportTask.getApplicationName(), exportTask.getTaskCode())) {
          log.info("导出任务{}已存在正在执行的异步分段子任务线程,跳过执行", taskCode);
          continue;
        }
        // 将租户数据传入[刷新登陆使用]
        IeExecutorVoForExportEsParagraph vo = new IeExecutorVoForExportEsParagraph();
        vo.setFunctionCode(exportTask.getFunctionCode());
        vo.setParentCode(exportTask.getParentCode());
        vo.setTaskCode(taskCode);
        vo.setAccount(exportTask.getCreateAccount());
        vo.setAccountName(exportTask.getCreateName());
        vo.setTenantCode(exportTask.getTenantCode());
        vo.setAppCode(exportTask.getAppCode());
        vo.setApplicationName(exportTask.getApplicationName());
        vo.setLoadStatus(LoadStatusEnum.YES.getDictCode());
        IeTaskExecutorForExportEsParagraph taskExe =
            this.applicationContext.getBean(IeTaskExecutorForExportEsParagraph.class, vo, exportTask, tenantInfo);
        threadPoolExecutor.submit(taskExe);
        log.debug("{}任务进入异步分段子任务：{}", idx++, taskCode);
        // 加入任务到缓存中防止重复执行
        this.exportTaskCache.addExecEsParagraphTask(exportTask.getAppCode(), exportTask.getApplicationName(), exportTask.getTaskCode());
      }
    }
  }
  /**
   * 加载导出任务
   * 
   * @param applicationName
   * @param loadFlag
   */
  private void loadTask4Export(String applicationName, Boolean loadFlag) {
    log.debug(loadFlag ? "读取：已加载【执行中、待执行】导出任务" : "读取：未加载导出任务");
    long threadId = Thread.currentThread().getId();
    Map<String, List<ExportTaskDetail>> exportTaskCodeMap = this.findExportTaskCode(applicationName, loadFlag);
    if (exportTaskCodeMap == null || exportTaskCodeMap.isEmpty()) {
      // 没有查到数据就返回
      return;
    }
    AbstractTenantInfo tenantInfo = TenantContextHolder.getTenantInfo();
    Map<String, ExportTask> exportTaskMap = new HashMap<>();
    for (Entry<String, List<ExportTaskDetail>> item : exportTaskCodeMap.entrySet()) {
      // step01: 获取一个租户的全部待执行任务List对象
      String appCode = item.getKey();
      List<ExportTaskDetail> exportTaskDetailList = item.getValue();
      if (CollectionUtils.isEmpty(exportTaskDetailList)) {
        continue;
      }
      // ThreadPoolExecutor exportThreadPoolExecutor =
      // this.importExportThreadPoolExecutor.findExportThreadPoolExecutorByKey(appCode);
      ThreadPoolExecutorPriorityBlocking exportThreadPoolExecutor =
          this.importExportThreadPoolExecutor.findExportThreadPoolExecutorByKeyPriorityBlocking(appCode);
      
      // CompletionService<IeExecutorVoForExport> completionService = new
      // ExecutorCompletionService<>(exportThreadPoolExecutor);
      int idx = 0;
      Vector<FutureTaskComparable<?>> futureTaskList = new Vector<>();
      Set<String> detailCodeSetByUpdate = Sets.newHashSet();
      Set<String> taskCodeSetBySkip = Sets.newHashSet();
      for (ExportTaskDetail task : exportTaskDetailList) {
        // 对一个租户的任务清单进行拆解，异步执行
        String taskCode = task.getTaskCode();
        if (taskCodeSetBySkip.contains(taskCode)) { // 该记录有问题，需要跳过----》0001
          continue;
        }
        String detailCode = task.getDetailCode();
        log.debug("应用执行导入taskCode:{},当前线程号:{}", taskCode, threadId);
        // 如果是需要跳过的，本次循环跳过该任务信息
        ExportTask exportTask = exportTaskMap.get(taskCode);// 缓存本次查询，如果有子任务可以-----------0002
        if (exportTask == null) {
          exportTask = this.exportTaskService.findExportTask(taskCode);
          if (exportTask == null) {
            log.info("应用执行导入taskCode:{},数据库无法查询到主表数据{}，本次循环跳过", taskCode, detailCode);
            taskCodeSetBySkip.add(taskCode); // 该记录有问题，后续跳过------------》0001
            continue;
          }
          exportTaskMap.put(taskCode, exportTask); // 缓存本次查询，如果有子任务可以--------------------0002
        }
        String functionCode = exportTask.getFunctionCode();
        String parentCode = exportTask.getParentCode();

        IeExecutorVoForExport vo = new IeExecutorVoForExport();
        vo.setFunctionCode(functionCode);
        vo.setParentCode(parentCode);
        vo.setExportTaskDetailCode(detailCode);
        // vo.setIeType(TypeEnum.EXPORT.getDictCode());
        vo.setTaskCode(taskCode);
        vo.setAccount(task.getCreateAccount());
        vo.setAccountName(task.getCreateName());
        vo.setTenantCode(task.getTenantCode());
        vo.setAppCode(task.getAppCode());
        vo.setApplicationName(task.getApplicationName());
        vo.setTotal(exportTask.getTotal());
        vo.setTaskSource(exportTask.getTaskSource());
        vo.setLoadStatus(LoadStatusEnum.YES.getDictCode());
        
        // 将租户数据传入
        IeTaskExecutorForExport taskExe =
            this.applicationContext.getBean(IeTaskExecutorForExport.class, vo, tenantInfo);
        // exportThreadPoolExecutor.execute(taskExe);
        FutureTaskComparable<?> exportFuture = exportThreadPoolExecutor.submit(taskExe);
        futureTaskList.add(exportFuture);
        log.debug("{}任务加入队列：{}", idx++, taskCode);
        taskNumForExport.increment();
        // 加入状态修改任务
        detailCodeSetByUpdate.add(detailCode); // 记录后续批量修改状态的set对象==》step02
      }

      // step02:对当前租户的任务清单状态修改为运行中
      {
        this.exportTaskDetailService.updateLocalStatus(detailCodeSetByUpdate, LoadStatusEnum.YES.getDictCode());
      }

      if (false) {
        // 暂时屏蔽，异步流，哪个任务结束，哪个任务线返回，但是会阻塞到所有任务执行完才能结束
        futureTaskList.parallelStream().forEach(exportFuture -> {
          Object x;
          try {
            x = exportFuture.get();
            if (x == null) {
              IeTaskExecutorForExport comparable = (IeTaskExecutorForExport) exportFuture.getComparable();
              log.info("{}",comparable);
            }
            // log.info("完成：主任务{}-子任务{}", vo.getTaskCode(), vo.getExportTaskDetailCode());
            log.info("{}",x);
          } catch (InterruptedException | ExecutionException e) {
            log.error(e.getMessage(),e);
          } finally {
            taskNumForExport.decrement();
          }
        });
      }
    }
  }

  /**
   * 加载导入任务
   * 
   * @param applicationName
   * @param loadFlag
   */
  private void loadTask4Import(String applicationName, Boolean loadFlag) {
    // k-顶级租户编码，v-租户对应的任务编码
    Map<String, List<ImportTask>> importTaskCodeMap = this.findImportTaskCode(applicationName, loadFlag);
    if (importTaskCodeMap == null || importTaskCodeMap.isEmpty()) {
      // 没有查到数据就返回
      return;
    }

    Set<ImportTask> set = Sets.newHashSet();
    Vector<Future<?>> futureTaskList = new Vector();
    AbstractTenantInfo tenantInfo = TenantContextHolder.getTenantInfo();
    for (Entry<String, List<ImportTask>> item : importTaskCodeMap.entrySet()) {
      set.addAll(item.getValue());
      ThreadPoolExecutor importThreadPoolExecutor =
          this.importExportThreadPoolExecutor.findImportThreadPoolExecutorByKey(item.getKey());
      for (ImportTask task : item.getValue()) {
        IeExecutorVoForImport vo = new IeExecutorVoForImport();
        vo.setTaskCode(task.getTaskCode());
        vo.setAccount(task.getCreateAccount());
        vo.setAccountName(task.getCreateName());
        vo.setTenantCode(task.getTenantCode());
        vo.setAppCode(task.getAppCode());
        vo.setApplicationName(task.getApplicationName());
        vo.setLoadStatus(LoadStatusEnum.YES.getDictCode());
        IeTaskExecutorForImport taskExe = this.applicationContext.getBean(IeTaskExecutorForImport.class, vo, tenantInfo);
        Future<?> importFuture = importThreadPoolExecutor.submit(taskExe);
        futureTaskList.add(importFuture);
        taskNumForImport.increment();
      }
    }
    // step02:对当前租户的任务清单状态修改为运行中
    if (CollectionUtils.isNotEmpty(set)) {
      Set<String> taskCodes = set.parallelStream().map(ImportTask::getTaskCode).collect(Collectors.toSet());
      this.importTaskService.updateLocalStatus(taskCodes, LoadStatusEnum.YES.getDictCode());
    }

    if (false) {
      // 暂时屏蔽，异步流，哪个任务结束，哪个任务线返回，但是会阻塞到所有任务执行完才能结束
      futureTaskList.parallelStream().forEach(exportFuture -> {
        Object x;
        try {
          x = exportFuture.get();
          // log.info("完成：主任务{}-子任务{}", vo.getTaskCode(), vo.getExportTaskDetailCode());
          log.info("{}",x);
        } catch (InterruptedException | ExecutionException e) {
          log.error(e.getMessage(),e);
        } finally {
          taskNumForImport.decrement();
        }
      });
    }
  }

  /**
   * 获取导入需要加载的导入任务编码集合
   *
   * @param applicationName
   * @param loadFlag
   * @return k-顶级租户编码，v-租户对应的任务编码
   */
  private Map<String, List<ImportTask>> findImportTaskCode(String applicationName, Boolean loadFlag) {
    ImportTaskModelDto importTaskModelDto = new ImportTaskModelDto();
    importTaskModelDto.setApplicationName(applicationName);
    if (Boolean.TRUE.equals(loadFlag)) {
      importTaskModelDto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.RUNNING.getDictCode(), ExecStatusEnum.DEFAULT.getDictCode()));
    } else {
      importTaskModelDto.setLoadStatus(LoadStatusEnum.NO.getDictCode());
    }
    List<ImportTask> result = this.importTaskService.findByImportTaskModelDto(importTaskModelDto);
    if (CollectionUtils.isEmpty(result)) {
      return Maps.newHashMap();
    }
    List<ImportTask> list= new ArrayList<>();
    //等待时间
    long importSleep = importExportProperties.getImportSleep();
    //项目在使用集群数据库时可能需要等待数据库同步数据
    if (importSleep > 0) {
      for (ImportTask importTask : result) {
        Date createTime = importTask.getCreateTime();
        Date now = new Date();
        if ((now.getTime() - createTime.getTime()) >= importSleep) {
          list.add(importTask);
        }
      }
      log.warn("导入等待时间{}",importSleep/1000);
      log.warn("加载导入因等待时间过滤{}条数据",result.size()-list.size());
    }else {
      list=result;
    }
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, List<ImportTask>> map = Maps.newHashMap();
    if (Boolean.TRUE.equals(this.importExportProperties.getAppFlag())) {
      map.put(applicationName, list);
    } else {
      final Map<String, List<ImportTask>> cur = list.parallelStream()
          .filter(a -> StringUtils.isNoneBlank(a.getAppCode(), a.getTaskCode()))
          .collect(Collectors.groupingBy(ImportTask::getAppCode));
      for (Entry<String, List<ImportTask>> item : cur.entrySet()) {
        map.put(applicationName + ":" + item.getKey(), item.getValue());
      }
    }
    return map;
  }
  
  /**
   * 获取导出需要加载的导出任务子任务编码集合
   *
   * @param applicationName
   * @param loadFlag true：执行中、待执行；false：加载状态=未加载
   * @return k-顶级租户编码，v-租户对应的任务编码
   */
  private Map<String, List<ExportTaskDetail>> findExportTaskCode(String applicationName, Boolean loadFlag) {
    ExportTaskDetailModelDto exportTaskDetailModelDto = new ExportTaskDetailModelDto();
    exportTaskDetailModelDto.setApplicationName(applicationName);
    if (Boolean.TRUE.equals(loadFlag)) {
      exportTaskDetailModelDto.setLoadStatus(LoadStatusEnum.YES.getDictCode());
      exportTaskDetailModelDto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.RUNNING.getDictCode(), ExecStatusEnum.DEFAULT.getDictCode()));
    } else {
      exportTaskDetailModelDto.setLoadStatus(LoadStatusEnum.NO.getDictCode());
      exportTaskDetailModelDto.setExecStatusSet(Sets.newHashSet(ExecStatusEnum.DEFAULT.getDictCode()));
    }
    List<ExportTaskDetail> list =
        this.exportTaskDetailService.findByExportTaskDetailModelDto(exportTaskDetailModelDto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, List<ExportTaskDetail>> map = Maps.newHashMap();
    if (Boolean.TRUE.equals(this.importExportProperties.getAppFlag())) {
      map.put(applicationName, list);
    } else {
      final Map<String, List<ExportTaskDetail>> cur = list.parallelStream()
          .filter(a -> StringUtils.isNoneBlank(a.getAppCode(), a.getDetailCode()))
          .collect(Collectors.groupingBy(ExportTaskDetail::getAppCode));
      for (Entry<String, List<ExportTaskDetail>> item : cur.entrySet()) {
        map.put(applicationName + ":" + item.getKey(), item.getValue());
      }
    }
    return map;
  }

  /**
   * 获取导出需要加载的导出任务子任务编码集合
   *
   * @param applicationName
   * @param loadFlag true：执行中、待执行；false：加载状态=未加载
   * @return k-顶级租户编码，v-租户对应的任务编码
   */
  private Map<String, List<ExportTask>> findExportTaskNoGetTotal(String applicationName, Boolean loadFlag) {
    ExportTaskModelDto dto = new ExportTaskModelDto();
    dto.setApplicationName(applicationName);
    if (Boolean.TRUE.equals(loadFlag)) {
      dto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.NEED_EXPORT_TASK_DETAIL.getDictCode(), ExecStatusEnum.DEFAULT.getDictCode()));
    } else {
      dto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.NEED_EXPORT_TASK_DETAIL.getDictCode()));
    }
    List<ExportTask> list = this.exportTaskService.findExportTaskNoGetTotal(dto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, List<ExportTask>> map = Maps.newHashMap();
    if (Boolean.TRUE.equals(this.importExportProperties.getAppFlag())) {
      map.put(applicationName, list);
    } else {
      final Map<String, List<ExportTask>> cur = list.parallelStream()
          .filter(a -> a.getTotal()==null)
          .collect(Collectors.groupingBy(ExportTask::getAppCode));
      for (Entry<String, List<ExportTask>> item : cur.entrySet()) {
        map.put(applicationName + ":" + item.getKey(), item.getValue());
      }
      // 将已经加载的任务更改标记
//      this.exportTaskService.updateExecStatus(applicationName, ExecStatusEnum.DEFAULT, cur.keySet().toArray(new String[0]));
    }
    return map;
  }

  /**
   * 获取需要ES数据分段的导出任务
   *
   * @param applicationName
   * @param loadFlag true：执行中、待执行；false：加载状态=未加载
   * @return k-顶级租户编码，v-租户对应的任务编码
   */
  private Map<String, List<ExportTask>> findExportTaskNoEsParagraph(String applicationName, Boolean loadFlag) {
    ExportTaskModelDto dto = new ExportTaskModelDto();
    dto.setApplicationName(applicationName);
    if (Boolean.TRUE.equals(loadFlag)) {
      dto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.NEED_PARAGRAPH_EXPORT_TASK_DETAIL.getDictCode(), ExecStatusEnum.DEFAULT.getDictCode()));
    } else {
      dto.setExecStatusSet(
          Sets.newHashSet(ExecStatusEnum.NEED_PARAGRAPH_EXPORT_TASK_DETAIL.getDictCode()));
    }
    List<ExportTask> list = this.exportTaskService.findExportTaskNoEsParagraph(dto);
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, List<ExportTask>> map = Maps.newHashMap();
    if (Boolean.TRUE.equals(this.importExportProperties.getAppFlag())) {
      map.put(applicationName, list);
    } else {
      final Map<String, List<ExportTask>> cur = list.parallelStream()
          .filter(a -> a.getTotal()==null)
          .collect(Collectors.groupingBy(ExportTask::getAppCode));
      for (Entry<String, List<ExportTask>> item : cur.entrySet()) {
        map.put(applicationName + ":" + item.getKey(), item.getValue());
      }
      // 将已经加载的任务更改标记
//      this.exportTaskService.updateExecStatus(applicationName, ExecStatusEnum.DEFAULT, cur.keySet().toArray(new String[0]));
    }
    return map;
  }
}
