package com.biz.crm.common.ie.local.service.task;

import cn.hutool.core.net.NetUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ThreadUtil;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.util.IpUtil;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * 导入导出任务master独占权保持任务，每一个任务都负责都过applicationName_appCode的独占权保持
 *
 * @author sunx
 * @date 2022/5/11
 */
@Component("_ImportExportMasterKeeperTask")
@Slf4j
public class ImportExportMasterKeeperTask implements Runnable {

    @Autowired
    private RedisMutexService redisMutexService;

    @Autowired
    private ImportExportLoadTask importExportLoadTask;

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private RedisTemplate redisTemplate;

    @Autowired
    private Environment environment;

    @Autowired
    private ImportExportProperties importExportProperties;

    @Override
    public void run() {
        String appName = this.importExportProperties.getIeTaskGroup();
        String ieLockKey = CharSequenceUtil.format(ImportExportConstant.IE_LOAD_LOCK_PREFIX_FORMAT, appName);
        while (true) {
            if (this.redisMutexService.tryLock(ieLockKey, TimeUnit.SECONDS, 10)) {
                log.info("应用{},已获取任务操作权限，可以开始执行任务", appName);
                ValueOperations<String, String> opsForValue = this.redisTemplate.opsForValue();
                String hostKey = CharSequenceUtil.format(ImportExportConstant.IE_MAIN_HOST_PREFIX_FORMAT, appName);
                opsForValue.set(hostKey, this.findServer());
                String lockKey = CharSequenceUtil.format(ImportExportConstant.IE_LOAD_LOCK_SERVER_PREFIX_FORMAT, appName);
                opsForValue.set(lockKey, this.findServerInfo());
                // 加载执行中和待执行的任务
                this.importExportLoadTask.load(appName, true);
                // 新开线程加载未加载的任务
                ImportExportReloadTask task = applicationContext.getBean(ImportExportReloadTask.class);
                ThreadUtil.execAsync(task, true);
                break;
            }
        }
    }

    /**
     * 获取主机信息
     *
     * @return
     */
    private String findServerInfo() {
        String hostName = "";
        try {
            InetAddress addr = InetAddress.getLocalHost();
            hostName = addr.getHostName();
        } catch (UnknownHostException e1) {
            log.error("Error while getting host name", e1);
        }
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        String runtimeMxName = runtimeMXBean.getName();
        String osName = System.getProperty("os.name");
        String ip = IpUtil.getLocalIp();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String startTime = sdf.format(new Date(runtimeMXBean.getStartTime()));
        String serverInfo =
                String.format("IP:%s,主机名:%s,OS:%s,应用:%s,启动时间:%s", ip, hostName, osName, runtimeMxName, startTime);
        return serverInfo;
    }

    /**
     * 获取主节点服务ip+端口信息
     *
     * @return
     */
    private String findServer() {
        String ip = IpUtil.getPublicIp();
        if (StringUtils.isBlank(ip)) {
            log.error("未获取到本机的公网ip");
            final LinkedHashSet<String> strings = NetUtil.localIpv4s();
            final Optional<String> first = strings.stream().findFirst();
            if (first.isPresent()) {
                ip = first.get();
            }
        }
        String server = CharSequenceUtil.format("{}:{}{}", ip, this.environment.getProperty("server.port"),
                this.environment.getProperty("server.servlet.context-path"));
        log.info(server);
        return server;
    }
}
