package com.biz.crm.common.ie.local.service.task;

import com.biz.crm.common.ie.local.concurrent.threadpool.priority.ThreadPoolExecutorPriorityBlocking;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.google.common.collect.Maps;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ExecutorBuilder;
import cn.hutool.core.thread.RejectPolicy;
import cn.hutool.core.thread.ThreadFactoryBuilder;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 导入导出线程池
 *
 * @author sunx
 * @date 2022/5/10
 */
@Slf4j
public class ImportExportThreadPoolExecutor {

  /** 导入并行任务数量 */
  @Getter
  @Setter
  private Integer importTaskCount;

  /** 导出并行任务数量 */
  @Getter
  @Setter
  private Integer exportTaskCount;
  
  /** 导出获取总数任务数量 */
  @Getter
  @Setter
  private Integer exportGetTotalTaskCount;

  /** 导出ES分段任务数量 */
  @Getter
  @Setter
  private Integer exportEsParagraphTaskCount;

  /** 导入任务线程池 */
  private static final Map<String, ThreadPoolExecutor> IMPORT_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();

  /** 导出任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /** 导出任务线程池 */
  private static final Map<String, ThreadPoolExecutorPriorityBlocking> EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING =
      Maps.newHashMap();
  /** 导出任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /** 导出ES分段任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /**
   * 根据应用-顶级租户编码获取导入任务线程池
   *
   * @param key
   * @return
   */
  public ThreadPoolExecutor findImportThreadPoolExecutorByKey(String key) {
    this.createImportPoolByKey(key);
    return IMPORT_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  public Map<String, ThreadPoolExecutor> getImportThreadPoolExecutorMap() {
    return IMPORT_THREAD_POOL_EXECUTOR_MAP;
  }

  /**
   * 根据应用-顶级租户编码获取导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutor findExportThreadPoolExecutorByKey(String key) {
    this.createExportPoolByKey(key);
    return EXPORT_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  /**
   * 根据应用-顶级租户编码获取导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutorPriorityBlocking findExportThreadPoolExecutorByKeyPriorityBlocking(String key) {
    this.createExportPoolByKeyPriorityBlocking(key);
    return EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.get(key);
  }

  public Map<String, ThreadPoolExecutor> getExportThreadPoolExecutorMap() {
    return EXPORT_THREAD_POOL_EXECUTOR_MAP;
  }

  /**
   * 根据应用-租户创建对应的导入任务线程池
   *
   * @param key
   */
  private void createImportPoolByKey(String key) {
    if (IMPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (IMPORT_THREAD_POOL_EXECUTOR_MAP) {
      if (IMPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      ThreadPoolExecutor threadPoolExecutor = ExecutorBuilder.create()
          .setCorePoolSize(this.getImportTaskCount())
          .setMaxPoolSize(this.getImportTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(Integer.MAX_VALUE))
          .setThreadFactory(
              ThreadFactoryBuilder.create()
                  .setNamePrefix(
                      CharSequenceUtil.format(
                          ImportExportConstant.IMPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key))
                  .build())
          .build();
      IMPORT_THREAD_POOL_EXECUTOR_MAP.put(key, threadPoolExecutor);
      log.info("---应用租户信息为{}导入线程池构建成功---", key);
    }
  }

  /**
   * 根据应用-租户创建对应的导出任务线程池
   *
   * @param key
   */
  private void createExportPoolByKey(String key) {
    if (EXPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportTaskCount())
          .setMaxPoolSize(this.getExportTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportTaskCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出线程池构建成功---", key);
    }
  }

  private void createExportPoolByKeyPriorityBlocking(String key) {
    if (EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING) {
      if (EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutorPriorityBlocking poolExe =
          new ThreadPoolExecutorPriorityBlocking(exportTaskCount, exportTaskCount, 1L, TimeUnit.SECONDS,
              new PriorityBlockingQueue<Runnable>(exportTaskCount), threadFactory, RejectPolicy.BLOCK.getValue());
      EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.put(key, poolExe);
      log.info("---应用租户信息为{}导出线程池构建成功---", key);
    }
  }

  public ThreadPoolExecutor findExportGetTotalThreadPoolExecutorByKey(String key) {
    this.createExportGetTotalThreadPoolExecutorByKey(key);
    return EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.get(key);
  }
  
  private void createExportGetTotalThreadPoolExecutorByKey(String key) {
    if (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_GET_TOTAL_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportGetTotalTaskCount())
          .setMaxPoolSize(this.getExportGetTotalTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportGetTotalTaskCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出创建异步获取总数任务线程池构建成功---", key);
    }
  }

  public ThreadPoolExecutor findExportEsParagraphThreadPoolExecutorByKey(String key) {
    this.createExportEsParagraphThreadPoolExecutorByKey(key);
    return EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  private void createExportEsParagraphThreadPoolExecutorByKey(String key) {
    if (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_ES_PARAGRAPH_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportEsParagraphTaskCount())
          .setMaxPoolSize(this.getExportEsParagraphTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportEsParagraphTaskCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出创建ES分段任务线程池构建成功---", key);
    }
  }
}
