package com.biz.crm.common.ie.local.util;

import lombok.extern.slf4j.Slf4j;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author sunx
 * @date 2022/5/24
 */
@Slf4j
public class IpUtil {

  private IpUtil() {}

  /** 外网ip地址 */
  private static String publicIp;

  /** 下面url返回地址都包含ip地址，为防止某个url失效， 遍历url获取ip地址，有一个能成功获取就返回 */
  private static String[] urls = {
    "http://whatismyip.akamai.com",
    "http://icanhazip.com",
    "http://members.3322.org/dyndns/getip",
    "http://checkip.dyndns.com/",
    "http://pv.sohu.com/cityjson",
    "http://ip.taobao.com/service/getIpInfo.php?ip=myip",
    "http://www.ip168.com/json.do?view=myipaddress",
    "http://www.net.cn/static/customercare/yourip.asp",
    "http://ipecho.net/plain",
    "http://myip.dnsomatic.com",
    "http://tnx.nl/ip",
    "http://ifconfig.me"
  };

  /** ip地址的匹配正则表达式 */
  private static String regEx =
      "((2[0-4]\\d|25[0-5]|[01]?\\d\\d?)\\.){3}(2[0-4]\\d|25[0-5]|[01]?\\d\\d?)";

  private static Pattern pattern = Pattern.compile(regEx);

  /**
   * 获取本机外网地址
   *
   * @return
   */
  public static String getPublicIp() {
    if (publicIp != null && !"".equals(publicIp.trim())) {
      return publicIp;
    }
    for (String url : urls) {
      // http访问url获取带ip的信息
      String result = getUrlResult(url);
      // 正则匹配查找ip地址
      Matcher m = pattern.matcher(result);
      while (m.find()) {
        publicIp = m.group();
        // 只获取匹配到的第一个IP地址
        return publicIp;
      }
    }
    return "";
  }

  /**
   * http访问url
   *
   * @param url 获取外网ip的url地址
   * @return
   */
  private static String getUrlResult(String url) {
    StringBuilder sb = new StringBuilder();
    BufferedReader in = null;
    try {
      URL realUrl = new URL(url);
      URLConnection connection = realUrl.openConnection();
      connection.setConnectTimeout(1000);
      connection.setReadTimeout(1000);
      in = new BufferedReader(new InputStreamReader(connection.getInputStream()));
      String line;
      while ((line = in.readLine()) != null) {
        sb.append(line);
      }
    } catch (Exception e) {
      return "";
    }
    return sb.toString();
  }

  /**
   * 获取本地IP地址
   *
   * @return IP地址
   */
  public static String getLocalIp() {
    if (true == isWindowsOS()) {
      return getLocalIpForWindows();
    } else {
      return getLocalIpForLinux();
    }
  }
  /**
   * 获取Linux下的IP地址
   *
   * @return IP地址
   * @throws SocketException
   */
  public static String getLocalIpForLinux() {
    String ip = "";
    try {
      for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements();) {
        NetworkInterface intf = en.nextElement();
        String name = intf.getName();
        if (!name.contains("docker") && !name.contains("lo")) {
          for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements();) {
            InetAddress inetAddress = enumIpAddr.nextElement();
            if (!inetAddress.isLoopbackAddress()) {
              String ipaddress = inetAddress.getHostAddress().toString();
              if (!ipaddress.contains("::") && !ipaddress.contains("0:0:")
                  && !ipaddress.contains("fe80")) {
                ip = ipaddress;
              }
            }
          }
        }
      }
    } catch (SocketException ex) {
      ip = "127.0.0.1";
      log.error(ex.getMessage(),ex);
    }
    return ip;
  }
  
  /**
   * 获取Windows下的IP地址
   *
   * @return IP地址
   */
  public static String getLocalIpForWindows() {
    String ip = "";
    try {
      InetAddress addr = InetAddress.getLocalHost();
      ip = addr.getHostAddress();
    } catch (UnknownHostException ex) {
      ip = "127.0.0.1";
      log.error(ex.getMessage(),ex);
    }
    return ip;
  }
  /**
   * 判断操作系统是否是Windows
   *
   * @return
   */
  public static boolean isWindowsOS() {
    boolean isWindowsOS = false;
    // 注：这里的system，系统指的是 JRE (runtime)system，不是指 OS
    String osName = System.getProperty("os.name");
    if (osName.toLowerCase().indexOf("windows") > -1) {
      isWindowsOS = true;
    }
    return isWindowsOS;
  }
}
