package com.biz.crm.common.ie.local.config;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.convert.DataSizeUnit;
import org.springframework.util.unit.DataUnit;
import cn.hutool.core.io.unit.DataSize;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

/**
 * 导入导出配置信息
 *
 * @author sunx
 * @date 2022/5/10
 */
@Setter
@ToString
@ConfigurationProperties(prefix = "crm.common.ie")
public class ImportExportProperties {
  /**
   * 应用名称，留作兼容已存在老项目使用<br/>
   * <b>CRM4.0版本中，该属性将会废弃</b>
   * 
   * @deprecated 推荐使用applicationName属性，如果已经配置appName，则忽略applicationName的配置信息
   */
  private String appName = "";

  // ##################################################
  /** 是否执行导入导出任务 */
  @Getter
  private Boolean execTask = Boolean.TRUE;
  /** 导入导出临时文件存放根路径 */
  @Getter
  private String root = "";
  /** 任务组名称 */
  @Getter
  private String taskGroup = "";

  /** 任务扫描间隔,单位毫秒 */
  @Getter
  private long taskInterval = 5000;

  /** 导入任务等候时间，单位毫秒 **/
  @Getter
  private long importSleep = 0;
  
  /** 任务执行后休眠时长,默认500毫秒 */
  @Getter
  private long taskExecutionIntervalSleep = 500;

  /** spring boot应用程序名 */
  @Getter
  @Value("${spring.application.name}")
  private String applicationName;
  
  /** 当前JVM可以使用的处理器内核数 */
  public static final int NCPU = Runtime.getRuntime().availableProcessors();
  
  /** 获取任务名称 */
  public String getIeTaskGroup() {
    if (StringUtils.isNotBlank(appName)) {
      return appName;
    } else {
      StringBuilder sb = new StringBuilder();

      sb.append(applicationName).append(":ie");
      if (StringUtils.isNotBlank(taskGroup)) {
        if (taskGroup.startsWith("ie-")) {
          taskGroup = taskGroup.substring("ie-".length());
        }
        sb.append("-").append(taskGroup);
      }
      return sb.toString();
    }
  }

  /** 导入导出任务线程池隔离级别，true-应用级，false-顶级租户级 */
  @Getter
  private Boolean appFlag = true;

  /** 导入配置参数 */
  @Getter
  private ImportProperties importProperties = new ImportProperties();

  /** 导处入配置参数 */
  @Getter
  private ExportProperties exportProperties = new ExportProperties();
  
  /** 导处异步获取数据参数 */
  @Getter
  private ExportAsyncGetTotalProperties exportAsyncGetDataProperties = new ExportAsyncGetTotalProperties();

  /** 导出ES分段配置参数 */
  @Getter
  private ExportEsParagraphProperties exportEsParagraphProperties = new ExportEsParagraphProperties();

  /** 导出配置参数 */
  @Data
  public static class ImportProperties {
    /** 导入文件最大限制 xx M */
    @DataSizeUnit(DataUnit.MEGABYTES)
    private DataSize excelMaxSize = DataSize.ofMegabytes(40);
    /** 导入并行任务数量，默认5个，请根据服务器CPU内核内存数进行调整，避免执行过程中频繁FullGC和内核切换 */
    private int taskCount = 5;
    /** 大任务的记录行数,默认20000 */
    private int bigTaskRowNum = 20000;
    /** 导入线程池等待队列大小,默认200 */
    private int workQueueCount = 200;
  }
  /** 导出配置参数 */
  @Data
  public static class ExportProperties {
    /** 同一个文件导出数量最大限制 xx 条 */
    private int excelMaxCount = 500000;
    /** 导入并行任务数量，默认5个，请根据服务器CPU内核内存数进行调整，避免执行过程中频繁FullGC和内核切换 */
    @Deprecated
    private int taskCount = 5;
    /** 大任务的记录行数,默认20000 */
    @Deprecated
    private int bigTaskRowNum = 20000;
    /** 数据查询分页大小,默认20000 */
    private int queryPageSize = 20000;
    /** 获取合并文件最大条数(超过该数值将不会合并文件,如果是小于等于0则不会限制无论多少条都会合并) **/
    private int mergeFileMaxCount = -1;
    /** 大数据量导出任务并行数量,默认1个 */
    private int largeTaskCount = 1;
    /** 大数据量导出线程池等待队列大小,默认100 */
    private int largeWorkQueueCount = 100;
    /** 大数据量任务达标数据量,默认100000 */
    private int largeTaskDataNum = 100000;
    /** 中数据量导出任务并行数量,默认1个 */
    private int mediumTaskCount = 2;
    /** 中数据量导出线程池等待队列大小,默认200 */
    private int mediumWorkQueueCount = 200;
    /** 中数据量任务达标数据量,默认10000 */
    private int mediumTaskDataNum = 10000;
    /** 小数据量导出任务并行数量,默认1个 */
    private int smallTaskCount = 3;
    /** 小数据量导出线程池等待队列大小,默认300 */
    private int smallWorkQueueCount = 300;
    /** 导出存在图片时最大可导出行数限制,默认1000 */
    private int maxImageRowNum = 1000;
  }
  
  /** 导出异步获取数据配置 */
  @Data
  public static class ExportAsyncGetTotalProperties {
    private int taskCount = 3;
    /** 导出异步获取数据线程池等待队列大小,默认200 */
    private int workQueueCount = 200;
  }

  /** ES导出线程池配置 */
  @Data
  public static class ExportEsParagraphProperties {
    private int taskCount = 3;
    /** ES导出线程池等待队列大小,默认200 */
    private int workQueueCount = 200;
  }
}
