package com.biz.crm.common.ie.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.conditions.update.LambdaUpdateChainWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.entity.ExportTaskDetail;
import com.biz.crm.common.ie.local.mapper.ExportTaskDetailMapper;
import com.biz.crm.common.ie.local.model.dto.ExportTaskDetailModelDto;
import com.biz.crm.common.ie.local.model.vo.ExportTaskDetailUpdateModelVo;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.datasource.configuration.MultiDataSourceProperties;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 导出任务子任务(repository)
 *
 * @author sunx
 * @date 2022-05-18 16:31:07
 */
@Component
public class ExportTaskDetailRepository
        extends ServiceImpl<ExportTaskDetailMapper, ExportTaskDetail> {

  @Autowired
  private MultiDataSourceProperties multiDataSourceProperties;

  /**
   * 根据主任务编码获取关联的子任务信息
   *
   * @param taskCodeSet 任务编码集合
   * @return
   */
  public List<ExportTaskDetail> findByTaskCodes(Set<String> taskCodeSet) {
    return this.lambdaQuery()
            .in(ExportTaskDetail::getTaskCode, taskCodeSet)
            .orderByAsc(ExportTaskDetail::getTaskCode, ExportTaskDetail::getPageNo)
            .list();
  }

  /**
   * 根据主任务编码删除子任务
   *
   * @param taskCode 主任务编码
   */
  public void deleteByTaskCode(String taskCode) {
    LambdaQueryWrapper<ExportTaskDetail> lambdaQuery = Wrappers.lambdaQuery();
    lambdaQuery.in(ExportTaskDetail::getTaskCode, taskCode);
    this.baseMapper.delete(lambdaQuery);
  }

  /**
   * 根据ExportTaskDetailModelDto获取任务信息(主要用于加载任务)
   *
   * @param dto 导出任务model-dto
   * @return
   */
  public List<ExportTaskDetail> findByExportTaskDetailModelDto(ExportTaskDetailModelDto dto) {
    return this.baseMapper.findByExportTaskDetailModelDto(dto);
  }

  /**
   * 更新子任务加载状态
   *
   * @param detailCodeSet 子任务编码集合
   * @param loadStatus    加载状态
   */
  public void updateLocalStatus(Set<String> detailCodeSet, String loadStatus) {
    this.lambdaUpdate()
            .in(ExportTaskDetail::getDetailCode, detailCodeSet)
            .set(ExportTaskDetail::getLoadStatus, loadStatus)
            .update();
  }

  /**
   * 根据任务编码集合更新执行状态
   *
   * @param execStatusEnum 执行状态
   * @param taskCodes 任务编码集合
   */
  public void updateExecStatusByTaskCodes(ExecStatusEnum execStatusEnum, Set<String> taskCodes) {
    LambdaUpdateChainWrapper<ExportTaskDetail> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ExportTaskDetail::getAppCode, TenantUtils.getAppCode());
    }
    lambdaUpdate
        .in(ExportTaskDetail::getTaskCode, taskCodes)
        .set(ExportTaskDetail::getExecStatus, execStatusEnum.getDictCode())
        .update();
  }

  /**
   * 更新子任务信息
   *
   * @param vo 导出子任务更新信息
   */
  public void updateByExportTaskDetailUpdateModelVo(ExportTaskDetailUpdateModelVo vo) {
    if (vo == null || StringUtils.isBlank(vo.getDetailCode())) {
      return;
    }
    ExportTaskDetail detail = this.findByDetailCode(vo.getDetailCode());
    if (StringUtils.isNotBlank(vo.getExecStatus())) {
      detail.setExecStatus(vo.getExecStatus());
    }
    if (StringUtils.isNotBlank(vo.getFileCode())) {
      detail.setFileCode(vo.getFileCode());
    }
    if (StringUtils.isNotBlank(vo.getRemark())) {
      detail.setRemark(vo.getRemark());
    }
    if (StringUtils.isNotBlank(vo.getLoadStatus())) {
      detail.setLoadStatus(vo.getLoadStatus());
    }
    detail.setExecuteDuration(vo.getExecuteDuration());
    this.updateById(detail);
  }

  /**
   * 根据子任务编码获取子任务信息
   *
   * @param detailCode 子任务编码
   * @return
   */
  public ExportTaskDetail findByDetailCode(String detailCode) {
    return this.lambdaQuery().eq(ExportTaskDetail::getDetailCode, detailCode).one();
  }
}
