package com.biz.crm.common.ie.local.service.internal;

import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.model.vo.ImportTaskUpdateModelVo;
import com.biz.crm.common.ie.local.repository.ImportTaskRepository;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.datasource.configuration.MultiDataSourceProperties;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import java.util.Objects;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import lombok.extern.slf4j.Slf4j;

/**
 * ImportTask本地线程缓存服务
 * 
 * @author xzkne
 * @version 1.0 Copyright 2023年8月29日 下午2:45:58
 */
@Slf4j
@Service
public class ImportTaskThreadLocalServImpl {
  @Autowired(required = false)
  private ImportTaskRepository importTaskRepository;
  @Autowired
  private MultiDataSourceProperties multiDataSourceProperties;

  private static final Interner<String> TASKCODE_LOCKPOOL = Interners.newWeakInterner();
  private static final ThreadLocal<ImportTask> IMPORTTASK_THREADLOCAL = new ThreadLocal<ImportTask>();

  /**
   * 获取当前线程下获取到的ImportTask对象
   * 
   * @param taskCode
   * @return
   */
  public ImportTask findByTaskCode(String taskCode) {
    ImportTask importTask = IMPORTTASK_THREADLOCAL.get();
    if (importTask == null) {
      String taskCodeIntern = TASKCODE_LOCKPOOL.intern(taskCode);
      synchronized (taskCodeIntern) {
        if (importTask == null) {
          importTask = this.importTaskRepository.findByTaskCode(taskCode);
          IMPORTTASK_THREADLOCAL.set(importTask);
          log.debug("线程{}加载数据{}", Thread.currentThread().getId(), importTask);
        }
      }
    }
    return importTask;
  }

  /**
   * 清理当前线程缓存
   */
  public void clean() {
    IMPORTTASK_THREADLOCAL.remove();
  }

  public void updateByImportTaskUpdateModelVo(ImportTaskUpdateModelVo vo) {
    Validate.notNull(vo, "更新信息不能为空");
    Validate.notEmpty(vo.getTaskCode(), "更新任务编码不能为空");
    Validate.notEmpty(vo.getExecStatus(), "任务执行状态不能为空");

    String taskCode = vo.getTaskCode();
    ImportTask importTask = this.findByTaskCode(taskCode);
    if (StringUtils.equals(importTask.getTaskCode(), vo.getTaskCode())) {
      // taskCode相同，可以进行更新
      importTask.setExecStatus(vo.getExecStatus());
      if (Objects.nonNull(vo.getTotal())) {
        importTask.setTotal(vo.getTotal());
      }
      if (Objects.nonNull(vo.getFailedNum())) {
        importTask.setFailedNum(vo.getFailedNum());
      }
      if (StringUtils.isNotBlank(vo.getFailedFileCode())) {
        importTask.setFailedFileCode(vo.getFailedFileCode());
      }
      if (StringUtils.isNotBlank(vo.getRemark())) {
        importTask.setRemark(vo.getRemark());
      }
      if (StringUtils.isNotBlank(vo.getLoadStatus())) {
        importTask.setLoadStatus(vo.getLoadStatus());
      }
      if (vo.getImportDataStatus() != null) {
        importTask.setImportDataStatus(vo.getImportDataStatus().getDictCode());
      }
      if (multiDataSourceProperties.getSchemaForTenant()) {
        importTask.setAppCode(TenantUtils.getAppCode());
      }
      this.importTaskRepository.updateById(importTask);
    } else { // taskCode不同，报一个异常吧，后续数据将会不同
      log.warn("更新的数据TaskCode与缓存不同，请联系开发团队检查程序！！！");
      this.importTaskRepository.updateByImportTaskUpdateModelVo(vo);
    }

  }
}
