package com.biz.crm.common.ie.local.service.task;

import com.biz.crm.common.ie.local.concurrent.threadpool.priority.ThreadPoolExecutorPriorityBlocking;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.google.common.collect.Maps;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ExecutorBuilder;
import cn.hutool.core.thread.RejectPolicy;
import cn.hutool.core.thread.ThreadFactoryBuilder;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 导入导出线程池
 *
 * @author sunx
 * @date 2022/5/10
 */
@Slf4j
public class ImportExportThreadPoolExecutor {

  /** 导入并行任务数量 */
  @Getter
  @Setter
  private Integer importTaskCount;

  /** 导入线程池等待队列大小 */
  @Getter
  @Setter
  private Integer importWorkQueueCount;

  /** 导出并行任务数量 */
  @Getter
  @Setter
  @Deprecated
  private Integer exportTaskCount;

  /** 大数据量导出任务并行数量 */
  @Getter
  @Setter
  private Integer largeExportTaskCount;

  /** 大数据量导出线程池等待队列大小 */
  @Getter
  @Setter
  private Integer largeExportWorkQueueCount;

  /** 大数据量任务达标数据量 */
  @Getter
  @Setter
  private Integer largeExportTaskDataNum;

  /** 中数据量导出任务并行数量 */
  @Getter
  @Setter
  private Integer mediumExportTaskCount;

  /** 中数据量导出线程池等待队列大小 */
  @Getter
  @Setter
  private Integer mediumExportWorkQueueCount;

  /** 中数据量任务达标数据量 */
  @Getter
  @Setter
  private Integer mediumExportTaskDataNum;

  /** 小数据量导出任务并行数量 */
  @Getter
  @Setter
  private Integer smallExportTaskCount;

  /** 小数据量导出线程池等待队列大小 */
  @Getter
  @Setter
  private Integer smallExportWorkQueueCount;
  
  /** 导出获取总数任务数量 */
  @Getter
  @Setter
  private Integer exportGetTotalTaskCount;

  /** 导出获取总数任务线程池等待队列大小 */
  @Getter
  @Setter
  private Integer exportGetTotalWorkQueueCount;

  /** 导出ES分段任务数量 */
  @Getter
  @Setter
  private Integer exportEsParagraphTaskCount;

  /** 导出ES分段任务线程池等待队列大小 */
  @Getter
  @Setter
  private Integer exportEsParagraphWorkQueueCount;

  /** 导入任务线程池 */
  private static final Map<String, ThreadPoolExecutor> IMPORT_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();

  /** 导出任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /** 大量数据导出任务线程池 */
  private static final Map<String, ThreadPoolExecutorPriorityBlocking> LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING =
      Maps.newHashMap();
  /** 中量数据导出任务线程池 */
  private static final Map<String, ThreadPoolExecutorPriorityBlocking> MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING =
      Maps.newHashMap();
  /** 小量数据导出任务线程池 */
  private static final Map<String, ThreadPoolExecutorPriorityBlocking> SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING =
      Maps.newHashMap();
  /** 导出任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /** 导出ES分段任务线程池 */
  private static final Map<String, ThreadPoolExecutor> EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP = Maps.newHashMap();
  /**
   * 根据应用-顶级租户编码获取导入任务线程池
   *
   * @param key
   * @return
   */
  public ThreadPoolExecutor findImportThreadPoolExecutorByKey(String key) {
    this.createImportPoolByKey(key);
    return IMPORT_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  public Map<String, ThreadPoolExecutor> getImportThreadPoolExecutorMap() {
    return IMPORT_THREAD_POOL_EXECUTOR_MAP;
  }

  /**
   * 根据应用-顶级租户编码获取导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutor findExportThreadPoolExecutorByKey(String key) {
    this.createExportPoolByKey(key);
    return EXPORT_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  /**
   * 根据应用-顶级租户编码获取大量数据导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutorPriorityBlocking findLargeExportThreadPoolExecutorByKeyPriorityBlocking(String key) {
    this.createLargeExportPoolByKeyPriorityBlocking(key);
    return LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.get(key);
  }

  /**
   * 根据应用-顶级租户编码获取中量数据导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutorPriorityBlocking findMediumExportThreadPoolExecutorByKeyPriorityBlocking(String key) {
    this.createMediumExportPoolByKeyPriorityBlocking(key);
    return MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.get(key);
  }

  /**
   * 根据应用-顶级租户编码获取小量数据导出任务线程池[阻塞队列线程池]
   *
   * @param key
   * @return 阻塞队列线程池
   */
  public ThreadPoolExecutorPriorityBlocking findSmallExportThreadPoolExecutorByKeyPriorityBlocking(String key) {
    this.createSmallExportPoolByKeyPriorityBlocking(key);
    return SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.get(key);
  }

  public Map<String, ThreadPoolExecutor> getExportThreadPoolExecutorMap() {
    return EXPORT_THREAD_POOL_EXECUTOR_MAP;
  }

  /**
   * 根据应用-租户创建对应的导入任务线程池
   *
   * @param key
   */
  private void createImportPoolByKey(String key) {
    if (IMPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (IMPORT_THREAD_POOL_EXECUTOR_MAP) {
      if (IMPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      ThreadPoolExecutor threadPoolExecutor = ExecutorBuilder.create()
          .setCorePoolSize(this.getImportTaskCount())
          .setMaxPoolSize(this.getImportTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getImportWorkQueueCount()))
          .setThreadFactory(
              ThreadFactoryBuilder.create()
                  .setNamePrefix(
                      CharSequenceUtil.format(
                          ImportExportConstant.IMPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key))
                  .build())
          .build();
      IMPORT_THREAD_POOL_EXECUTOR_MAP.put(key, threadPoolExecutor);
      log.info("---应用租户信息为{}导入线程池构建成功---", key);
    }
  }

  /**
   * 根据应用-租户创建对应的导出任务线程池
   *
   * @param key
   */
  private void createExportPoolByKey(String key) {
    if (EXPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportTaskCount())
          .setMaxPoolSize(this.getExportTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportTaskCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出线程池构建成功---", key);
    }
  }

  /**
   * 构建大量数据导出任务线程池
   * @param key
   */
  private void createLargeExportPoolByKeyPriorityBlocking(String key) {
    if (LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
      return;
    }
    synchronized (LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING) {
      if (LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key, "large");
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutorPriorityBlocking poolExe =
          new ThreadPoolExecutorPriorityBlocking(largeExportTaskCount, largeExportTaskCount, 1L, TimeUnit.SECONDS,
              new PriorityBlockingQueue<Runnable>(largeExportWorkQueueCount), threadFactory, RejectPolicy.BLOCK.getValue());
      LARGE_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.put(key, poolExe);
      log.info("---应用租户信息为{}大量数据导出线程池构建成功---", key);
    }
  }

  /**
   * 构建中量数据导出任务线程池
   * @param key
   */
  private void createMediumExportPoolByKeyPriorityBlocking(String key) {
    if (MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
      return;
    }
    synchronized (MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING) {
      if (MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key, "medium");
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutorPriorityBlocking poolExe =
          new ThreadPoolExecutorPriorityBlocking(mediumExportTaskCount, mediumExportTaskCount, 1L, TimeUnit.SECONDS,
              new PriorityBlockingQueue<Runnable>(mediumExportWorkQueueCount), threadFactory, RejectPolicy.BLOCK.getValue());
      MEDIUM_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.put(key, poolExe);
      log.info("---应用租户信息为{}中量数据导出线程池构建成功---", key);
    }
  }

  /**
   * 构建小量数据导出任务线程池
   * @param key
   */
  private void createSmallExportPoolByKeyPriorityBlocking(String key) {
    if (SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
      return;
    }
    synchronized (SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING) {
      if (SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_TASK_FACTORY_NAME_PREFIX_FORMAT, key, "small");
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutorPriorityBlocking poolExe =
          new ThreadPoolExecutorPriorityBlocking(smallExportTaskCount, smallExportTaskCount, 1L, TimeUnit.SECONDS,
              new PriorityBlockingQueue<Runnable>(smallExportWorkQueueCount), threadFactory, RejectPolicy.BLOCK.getValue());
      SMALL_EXPORT_THREAD_POOL_EXECUTOR_MAP_PRIORITY_BLOCKING.put(key, poolExe);
      log.info("---应用租户信息为{}小量数据导出线程池构建成功---", key);
    }
  }

  public ThreadPoolExecutor findExportGetTotalThreadPoolExecutorByKey(String key) {
    this.createExportGetTotalThreadPoolExecutorByKey(key);
    return EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.get(key);
  }
  
  private void createExportGetTotalThreadPoolExecutorByKey(String key) {
    if (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_GET_TOTAL_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportGetTotalTaskCount())
          .setMaxPoolSize(this.getExportGetTotalTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportGetTotalWorkQueueCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_GET_TOTAL_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出创建异步获取总数任务线程池构建成功---", key);
    }
  }

  public ThreadPoolExecutor findExportEsParagraphThreadPoolExecutorByKey(String key) {
    this.createExportEsParagraphThreadPoolExecutorByKey(key);
    return EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.get(key);
  }

  private void createExportEsParagraphThreadPoolExecutorByKey(String key) {
    if (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
      return;
    }
    synchronized (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP) {
      if (EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.containsKey(key)) {
        return;
      }
      String threadGroupName =
          CharSequenceUtil.format(ImportExportConstant.EXPORT_ES_PARAGRAPH_TASK_FACTORY_NAME_PREFIX_FORMAT, key);
      ThreadFactory threadFactory = ThreadFactoryBuilder.create().setNamePrefix(threadGroupName).build();
      // // 建立阻塞队列，将任务阻塞在外部
      ThreadPoolExecutor poolExe = ExecutorBuilder.create()
          .setCorePoolSize(this.getExportEsParagraphTaskCount())
          .setMaxPoolSize(this.getExportEsParagraphTaskCount())
          .setWorkQueue(new LinkedBlockingQueue<>(this.getExportEsParagraphWorkQueueCount()))
          .setThreadFactory(threadFactory)
          .build();
      EXPORT_ES_PARAGRAPH_THREAD_POOL_EXECUTOR_MAP.put(key, poolExe);
      log.info("---应用租户信息为{}导出创建ES分段任务线程池构建成功---", key);
    }
  }
}
