package com.biz.crm.common.ie.local.service.task.cache;

import com.google.common.collect.Sets;
import java.util.Set;
import org.springframework.stereotype.Component;

/**
 * @author ning.zhang
 * @description 导出任务相关缓存
 * @date 2024/07/22
 */
@Component
public class ExportTaskCache {

  /**
   * 执行中的异步获取导出总数的任务缓存
   */
  private final Set<String> execAsyncGetTotalTasks = Sets.newHashSet();

  /**
   * 执行中的ES子任务分割任务缓存
   */
  private final Set<String> execEsParagraphTasks = Sets.newHashSet();

  /**
   * 加入任务到正在执行中的异步获取导出总数缓存
   *
   * @param appCode 租户编码
   * @param applicationName 任务编码
   * @param taskCode 任务编码
   */
  public void addExecAsyncGetTotalTask(String appCode, String applicationName, String taskCode) {
    this.execAsyncGetTotalTasks.add(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }

  /**
   * 检查任务是否在正在执行中的异步获取导出总数缓存中
   *
   * @param taskCode 任务编码
   * @return 检查结果(true_在,false_不在)
   */
  public Boolean checkExecAsyncGetTotalTask(String appCode, String applicationName, String taskCode) {
    return this.execAsyncGetTotalTasks.contains(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }

  /**
   * 从正在执行中的异步获取导出总数缓存移除任务
   *
   * @param taskCode 任务编码
   */
  public void removeExecAsyncGetTotalTask(String appCode, String applicationName, String taskCode) {
    this.execAsyncGetTotalTasks.remove(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }

  /**
   * 加入任务到正在执行中的ES子任务分割任务缓存
   *
   * @param taskCode 任务编码
   */
  public void addExecEsParagraphTask(String appCode, String applicationName, String taskCode) {
    this.execEsParagraphTasks.add(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }

  /**
   * 检查任务是否在正在执行中的ES子任务分割任务缓存中
   *
   * @param taskCode 任务编码
   * @return 检查结果(true_在,false_不在)
   */
  public Boolean checkExecEsParagraphTask(String appCode, String applicationName, String taskCode) {
    return this.execEsParagraphTasks.contains(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }

  /**
   * 从正在执行中的ES子任务分割任务缓存存移除任务
   *
   * @param taskCode 任务编码
   */
  public void removeExecEsParagraphTask(String appCode, String applicationName, String taskCode) {
    this.execEsParagraphTasks.remove(String.format("%s-%s-%s", appCode, applicationName, taskCode));
  }
}
