package com.biz.crm.common.ie.local.bean;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.dto.ExportTaskQueueDto;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExportDetailProcessEnum;
import com.biz.crm.common.ie.sdk.enums.ExportProcessEnum;
import com.biz.crm.common.ie.sdk.enums.TypeEnum;
import com.biz.crm.common.ie.sdk.vo.ExportProcessMsgVo;
import com.biz.crm.common.ie.sdk.vo.ExportTaskProcessVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.gateway.websocket.client.sdk.service.ChannelMsgService;

import java.util.Date;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * 导出消息推送
 *
 * @author sunx
 * @date 2022/5/30
 */
@Component
@Slf4j
public class ExportSendProcessMsgBean {

  @Value("${spring.application.name}")
  private String applicationName;
  @Autowired(required = false)
  private ChannelMsgService channelMsgService;

  /**
   * 导入导出模块中的导入信息
   */
  private static final String IE_EXPORT_MODEL_CODE = "ie_export_model";

  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ExportSendProcessMsgBean.class);

  /**
   * 新-主任务消息通知-不带进度
   * 
   * @param task 导出任务实体
   * @param execStatus 执行状态
   * @param process 导出执行阶段
   * @param account 当前账户
   * @param mainFlag 主任务标识
   */
  public void sendTaskProcessMsg(
      String taskCode, String execStatus, ExportProcessEnum process, String account, Boolean mainFlag) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setMainFlag(mainFlag);
    vo.setExecStatus(execStatus);
    vo.setProcessType(process.getCode());
    vo.setRemark(process.getFormat());
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendMsg(vo);
  }

  /**
   * 新-主任务消息通知-不带进度
   * 
   * @param task 导出任务实体
   * @param execStatus 执行状态
   * @param process 导出执行阶段
   * @param account 当前账户
   * @param mainFlag 主任务标识
   */
  public void sendTaskProcessMsg(ExportTaskProcessVo task, String execStatus, ExportProcessEnum process, String account,
      Boolean mainFlag) {
    final String taskCode = task.getTaskCode();
    this.sendTaskProcessMsg(taskCode, execStatus, process, account, mainFlag);
  }

  /**
   * 新-主任务消息通知-带进度
   * @param task  导出任务实体
   * @param execStatus  执行状态
   * @param process 导出执行阶段
   * @param cursor  进度
   * @param account 当前账户
   * @param mainFlag  主任务标识
   */
  public void sendTaskProcessMsg(ExportTaskProcessVo task, String execStatus, ExportProcessEnum process, Integer cursor, String account, Boolean mainFlag) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    //主子任务区分
    vo.setTaskCode(mainFlag ? task.getTaskCode() : task.getDetailCode());
    vo.setMainFlag(mainFlag);
    vo.setExecStatus(execStatus);
    vo.setProcessType(process.getCode());
    vo.setRemark(process.getFormat());
    vo.setPageSize(100);
    vo.setCursor(cursor);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendMsg(vo);
  }

  /**
   * 新-主任务消息通知-带进度
   * @param task  导出任务实体
   * @param execStatus  执行状态
   * @param process 导出执行阶段
   * @param cursor  进度
   * @param account 当前账户
   * @param mainFlag  主任务标识
   * @param fileCode  住文件编码
   */
  public void sendTaskProcessMsg(ExportTaskProcessVo task, String execStatus, ExportProcessEnum process, Integer cursor, String account, Boolean mainFlag, String fileCode) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    //主子任务区分
    vo.setTaskCode(mainFlag ? task.getTaskCode() : task.getDetailCode());
    vo.setMainFlag(mainFlag);
    vo.setExecStatus(execStatus);
    vo.setProcessType(process.getCode());
    vo.setRemark(process.getFormat());
    vo.setPageSize(100);
    vo.setCursor(cursor);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    //判断是否主任务，返回主任务文件code
    if(mainFlag){
        vo.setFileCode(fileCode);
    }
    this.sendMsg(vo);
  }

  /**
   * 子任务执行中发送消息
   *
   * @param task   导出任务实体
   * @param cursor 进度
   */
  public void sendSubTaskProcessMsg(ExportTaskProcessVo task, String execStatus, ExportProcessEnum process, Integer cursor, String account) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    vo.setTaskCode(task.getDetailCode());
    vo.setMainFlag(false);
    vo.setExecStatus(execStatus);
    vo.setProcessType(process.getCode());
    vo.setRemark(process.getFormat());
    vo.setPageSize(100);
    vo.setCursor(cursor);
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendMsg(vo);
  }

  /**
   * 子任务开始、结束消息通知 ExportDetailProcessEnum（START、END）
   *
   * @param detailCode 子任务编码
   * @param execStatus 执行状态
   * @param process    导出执行阶段
   */
  public void sendDetailTaskProcessMsg(
          String detailCode, String execStatus, ExportDetailProcessEnum process, String account) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    vo.setTaskCode(detailCode);
    vo.setExecStatus(execStatus);
    vo.setProcessType(process.getCode());
    vo.setRemark(process.getFormat());
    vo.setAccount(account);
    vo.setApplicationName(applicationName);
    this.sendMsg(vo);
  }

  /**
   * 子任务处理过程中 发送 主子任务消息
   *
   * @param task          导出任务实体
   * @param detailProcess 子任务执行阶段
   * @param process       主任务执行阶段
   */
  public void sendStartProcessMsg(
          ExportTaskProcessVo task, ExportDetailProcessEnum detailProcess, ExportProcessEnum process, String account) {

    String[] str = task.getDetailCode().split("_");
    for(String item : str){
      if(ImportExportConstant.ONE_MAIN_FLAG.equals(item)){
        final ExportProcessMsgVo mainVo = new ExportProcessMsgVo();
        mainVo.setTaskCode(task.getTaskCode());
        mainVo.setMainFlag(true);
        mainVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
        mainVo.setProcessType(process.getCode());
        mainVo.setRemark(process.getFormat());
        mainVo.setPageSize(100);
        mainVo.setCursor(0);
        mainVo.setAccount(account);
        mainVo.setApplicationName(applicationName);
        this.sendMsg(mainVo);
      }
    }


    final ExportProcessMsgVo detailVo = new ExportProcessMsgVo();
    detailVo.setTaskCode(task.getDetailCode());
    detailVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
    detailVo.setProcessType(detailProcess.getCode());
    detailVo.setRemark(detailProcess.getFormat());
    detailVo.setPageSize(task.getTotal());
    detailVo.setCursor(0);
    detailVo.setMainFlag(false);
    detailVo.setAccount(account);
    detailVo.setApplicationName(applicationName);
    this.sendMsg(detailVo);
  }

  /**
   * 发送失败信息消息(FAILED)
   *
   * @param taskCode 任务编码
   * @param mainFlag 是否是主任务
   * @param msg      进度消息
   */
  public void sendFailedProcessMsg(String taskCode, Boolean mainFlag, String msg, String account) {
    ExportProcessMsgVo vo = new ExportProcessMsgVo();
    vo.setTaskCode(taskCode);
    vo.setMainFlag(mainFlag);
    vo.setIeType(TypeEnum.EXPORT.getDictCode());
    vo.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    vo.setProcessType(ExportProcessEnum.END.getCode());
    vo.setRemark(msg);
    vo.setAccount(account);
    vo.setCursor(0);
    vo.setApplicationName(applicationName);
    this.sendMsg(vo);
  }

  /**
   * 推送主任务消息排队进度
   *
   * @param task
   */
  public void sendTaskProcessQueueMsg(ExportTaskQueueDto task, Integer sort) {
    final ExportProcessMsgVo vo = new ExportProcessMsgVo();
    vo.setTaskCode(task.getTaskCode());
    vo.setMainFlag(task.isMainFlag());
    vo.setRemark("当前排队顺序...");
    vo.setAccount(task.getAccount());
    vo.setTaskSort(sort);
    this.sendMsg(vo);
  }

  /**
   * 发送消息
   *
   * @param vo 导出任务消息实体
   */
  public void sendMsg(ExportProcessMsgVo vo) {
    vo.setTime(new Date());
    byte[] jsonBytes = JSONObject.toJSONBytes(vo);
    String tenantCode = TenantUtils.getTenantCode();
    //导入导出公用通道
    String ieChannelTaskCode = ImportExportConstant.IE_CHANNEL_TASK_CODE;
    //所有业务所属通道标识
    String exportAndImportModelCode = ImportExportConstant.IE_EXPORT_IMPORT_MODEL_CODE;
    LOGGER.info(" export model send msg: tenantCode = {} , applicationName = {} , modelCode = {} ", tenantCode, applicationName, IE_EXPORT_MODEL_CODE);
    if (StringUtils.isNotBlank(vo.getAccount())) {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, applicationName, IE_EXPORT_MODEL_CODE, vo.getAccount(), jsonBytes);
      //使用公共通道
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, ieChannelTaskCode, exportAndImportModelCode, vo.getAccount(), jsonBytes);
    } else {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, applicationName, IE_EXPORT_MODEL_CODE, jsonBytes);
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, ieChannelTaskCode, exportAndImportModelCode, jsonBytes);
    }
  }

  /**
   * 新-发送消息 只推新版消息通道
   *
   * @param vo 导出任务消息实体
   */
  public void sendMsgByAll(ExportProcessMsgVo vo) {
    vo.setTime(new Date());
    byte[] jsonBytes = JSONObject.toJSONBytes(vo);
    String tenantCode = TenantUtils.getTenantCode();
    //导入导出公用通道
    String ieChannelTaskCode = ImportExportConstant.IE_CHANNEL_TASK_CODE;
    //所有业务所属通道标识
    String exportAndImportModelCode = ImportExportConstant.IE_EXPORT_IMPORT_MODEL_CODE;
    LOGGER.info(" export model send msg: tenantCode = {} , applicationName = {} , modelCode = {} ", tenantCode, applicationName, IE_EXPORT_MODEL_CODE);
    if (StringUtils.isNotBlank(vo.getAccount())) {
      //使用公共通道
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCodeAndAccount(tenantCode, ieChannelTaskCode, exportAndImportModelCode, vo.getAccount(), jsonBytes);
    } else {
      this.channelMsgService.sendByTenantCodeAndApplicationNameAndModelCode(tenantCode, ieChannelTaskCode, exportAndImportModelCode, jsonBytes);
    }
  }
}
