package com.biz.crm.common.ie.local.service.internal;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.MultipartFileUtil;
import com.biz.crm.common.ie.local.bean.ImportSendProcessMsgBean;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.entity.ImportTemplateDetail;
import com.biz.crm.common.ie.local.model.vo.ImportTaskUpdateModelVo;
import com.biz.crm.common.ie.local.repository.ImportTaskRepository;
import com.biz.crm.common.ie.local.repository.ImportTemplateDetailRepository;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.enums.CallbackStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExcelFileTypeEnum;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ImportDataStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ImportProcessEnum;
import com.biz.crm.common.ie.sdk.enums.TypeEnum;
import com.biz.crm.common.ie.sdk.event.ImportNotifyEventListener;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelColumn;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelImport;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.excel.strategy.CrmExcelProcessStrategy;
import com.biz.crm.common.ie.sdk.service.ImportProcessService;
import com.biz.crm.common.ie.sdk.vo.ExcelFileVo;
import com.biz.crm.common.ie.sdk.vo.IeExecutorVo;
import com.biz.crm.common.ie.sdk.vo.ImportExcelLocalFile;
import com.biz.crm.common.ie.sdk.vo.ImportProcessMsgVo;
import com.biz.crm.common.ie.sdk.vo.ImportTaskEventVo;
import com.biz.crm.common.ie.sdk.vo.ImportTaskHandlerResultVo;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.venus.sdk.dto.Base64UploadDto;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.web.multipart.MultipartFile;

/**
 * 导入操作实现
 *
 * @author sunx
 * @date 2022/5/12
 */
@Service
@Slf4j
public class ImportProcessServiceImpl implements ImportProcessService {

  @Autowired(required = false)
  private ImportTaskThreadLocalServImpl importTaskThreadLocalServImpl;

  @Autowired(required = false)
  private ImportTemplateDetailRepository importTemplateDetailRepository;

  @Value("${spring.application.name:}")
  private String subsystem;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private FileHandleService fileHandleService;

  @Autowired(required = false)
  private ImportSendProcessMsgBean importSendProcessMsgBean;

  @Autowired(required = false)
  private List<ImportProcess> importProcesses;

  @Autowired(required = false)
  private CrmExcelProcessStrategy crmExcelProcessStrategy;

  @Autowired(required = false)
  private ImportExportProperties importExportProperties;

  @Autowired(required = false)
  private List<ImportNotifyEventListener> notifyEventListeners;
  @Autowired(required = false)
  private RedisMutexService redisMutexService;
  @Autowired(required = false)
  private ImportTaskRepository importTaskRepository;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Override
  public void importProcess(IeExecutorVo vo) {
    try {
      // 检查流程进度
      ImportTask task = this.importTaskThreadLocalServImpl.findByTaskCode(vo.getTaskCode());
      // 推送导入文件名称
      OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(task.getFileCode());
      ImportProcessMsgVo msgVo = new ImportProcessMsgVo();
      msgVo.setTaskCode(vo.getTaskCode());
      msgVo.setIeType(TypeEnum.IMPORT.getDictCode());
      msgVo.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
      msgVo.setProcessType(ImportProcessEnum.GET_PROCESS.getCode());
      msgVo.setRemark("推送文件名称");
      msgVo.setAccount(task.getCreateAccount());
      msgVo.setApplicationName(subsystem);
      if (ordinaryFileVo != null) {
        msgVo.setFileName(ordinaryFileVo.getOriginalFileName());
      }
      this.importSendProcessMsgBean.sendMsgByAll(msgVo);

      vo.setAccount(task.getCreateAccount());
      Validate.notNull(task, CharSequenceUtil.format("导入任务编码为{}的任务不存在", vo.getTaskCode()));
      boolean f =
          task.getExecStatus().equals(ExecStatusEnum.CANCEL.getDictCode())
              || task.getExecStatus().equals(ExecStatusEnum.FINISH.getDictCode());
      if (f) {
        this.importTaskThreadLocalServImpl.clean();// 清理缓存
        return;
      }
      // 推送进度信息
      this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), ImportProcessEnum.START, vo.getAccount());
      
      
      // -- 1. 执行导入操作
      this.process(vo);
      
      // -- 2. 发送最后的成功消息
      ImportTask importTask = this.importTaskThreadLocalServImpl.findByTaskCode(vo.getTaskCode());
      // 增加失败数据的显示
      ExecStatusEnum execStatusEnum = null;
      ImportProcessEnum importProcessEnum = null;
      if (importTask.getFailedNum() == 0) {
        // 没有失败的记录，任务成功
        execStatusEnum = ExecStatusEnum.FINISH;
      } else {
        importProcessEnum = ImportProcessEnum.VERIFY_FAIL;
        if (Objects.equals(importTask.getTotal(), importTask.getFailedNum())) {
          // 如果失败数量等于总记录数
          execStatusEnum = ExecStatusEnum.FAILED;
        } else {
          // 如果失败数量小于总记录数
          execStatusEnum = ExecStatusEnum.PARTIAL_SUCCESS;
        }
        // 校验失败发送
        this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), importProcessEnum,
            execStatusEnum.getDictCode(), importProcessEnum.getDesc(), vo.getAccount(), importTask);
        ThreadUtil.sleep(20);
      }
      //当文件处理结束 失败和部分失败携带失败的文件code
      importProcessEnum = ImportProcessEnum.END;
      this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), importProcessEnum,
          execStatusEnum.getDictCode(), importProcessEnum.getDesc(), vo.getAccount(), importTask);
    } catch (feign.FeignException fe) {
      this.handlerTaskFailed(vo.getTaskCode(), "导入Feign调用失败", vo.getAccount());
      log.error("导入任务{}调用Fegin失败,err[{}]", vo.getTaskCode(), Throwables.getStackTraceAsString(fe));
    } catch (RuntimeException ex) {
      this.handlerTaskFailed(vo.getTaskCode(), ex.getMessage(), vo.getAccount());
      log.error("导入任务{}处理失败,err[{}]", vo.getTaskCode(), Throwables.getStackTraceAsString(ex));
    } catch (Exception ex) {
      this.handlerTaskFailed(vo.getTaskCode(), "导入失败", vo.getAccount());
      log.error("导入任务{}处理失败,err[{}]", vo.getTaskCode(), Throwables.getStackTraceAsString(ex));
    } finally {
      // 触发回调操作
      this.callBack(vo.getTaskCode());
      this.importTaskThreadLocalServImpl.clean();// 清理缓存
    }
  }

  /**
   * 导入任务执行
   *
   * <p>1、任务预处理
   *
   * <p>2、获取任务处理器（process）
   *
   * <p>3、获取导入文件原文件+模板文件信息
   *
   * <p>4、处理excel导入
   *
   * @param vo
   */
  private void process(IeExecutorVo vo) {
    StopWatch sw = new StopWatch(CharSequenceUtil.format("导入任务执行{}", vo.getTaskCode()));
    
    // * <p>1、任务预处理
    sw.start("1、任务预处理");
    ImportTask task = this.importTaskHandler(vo);
    final ImportTemplateDetail templateDetail =
            this.importTemplateDetailRepository.findDetailByTemplateCode(task.getTemplateCode());
    Validate.notNull(templateDetail, "未配置导入的模板信息");
    final String templateFileCode = templateDetail.getFileCode();
    Validate.notEmpty(templateFileCode, "未配置导入的模板信息");
    sw.stop();

    // * <p>2、获取任务处理器（process）
    // 推送消息
    this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), ImportProcessEnum.GET_PROCESS, vo.getAccount());
    ImportProcess process = this.findImportProcess(task.getTemplateCode());
    CrmExcelImport crmExcelImport =
            (CrmExcelImport) process.findCrmExcelVoClass().getAnnotation(CrmExcelImport.class);
    Validate.notNull(crmExcelImport, "excel导入bean未配置CrmExcelImport注解");
    final Field[] fields = ReflectUtil.getFields(process.findCrmExcelVoClass());
    long count =
            Stream.of(fields)
                    .filter(a -> Objects.nonNull(a.getAnnotation(CrmExcelColumn.class)))
                    .count();
    Validate.isTrue(count > 0, "excel导入bean的字段属性未配置CrmExcelColumn注解");

    
    // * <p>3、获取导入文件原文件+模板文件信息
    sw.start("3、获取原文件+模板文件信息");
    // 推送消息
    this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), ImportProcessEnum.DOWNLOAD, vo.getAccount());

    Map<String, OrdinaryFileVo> downloadFileMap = Maps.newHashMap();
    OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(task.getFileCode());
    if (ordinaryFileVo != null) {
      downloadFileMap.put(task.getFileCode(), ordinaryFileVo); 
      // XZK String originalFileName = ordinaryFileVo.getOriginalFileName(); // 原始文件名，eg：企业组织导入模板_111.xls
    }

    OrdinaryFileVo templateFileVo = this.fileHandleService.findById(templateFileCode);
    if (templateFileVo != null) {
      downloadFileMap.put(templateFileCode, templateFileVo);
    }

    boolean fileFlag = downloadFileMap.containsKey(task.getFileCode())
            && downloadFileMap.get(task.getFileCode()).getFileSize() > 0;
    Validate.isTrue(fileFlag, CharSequenceUtil.format("未获取到任务{}对应的原文件信息", vo.getTaskCode()));
    boolean templateFileFlag =
            downloadFileMap.containsKey(templateFileCode)
                    && downloadFileMap.get(templateFileCode).getFileSize() > 0;
    Validate.isTrue(templateFileFlag, CharSequenceUtil.format("未获取到任务{}对应的模板文件信息", vo.getTaskCode()));
    sw.stop();

    // * <p>4、处理excel导入【核心】
    sw.start("4、处理excel导入");
    // XZK ImportExcelLocalFile
    ImportExcelLocalFile localExcelFileMap =
        this.createLocalExcelFile(downloadFileMap, task.getFileCode(), templateFileCode, task.getTaskCode());
    final String parentPath =localExcelFileMap.getOrdinaryFile().getParentPath();

    // 推送消息
    this.importSendProcessMsgBean.sendProcessMsg(vo.getTaskCode(), ImportProcessEnum.START_EXEC, vo.getAccount());
    // [核心调用：处理Excel]
    try {
      this.excelHandler(task, process, localExcelFileMap);
    } catch (Exception e) {
      log.error("导入出错了{},error[{}]", vo.getTaskCode(), Throwables.getStackTraceAsString(e));
      Validate.isTrue(false, e.getMessage());
    } finally {
      FileUtil.del(parentPath);
    }
    sw.stop();
    log.info(CharSequenceUtil.format("******导入任务执行{}，执行统计{}****", vo.getTaskCode(), sw.prettyPrint()));
  }

  /**
   * 该私有方法为venus文件上传接口参数封装
   */
  private List<OrdinaryFileVo> venusFileUpload(MultipartFile multipartFile) throws IOException {
    Base64UploadDto base64UploadDto = new Base64UploadDto();
    base64UploadDto.setCreator(loginUserService.findCurrentAccount());
    base64UploadDto.setFileNanmes(new String[]{multipartFile.getOriginalFilename()});
    base64UploadDto.setBase64Contents(new String[]{Base64Utils.encodeToString(multipartFile.getBytes())});
    //subsystem不能有横杠只能用下划线分隔
    List<OrdinaryFileVo> ordinaryFileVoList = this.fileHandleService.fileUpload(subsystem.replace("-", "_"), base64UploadDto, null);
    Validate.notEmpty(ordinaryFileVoList, "文件上传失败！");
    return ordinaryFileVoList;
  }

  /**
   * excel处理
   *
   * @param task
   * @param process
   * @param localFile
   */
  private void excelHandler(
          ImportTask task, ImportProcess process, ImportExcelLocalFile localFile) {
    if (localFile == null
        || localFile.getOrdinaryFile() == null
        || localFile.getTemplateFile() == null) {
      Validate.isTrue(false, "任务文件加载失败");
    }
    final ExcelFileVo originalFile = localFile.getOrdinaryFile();
    final ExcelFileVo templateFile = localFile.getTemplateFile();
    String account = task.getCreateAccount();
    String taskCode = task.getTaskCode();
    StopWatch sw = new StopWatch(CharSequenceUtil.format("====导入任务执行{}-excel处理===", taskCode));
    Map<String, Object> params = this.findParamsMap(task);
    TaskGlobalParamsVo paramsVo = this.findParams(task);

    
    String failedFileCode = StringUtils.EMPTY;
    sw.start("excel逻辑处理");
    // 该步骤会产生错误文件，将错误文件进行记录
    final ImportTaskHandlerResultVo resultVo =
            this.crmExcelProcessStrategy.importExcel(process, paramsVo, params, localFile);
    sw.stop();

    // 检查是否有错误文件
    ExcelFileVo errorFileVo = localFile.getErrorFile();
    File errorFile = new File(errorFileVo.getPath());
    // 上传错误文件
    if (FileUtil.exist(errorFile) && FileUtil.size(FileUtil.file(errorFile)) > 0) {
      // 推送进度信息
      this.importSendProcessMsgBean.sendProcessMsg(taskCode, ImportProcessEnum.ERROR_DATA, account);
      sw.start("上传错误文件");
      try (BufferedInputStream inputStream = FileUtil.getInputStream(errorFile)){
        final MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(inputStream, errorFileVo.getFileName());
        OrdinaryFileVo uploadVo = this.venusFileUpload(multipartFile).get(0);
        failedFileCode = uploadVo.getId();
      } catch (Exception e) {
        log.error("上传错误文件失败{},[{}]", taskCode, Throwables.getStackTraceAsString(e));
        Validate.isTrue(false, "上传错误文件失败");
      }
      sw.stop();
    }
    // 更新任务信息
    this.importSendProcessMsgBean.sendProcessMsg(taskCode, ImportProcessEnum.UPDATE_TASK, account);
    ImportTaskUpdateModelVo cur = new ImportTaskUpdateModelVo();
    cur.setTaskCode(taskCode);
    cur.setExecStatus(ExecStatusEnum.FINISH.getDictCode());
    cur.setTotal(resultVo.getTotal());
    cur.setFailedNum(resultVo.getFailedNum());
    cur.setImportDataStatus(resultVo.getImportDataStatus());
    cur.setFailedFileCode(failedFileCode);
    sw.start("更新任务状态");
    
    // 修改状态
    this.importTaskThreadLocalServImpl.updateByImportTaskUpdateModelVo(cur);
    sw.stop();
    log.info(
            CharSequenceUtil.format("+++++导入任务执行{}-excel处理，执行统计{}++++", taskCode, sw.prettyPrint()));
  }

  /**
   * 导入任务预处理
   *
   * @param vo
   * @return
   */
  private ImportTask importTaskHandler(IeExecutorVo vo) {
    ImportTask task = this.importTaskThreadLocalServImpl.findByTaskCode(vo.getTaskCode());
    Validate.notNull(task, CharSequenceUtil.format("导入任务编码为{}的任务不存在", vo.getTaskCode()));
    boolean f =
            task.getExecStatus().equals(ExecStatusEnum.CANCEL.getDictCode())
                    || task.getExecStatus().equals(ExecStatusEnum.FINISH.getDictCode());
    Validate.isTrue(!f, CharSequenceUtil.format("导入任务编码为{}的任务已完成或取消", vo.getTaskCode()));
    // 更新任务待执行任务状态为正在执行
    if (task.getExecStatus().equals(ExecStatusEnum.DEFAULT.getDictCode())) {
      ImportTaskUpdateModelVo cur = new ImportTaskUpdateModelVo();
      cur.setTaskCode(vo.getTaskCode());
      cur.setLoadStatus(vo.getLoadStatus());
      cur.setExecStatus(ExecStatusEnum.RUNNING.getDictCode());
      this.importTaskThreadLocalServImpl.updateByImportTaskUpdateModelVo(cur);
    }
    return task;
  }

  /**
   * 获取任务参数信息
   *
   * @param importTask
   * @return
   */
  private Map<String, Object> findParamsMap(ImportTask importTask) {
    if (Objects.isNull(importTask) || StringUtils.isBlank(importTask.getParametersJson())) {
      return Maps.newHashMap();
    }
    Map<String, Object> map = Maps.newHashMap();
    JSONObject jsonObject = JSONUtil.parseObj(importTask.getParametersJson());
    Set<String> set = jsonObject.keySet();
    for (String item : set) {
      map.put(item, jsonObject.get(item));
    }
    return map;
  }

  /**
   * 获取对应的数据处理器
   *
   * @param code
   * @return
   */
  private ImportProcess findImportProcess(String code) {
    if (CollectionUtils.isEmpty(importProcesses) || StringUtils.isBlank(code)) {
      return null;
    }
    Optional<ImportProcess> op =
            importProcesses.stream().filter(item -> item.getTemplateCode().equals(code)).findFirst();
    Validate.notNull(op, CharSequenceUtil.format("未获取到模板编码{}对应的处理器信息", code));
    return op.orElse(null);
  }

  /**
   * 加载源文件\并创建错误文件
   *
   * @param downloadFileMap
   * @param fileCode
   * @param templateFileCode
   * @param taskCode 任务编码，打印日志使用
   * @return
   */
  private ImportExcelLocalFile createLocalExcelFile(
          Map<String, OrdinaryFileVo> downloadFileMap, String fileCode, String templateFileCode, String taskCode) {
    ImportExcelLocalFile localFile = new ImportExcelLocalFile();
    if (downloadFileMap == null
            || !downloadFileMap.containsKey(fileCode)
            || !downloadFileMap.containsKey(templateFileCode)) {
      return localFile;
    }
    OrdinaryFileVo ordinaryFileVo = downloadFileMap.get(fileCode);
    OrdinaryFileVo templateFileVo = downloadFileMap.get(templateFileCode);
    File ordinaryFile, templateFile, errorFile;
    { // 1 导入文件、模板文件命名定义
      Date now = new Date();
      String root = this.importExportProperties.getRoot();
      if (StringUtils.isBlank(root)) {
        root = FileUtil.getTmpDirPath();
      }
      if (root.endsWith("/")) { // 如果以/结尾，则去掉最后一个/符号
        root = root.substring(0, root.length() - 1);
      }
      String parentPath = CharSequenceUtil.format("{}/import/{}/{}",
          root,
          DateUtil.format(now, "yyyy/MM-dd"),
          fileCode);
      // 导入文件
      String originalFileName = ordinaryFileVo.getOriginalFileName();
      ordinaryFile = new File(parentPath, originalFileName);
      // 模板文件
      String templateFileName = originalFileName.substring(0, originalFileName.lastIndexOf("."));
      templateFileName += ("_" + ExcelFileTypeEnum.TEMPLATE.getValue() + ImportExportConstant.IE_FILE_SUFFIX);
      templateFile = new File(parentPath, templateFileName);
      // 错误文件
      String errorFileName = this.crmExcelProcessStrategy.getErrorExcelFileName(originalFileName);
      errorFile = new File(parentPath, errorFileName);
    }
    

    CountDownLatch latch = new CountDownLatch(2);
    
    {// 2.1 下载导入文件
      byte[] bytes = this.fileHandleService.findContentByFilePathAndFileRename(ordinaryFileVo.getRelativeLocal(),
          ordinaryFileVo.getFileName());

      if (bytes != null) {
        final byte[] writeBytes = bytes;
        Thread writeFileThread = new Thread(() -> {
          File tmpFile = ordinaryFile;
          try {
            Thread.currentThread().setName("文件写入" + tmpFile);
            FileUtil.writeBytes(writeBytes, tmpFile);// 下载导入文件
            ExcelFileVo vo = this.convertExcelFileVo(tmpFile);
            localFile.setOrdinaryFile(vo);
          } catch (Exception e) {
            log.error("ERR230904-0101:导入文件写入异常:" + tmpFile, e);
          } finally {
            latch.countDown();
          }
        });
        writeFileThread.start();
      } else {
        throw new RuntimeException("ERR230904-0102:下载导入文件异常:" + ordinaryFileVo);
      }
    }
    {// 2.2 下载模板文件
      byte[] bytes = this.fileHandleService
          .findContentByFilePathAndFileRename(templateFileVo.getRelativeLocal(), templateFileVo.getFileName());
      if (bytes != null) {
        final byte[] writeBytes = bytes;
        Thread writeFileThread = new Thread(() -> {
          File tmpFile = templateFile;
          try {
            Thread.currentThread().setName("文件写入" + tmpFile);
            FileUtil.writeBytes(writeBytes, tmpFile); // 下载模板文件
            ExcelFileVo vo = this.convertExcelFileVo(tmpFile);
            localFile.setTemplateFile(vo);
          } catch (Exception e) {
            log.error("ERR230904-0201:模板文件写入异常:" + tmpFile, e);
          } finally {
            latch.countDown();
          }
        });
        writeFileThread.start();
      } else {
        throw new RuntimeException("ERR230904-0202:下载模板文件异常:" + ordinaryFileVo);
      }
    }
    {// 2.3 错误文件仅将文件定义保存
      ExcelFileVo errorFileExcelFileVo = this.convertExcelFileVo(errorFile);
      localFile.setErrorFile(errorFileExcelFileVo);
    }
    try {
      latch.await();
    } catch (InterruptedException e) {
      log.error("写入本地文件发生异常{}", e);
      throw new RuntimeException("等待写入文件异常", e);
    }
    // 检查文件是否有准备好了
    Validate.isTrue(localFile.getOrdinaryFile() != null, "ERR230904-0301:导入文件下载失败 %s", taskCode);
    Validate.isTrue(localFile.getTemplateFile() != null, "ERR230904-0302:导入模板下载失败 %s", taskCode);
    return localFile;
  }
  
  /**
   * file对象转换为ExeclFileVo对象
   * @param file
   * @return
   */
  private ExcelFileVo convertExcelFileVo(File file) {
    ExcelFileVo vo = new ExcelFileVo();
    vo.setFileName(file.getName());
    vo.setParentPath(file.getParent());
    try {
      vo.setPath(file.getCanonicalPath());
    } catch (IOException e) {
      vo.setPath(file.getPath());
    }
    return vo;
  }

  /**
   * 根据导入任务获取公共参数信息
   *
   * @param task
   * @return
   */
  private TaskGlobalParamsVo findParams(ImportTask task) {
    Validate.notNull(task, "导入任务信息不存在");
    final TaskGlobalParamsVo vo = new TaskGlobalParamsVo();
    vo.setAppCode(task.getAppCode());
    vo.setApplicationName(task.getApplicationName());
    vo.setCreateAccount(task.getCreateAccount());
    vo.setCreateAccountName(task.getCreateName());
    vo.setTaskCreateTime(task.getCreateTime());
    vo.setTenantCode(task.getTenantCode());
    vo.setTaskCode(task.getTaskCode());
    return vo;
  }

  /**
   * 任务处理失败
   *
   * @param taskCode
   * @param msg
   */
  private void handlerTaskFailed(String taskCode, String msg, String account) {
    ImportTaskUpdateModelVo vo = new ImportTaskUpdateModelVo();
    vo.setTaskCode(taskCode);
    vo.setExecStatus(ExecStatusEnum.FAILED.getDictCode());
    vo.setRemark(msg);
    this.importTaskThreadLocalServImpl.updateByImportTaskUpdateModelVo(vo);
    // 发送任务执行失败处理消息
    this.importSendProcessMsgBean.sendFailedProcessMsg(taskCode, msg, account);
  }

  /**
   * 执行回调操作
   *
   * @param taskCode 导入任务编码
   */
  private void callBack(String taskCode) {
    if (CollectionUtils.isEmpty(notifyEventListeners)) {
      return;
    }
    boolean locked = false;
    String lockKey = CharSequenceUtil.format(ImportExportConstant.IE_IMPORT_CALL_BACK_LOCK_PREFIX_FORMAT, taskCode);
    try {
      locked = this.redisMutexService.tryLock(lockKey, TimeUnit.SECONDS, 5);
      if (locked) {
        long taskExecutionAfterSleep = importExportProperties.getTaskExecutionIntervalSleep();
        // 应对主从数据不同步问题
        if (taskExecutionAfterSleep > 0) {
          ThreadUtil.sleep(taskExecutionAfterSleep);
          log.debug("******休眠释放数据库压力*********");
        }
        ImportTask importTask = this.importTaskRepository.findByTaskCode(taskCode);
        if (Objects.isNull(importTask)) {
          log.warn("执行失败回调未能找到对应导入任务信息[{}]", taskCode);
          return;
        }
        CallbackStatusEnum callbackStatusEnum = CallbackStatusEnum.getByDictCode(importTask.getCallBackStatus());
        if (Objects.nonNull(callbackStatusEnum)) {
          log.warn("导入任务[{}],跳过本次回调执行操作[{}]", callbackStatusEnum.getValue(), importTask.getExecStatus());
          return;
        }
        ImportTaskEventVo importTaskEventVo = this.nebulaToolkitService.copyObjectByWhiteList(
            importTask, ImportTaskEventVo.class, HashSet.class, ArrayList.class);
        CallbackStatusEnum callbackStatus = null;
        try {
          if (ExecStatusEnum.FAILED.getDictCode().equals(importTask.getExecStatus())
              || (!ImportDataStatusEnum.IMPORT_SUCCESS.getDictCode().equals(importTask.getImportDataStatus())
                && !ImportDataStatusEnum.IMPORT_FAILED_SECTION.getDictCode().equals(importTask.getImportDataStatus()))) {
            log.info("导入任务[{}]当前任务状态[{}]数据导入状态[{}],执行失败回调操作", taskCode, importTask.getExecStatus(), importTask.getImportDataStatus());
            notifyEventListeners.forEach(eventListener -> eventListener.onFail(importTaskEventVo));
            callbackStatus = CallbackStatusEnum.HAVE_CALL_BACK_FAIL;
          } else if (ExecStatusEnum.FINISH.getDictCode().equals(importTask.getExecStatus())
              && ImportDataStatusEnum.IMPORT_SUCCESS.getDictCode().equals(importTask.getImportDataStatus())) {
            log.info("导入任务[{}]当前任务状态[{}]数据导入状态[{}],执行成功回调操作", taskCode, importTask.getExecStatus(), importTask.getImportDataStatus());
            notifyEventListeners.forEach(eventListener -> eventListener.onSuccess(importTaskEventVo));
            callbackStatus = CallbackStatusEnum.HAVE_CALL_BACK_SUCCESS;
          } else if (ExecStatusEnum.FINISH.getDictCode().equals(importTask.getExecStatus())
              && ImportDataStatusEnum.IMPORT_FAILED_SECTION.getDictCode().equals(importTask.getImportDataStatus())) {
            log.info("导入任务[{}]当前任务状态[{}]数据导入状态[{}],执行部分成功回调操作", taskCode, importTask.getExecStatus(), importTask.getImportDataStatus());
            notifyEventListeners.forEach(eventListener -> eventListener.onPartialSuccess(importTaskEventVo));
            callbackStatus = CallbackStatusEnum.HAVE_CALL_BACK_PARTIAL_SUCCESS;
          } else {
            log.info("导入任务[{}]当前任务状态[{}]数据导入状态[{}],没有匹配的回调操作", taskCode, importTask.getExecStatus(), importTask.getImportDataStatus());
          }
        } catch (Exception e) {
          log.error("导入任务执行回调发生异常", e);
        }
        if (Objects.nonNull(callbackStatus)) {
          this.importTaskRepository.updateCallBackStatusByIds(Lists.newArrayList(importTask.getId()), callbackStatus);
        }
      }
    } catch (Exception e) {
      log.error("导入任务执行回调发生异常", e);
    } finally {
      if (locked) {
        redisMutexService.unlock(lockKey);
      }
    }
  }
}
