package com.biz.crm.common.ie.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.ie.local.bean.ExportSendProcessMsgBean;
import com.biz.crm.common.ie.local.context.ExportTaskContext;
import com.biz.crm.common.ie.local.context.ExportTaskContextHolder;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.entity.ExportTaskDetail;
import com.biz.crm.common.ie.local.model.dto.ExportTaskDetailModelDto;
import com.biz.crm.common.ie.local.model.vo.FileModelVo;
import com.biz.crm.common.ie.local.repository.ExportTaskDetailRepository;
import com.biz.crm.common.ie.local.repository.ExportTaskRepository;
import com.biz.crm.common.ie.local.service.ExportTaskDetailService;
import com.biz.crm.common.ie.sdk.dto.ExportTaskQueueDto;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExportProcessEnum;
import com.biz.crm.common.ie.sdk.enums.LoadStatusEnum;
import com.biz.crm.common.ie.sdk.event.ImportExportTaskEventListener;
import com.biz.crm.common.ie.sdk.excel.process.AbstractEsParagraphExportProcess;
import com.biz.crm.common.ie.sdk.excel.process.ExportProcess;
import com.biz.crm.common.ie.sdk.service.ExportProcessService;
import com.biz.crm.common.ie.sdk.utils.IeJsonUtils;
import com.biz.crm.common.ie.sdk.utils.WebApiParamsTools;
import com.biz.crm.common.ie.sdk.vo.EsParagraphFieldRangeVo;
import com.biz.crm.common.ie.sdk.vo.ExportProcessMsgVo;
import com.biz.crm.common.ie.sdk.vo.ExportTaskProcessVo;
import com.bizunited.nebula.common.register.ElasticsearchQueryRegister;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.es.ElasticsearchQueryService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Comparators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.ComparatorUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import org.elasticsearch.action.admin.indices.settings.get.GetSettingsRequest;
import org.elasticsearch.action.admin.indices.settings.get.GetSettingsResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 导出任务子任务(ExportTaskDetail)表服务实现类
 *
 * @author sunx
 * @date 2022-05-18 16:31:07
 */
@Service("exportTaskDetailService")
@Slf4j
public class ExportTaskDetailServiceImpl implements ExportTaskDetailService {

  @Autowired(required = false)
  private ExportTaskDetailRepository exportTaskDetailRepository;
  @Autowired(required = false)
  private ExportProcessService exportProcessService;
  @Autowired(required = false)
  private FileHandleService fileHandleService;
  @Autowired(required = false)
  private ExportTaskRepository exportTaskRepository;
  @Autowired(required = false)
  private ExportSendProcessMsgBean exportSendProcessMsgBean;
  @Autowired(required = false)
  private List<ImportExportTaskEventListener> importExportTaskAuthRecordListeners;
  @Autowired(required = false)
  private ElasticsearchQueryService elasticsearchQueryService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Override
  public List<ExportTaskDetail> findByTaskCodes(Set<String> taskCodeSet) {
    if (CollectionUtils.isEmpty(taskCodeSet)) {
      return Lists.newLinkedList();
    }
    return this.exportTaskDetailRepository.findByTaskCodes(taskCodeSet);
  }

  @Override
  @Transactional
  public void create(List<ExportTaskDetail> list, String taskCode) {
    Validate.notBlank(taskCode, "任务编码不能为空");
    this.exportTaskDetailRepository.deleteByTaskCode(taskCode);
    for (ExportTaskDetail item : list) {
      item.setId(null);
      item.setTaskCode(taskCode);
      item.setTenantCode(TenantUtils.getTenantCode());
      item.setExecStatus(ExecStatusEnum.DEFAULT.getDictCode());
      item.setLoadStatus(LoadStatusEnum.NO.getDictCode());
    }
    this.exportTaskDetailRepository.saveBatch(list);
  }

  @Override
  public List<ExportTaskDetail> findByExportTaskDetailModelDto(ExportTaskDetailModelDto dto) {
    if (Objects.isNull(dto)) {
      return Lists.newLinkedList();
    }
    return this.exportTaskDetailRepository.findByExportTaskDetailModelDto(dto);
  }

  @Override
  @Transactional
  public void updateLocalStatus(Set<String> detailCodeSet, String loadStatus) {
    Validate.isTrue(!CollectionUtils.isEmpty(detailCodeSet), "导出子任务编码不能为空");
    Validate.notEmpty(loadStatus, "加载状态不能为空");
    this.exportTaskDetailRepository.updateLocalStatus(detailCodeSet, loadStatus);
  }

  @Override
  public ExportTaskDetail findByDetailCode(String detailCode) {
    if (StringUtils.isBlank(detailCode)) {
      return null;
    }
    ExportTaskDetail detail = this.exportTaskDetailRepository.findByDetailCode(detailCode);
    OrdinaryFileVo fileVo = this.fileHandleService.findById(detail.getFileCode());
    if (fileVo == null) {
      return detail;
    }
    FileModelVo fileModelVo = new FileModelVo();
    fileModelVo.setFileCode(fileVo.getId());
    fileModelVo.setFileName(fileVo.getOriginalFileName());
    detail.setFile(fileModelVo);
    return detail;
  }

  @Override
  public ExportTaskDetail findDetailByDetailCode(String detailCode) {
    if (StringUtils.isBlank(detailCode)) {
      return null;
    }
    final ExportTaskDetail detail = this.findByDetailCode(detailCode);
    if (Objects.isNull(detail)) {
      return null;
    }
    if (StringUtils.isBlank(detail.getFileCode())) {
      return detail;
    }

    OrdinaryFileVo fileVo = this.fileHandleService.findById(detail.getFileCode());
    if (fileVo == null) {
      return detail;
    }
    FileModelVo fileModelVo = new FileModelVo();
    fileModelVo.setFileCode(fileVo.getId());
    fileModelVo.setFileName(fileVo.getOriginalFileName());
    detail.setFile(fileModelVo);
    return detail;
  }

  @Override
  @Transactional
  public List<ExportTaskDetail> saveExportTaskAndCreateExportTaskDetail(ExportTask exportTask) {
    final String businessCode = exportTask.getBusinessCode();
    final String taskCode = exportTask.getTaskCode();
    final ExportProcess<?> exportProcess = this.exportProcessService.findExportProcess(businessCode);
    Map<String, Object> params = this.buildExportParamMap(exportTask, exportProcess);
    final Integer total;
    try {
      ExportTaskContext taskContext = new ExportTaskContext();
      taskContext.setTaskParamsMap(params);
      ExportTaskProcessVo taskProcessVo = nebulaToolkitService.copyObjectByWhiteList(exportTask, ExportTaskProcessVo.class, HashSet.class, ArrayList.class);
      taskContext.setTaskProcessVo(taskProcessVo);
      ExportTaskContextHolder.setContext(taskContext);
      total = exportProcess.getTotal(params);
    } finally {
      ExportTaskContextHolder.clearContext();
    }
    exportTask.setTotal(total);
    // 此处开始处理列表工具的导出还是二开团队自己定义的导出方式
    List<ExportTaskDetail> list = this.exportExcel(exportProcess, exportTask); // 二开团队自己定义的导出
    if (exportProcess instanceof AbstractEsParagraphExportProcess) {
      exportTask.setExecStatus(ExecStatusEnum.NEED_PARAGRAPH_EXPORT_TASK_DETAIL.getDictCode());
      list.forEach(taskDetail -> taskDetail.setExecStatus(ExecStatusEnum.NEED_PARAGRAPH_EXPORT_TASK_DETAIL.getDictCode()));
    }
    this.exportTaskRepository.saveOrUpdate(exportTask);
    this.exportTaskDetailRepository.deleteByTaskCode(taskCode);
    for (ExportTaskDetail item : list) {
      item.setId(null);
      item.setTaskCode(taskCode);
      item.setTenantCode(TenantUtils.getTenantCode());
      item.setExecStatus(StringUtils.isNotBlank(item.getExecStatus()) ? item.getExecStatus() : ExecStatusEnum.DEFAULT.getDictCode());
      item.setLoadStatus(LoadStatusEnum.NO.getDictCode());
    }
    this.exportTaskDetailRepository.saveBatch(list);
    // ====== 触发监听
    if (!CollectionUtils.isEmpty(this.importExportTaskAuthRecordListeners)) {
      for (ImportExportTaskEventListener listener : this.importExportTaskAuthRecordListeners) {
        listener.onTaskCreate(true, exportTask.getTaskCode(), exportTask.getAppCode(), exportTask.getTenantCode(),
            exportTask.getApplicationName(), exportTask.getCreateAccount());
      }
    }
    return list;
  }

  public Map<String, Object> buildExportParamMap(ExportTask exportTask,
      ExportProcess<?> exportProcess) {
    final String parametersJson = exportTask.getParametersJson();
    Map<String,Object> params = new HashMap<>();
    if (StringUtils.isNotBlank(parametersJson)){
      Map<String, Object> paramMap = IeJsonUtils.fromJson(parametersJson, HashMap.class);
      // 为导出请求设定各种页面传参信息（多数是客户的查询条件）
      params = this.findParams(exportProcess, paramMap);
    }
    params.put(WebApiParamsTools.CODE_PARAMETER_WEB_URL, exportTask.getWebApiUrl());
    return params;
  }

  @Override
  @Transactional
  public List<EsParagraphFieldRangeVo> execExportTaskDetailEsParagraph(ExportTask exportTask
      , ExportTaskDetail exportTaskDetail, List<EsParagraphFieldRangeVo> lastFieldRanges) {
    AbstractEsParagraphExportProcess<?> exportProcess = (AbstractEsParagraphExportProcess<?>) this.exportProcessService.findExportProcess(exportTask.getBusinessCode());
    Class<? extends ElasticsearchQueryRegister> registerClass = exportProcess.getElasticsearchQueryRegister();
    ElasticsearchQueryRegister queryRegister = this.elasticsearchQueryService.findRegisterByClass(registerClass);
    Validate.notNull(queryRegister, "未找到对应的ES查询注册器实现");
    Map<String,Object> params = this.buildExportParamMap(exportTask, exportProcess);
    SearchSourceBuilder sourceBuilder = queryRegister.buildSearch(params);
    List<Pair<String, SortOrder>> sortOrderList = this.buildEsIndexSort(queryRegister, sourceBuilder);
    // 重新构建分段排序
    sourceBuilder.sorts().clear();
    sortOrderList.forEach(orderPair -> sourceBuilder.sort(orderPair.getLeft(), orderPair.getRight()));
    int detailTotal = exportTaskDetail.getPageSize();
    int queryPageSize = exportProcess.getQueryPageSize();
    List<List<EsParagraphFieldRangeVo>> fieldRanges = Lists.newArrayList();
    long startTime = System.currentTimeMillis();
    log.debug("******子任务[{}]开始进行ES数据分段,总数据量[{}],查询分页大小[{}]*********", exportTaskDetail.getDetailCode(), detailTotal, queryPageSize);
    while (detailTotal > 0) {
      queryPageSize = Math.min(queryPageSize, detailTotal);
      if (!CollectionUtils.isEmpty(lastFieldRanges)) {
        sourceBuilder.searchAfter(lastFieldRanges.stream().map(EsParagraphFieldRangeVo::getEnd).toArray());
      }
      sourceBuilder.from(0).size(queryPageSize);
      SearchResponse searchResponse = this.elasticsearchQueryService.queryWithRequest(
          new SearchRequest(queryRegister.getIndexName()).source(sourceBuilder));
      List<Map> mapList = this.elasticsearchQueryService.convertResponse(searchResponse, Map.class);
      Validate.isTrue(!CollectionUtils.isEmpty(mapList), "ES查询数据匹配出错");

      // 封装当前分段的最后一个search after的值(对象中的start和end不代表真正的起始值,只用于标识改段数据的第一条数据和最后一条数据对应的排序字段的数值)
      lastFieldRanges.clear();
      for (Pair<String, SortOrder> orderPair : sortOrderList) {
        EsParagraphFieldRangeVo fieldRange = new EsParagraphFieldRangeVo();
        fieldRange.setFieldName(orderPair.getLeft());
        fieldRange.setSort(orderPair.getRight().toString());
        fieldRange.setStart(mapList.get(0).get(fieldRange.getFieldName()));
        fieldRange.setEnd(mapList.get(mapList.size() - 1).get(fieldRange.getFieldName()));
        lastFieldRanges.add(fieldRange);
      }

      // 封装数据查询时的排序字段范围(对象中的start和end代表排序字段在该段的最大值和最小值)
      List<EsParagraphFieldRangeVo> currentRanges = Lists.newArrayList();
      sortOrderList.forEach(orderPair -> {
        EsParagraphFieldRangeVo fieldRange = new EsParagraphFieldRangeVo();
        fieldRange.setFieldName(orderPair.getLeft());
        fieldRange.setSort(orderPair.getRight().toString());
        mapList.sort((o1, o2) -> {
          Object object1 = o1.get(fieldRange.getFieldName());
          Object object2 = o2.get(fieldRange.getFieldName());
          if (Objects.isNull(object1) && Objects.isNull(object2)) {
            return 0;
          } else if (Objects.isNull(object1)) {
            return -1;
          } else if (Objects.isNull(object2)) {
            return 1;
          }
          Validate.isTrue(object1 instanceof Comparable && object2 instanceof Comparable, "排序字段对应java类型无法比较大小");
          return ((Comparable) object1).compareTo(object2);
        });
        fieldRange.setStart(mapList.get(0).get(fieldRange.getFieldName()));
        fieldRange.setEnd(mapList.get(mapList.size() - 1).get(fieldRange.getFieldName()));
        currentRanges.add(fieldRange);
      });
      fieldRanges.add(currentRanges);
      detailTotal = detailTotal - queryPageSize;
    }
    Validate.isTrue(!CollectionUtils.isEmpty(fieldRanges), "ES分段数据出错");
    ExportTaskDetail updateTaskDetail = new ExportTaskDetail();
    updateTaskDetail.setId(exportTaskDetail.getId());
    updateTaskDetail.setParagraphRanges(JSONArray.toJSONString(fieldRanges));
    updateTaskDetail.setExecStatus(ExecStatusEnum.DEFAULT.getDictCode());
    log.debug("******子任务[{}]结束进行ES数据分段,总耗时[{}]*********", exportTaskDetail.getDetailCode(), System.currentTimeMillis() - startTime);
    this.exportTaskDetailRepository.updateById(updateTaskDetail);
    return lastFieldRanges;
  }

  private List<Pair<String, SortOrder>> buildEsIndexSort(ElasticsearchQueryRegister queryRegister, SearchSourceBuilder sourceBuilder) {
    List<Pair<String, SortOrder>> resultList = Lists.newArrayList();
    GetSettingsRequest settingsRequest = new GetSettingsRequest().indices(queryRegister.getIndexName());
    GetSettingsResponse settingsResponse = this.elasticsearchQueryService.queryWithSetting(settingsRequest);
    List<SortBuilder<?>> sorts = sourceBuilder.sorts();
    if (!CollectionUtils.isEmpty(sorts)) {
      for (SortBuilder<?> sortBuilder : sorts) {
        FieldSortBuilder fieldSortBuilder = (FieldSortBuilder) sortBuilder;
        if (Objects.isNull(fieldSortBuilder) || StringUtils.isBlank(fieldSortBuilder.getFieldName()) || Objects.isNull(fieldSortBuilder.order())) {
          continue;
        }
        resultList.add(Pair.of(fieldSortBuilder.getFieldName(), fieldSortBuilder.order()));
      }
    }
    List<String> filedNames = resultList.stream().map(Pair::getLeft).collect(Collectors.toList());
    Settings settings = settingsResponse.getIndexToSettings().get(queryRegister.getIndexName());
    if (Objects.nonNull(settings)) {
      String settingsStr = settings.toString();
      JSONObject indexSortJson = JSON.parseObject(settingsStr);
      JSONArray fieldArray = indexSortJson.getJSONArray("index.sort.field");
      JSONArray orderArray = indexSortJson.getJSONArray("index.sort.order");
      if (!CollectionUtils.isEmpty(fieldArray) && !CollectionUtils.isEmpty(orderArray) && fieldArray.size() == orderArray.size()) {
        for (Object filedObject : fieldArray) {
          // 如果原有查询里面就有setting里面的排序字段,则以原有的排序字段顺序为准
          if (!filedNames.contains(filedObject.toString())) {
            Object orderObject = orderArray.get(fieldArray.indexOf(filedObject));
            resultList.add(Pair.of(filedObject.toString(), SortOrder.fromString(orderObject.toString())));
          }
        }
      }
    }
    if (!filedNames.contains("id")) {
      resultList.add(Pair.of("id", SortOrder.DESC));
    }
    log.info("获取ES分段排序字段信息:{}", JSON.toJSONString(resultList));
    return resultList;
  }

  /**
   * 调用二开团队自己定义的导出getTotal方法，获取总数后进行数据拆分
   * 
   * @param exportProcess
   * @param exportTask
   */
  private List<ExportTaskDetail> exportExcel(ExportProcess<?> exportProcess, ExportTask exportTask) {
    Validate.isTrue(exportTask.getTotal().compareTo(0) > 0, "未查到匹配数据无需执行导出");
    // 对导出请求进行任务拆分
    final Integer pageSize = exportProcess.getPageSize();
    final Integer total = exportTask.getTotal();
    List<ExportTaskDetail> list = this.findDetail(exportTask, total, pageSize);
    

    // 推送当前任务编码
    ExportProcessMsgVo msgVo = new ExportProcessMsgVo();
    msgVo.setTaskCode(exportTask.getTaskCode());
    msgVo.setMainFlag(true);
    msgVo.setExecStatus(ExecStatusEnum.DEFAULT.getDictCode());
    msgVo.setProcessType(ExportProcessEnum.START.getCode());
    msgVo.setRemark(ExportProcessEnum.WAIT.getFormat());
    msgVo.setAccount(exportTask.getCreateAccount());
    msgVo.setFileName(exportTask.getFileName());
    this.exportSendProcessMsgBean.sendMsgByAll(msgVo);
    // 202211修正文件名--end

    // 补充全局导出排队数据
    ExportTaskQueueDto exportTaskQueueDto = new ExportTaskQueueDto();
    exportTaskQueueDto.setTaskCode(exportTask.getTaskCode());
    exportTaskQueueDto.setAccount(exportTask.getCreateAccount());
    exportTaskQueueDto.setMainFlag(true);
    exportTaskQueueDto.setTotalSize(total);
    // 当前任务加入队列后 重新推送消息
    exportSendProcessMsgBean.sendTaskProcessQueueMsg(exportTaskQueueDto, 0);

    return list;
  }

  /**
   * 任务拆分
   *
   * @param exportTask
   * @param total
   * @param pageSize 一个sheet
   * @return
   */
  private List<ExportTaskDetail> findDetail(ExportTask exportTask, Integer total, Integer pageSize) {
    Validate.isTrue(pageSize.compareTo(0) > 0, "处理器子任务拆分size需大于0");
    List<ExportTaskDetail> list = Lists.newArrayList();
    // 整除的时候需要减1,防止多出来一个分页
    final Integer page = total % pageSize == 0 ? total / pageSize - 1 : total / pageSize;
    final Integer lastPageSize = total % pageSize == 0 ? pageSize : total % pageSize;
    for (int i = 0; i <= page; i++) {
      final ExportTaskDetail cur = new ExportTaskDetail();
      cur.setPageSize((i == page) ? lastPageSize : pageSize);// 最后一页实际剩余页数，否则就是最大条数
      cur.setPageNo(i);
      cur.setTaskCode(exportTask.getTaskCode());
      cur.setAppCode(exportTask.getAppCode());
      cur.setApplicationName(exportTask.getApplicationName());
      cur.setDetailCode(exportTask.getTaskCode() + "_" + cur.getPageNo());
      cur.setMarsListCode(exportTask.getMarsListCode());
      list.add(cur);
    }
    // 屏蔽list
    // exportTask.setTotal(total);
    // exportTask.setList(list);
    return list;
  }

  private Map<String, Object> findParams(ExportProcess<?> exportProcess, Map<String, Object> paramMap) {
    if (Objects.isNull(exportProcess)) {
      return Maps.newHashMap();
    }
    Map<String, Object> map = exportProcess.getGlobalParams();
    if (map == null) {
      map = Maps.newHashMap();
    }
    map.putAll(paramMap);
    return map;
  }

}
