package com.biz.crm.common.ie.local.listener;

import cn.hutool.json.JSONUtil;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.enums.RowTypeEnum;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.metadata.data.ReadCellData;
import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.lang3.Validate;

/**
 * 该监听器用于统计Excel行数的时候跳过带空格的行数
 */
public class EasyExcelListener extends AnalysisEventListener<Object> {

    /**
     * 是否跳过空白行 true-跳过 false-不跳过
     */
    private boolean whetherSkipBlankLines;

    /**
     * 导入限制行数
     */
    private Integer limitRowNum;

    /**
     * 不验证表头
     */
    public EasyExcelListener() {
        this.whetherSkipBlankLines = false;
        this.limitRowNum = ImportExportConstant.IE_IMPORT_LIMIT_ROW_NUM;
    }

    /**
     * 不验证表头+是否跳过空白行
     *
     * @param whetherSkipBlankLines 是否跳过空白行
     */
    public EasyExcelListener(boolean whetherSkipBlankLines) {
        this.whetherSkipBlankLines = whetherSkipBlankLines;
        this.limitRowNum = ImportExportConstant.IE_IMPORT_LIMIT_ROW_NUM;
    }

    /**
     * 不验证表头+是否跳过空白行
     *
     * @param whetherSkipBlankLines 是否跳过空白行
     * @param limitRowNum 导入限制行数
     */
    public EasyExcelListener(boolean whetherSkipBlankLines, Integer limitRowNum) {
        this.whetherSkipBlankLines = whetherSkipBlankLines;
        this.limitRowNum = limitRowNum;
    }

    @Override
    public void invokeHead(Map<Integer, ReadCellData<?>> headMap, AnalysisContext context) {
        super.invokeHead(headMap, context);
        // 获取总行数
        int totalRowNumber = context.readSheetHolder().getApproximateTotalRowNumber() - context.readSheetHolder().getHeadRowNumber();
        Validate.isTrue(Objects.isNull(limitRowNum) || limitRowNum >= totalRowNumber, "导入失败, 导入数据不能超过%s行", limitRowNum);
    }

    /**
     * 创建list集合封装最终的数据
     */
    List<Object> list = new ArrayList<>();

    /**
     * 读取excle内容
     *
     * @param t
     * @param analysisContext
     */
    @Override
    public void invoke(Object t, AnalysisContext analysisContext) {
        analysisContext.readWorkbookHolder().setIgnoreEmptyRow(this.whetherSkipBlankLines);
        // 验证空数据
        if (whetherSkipBlankLines && ObjectUtils.checkObjAllFieldsIsNull(t)) {
            return;
        }
        // 将读取到的数据输出到控制台
        list.add(t);
    }

    /**
     * 读取excel表头信息
     *
     * @param headMap
     * @param context
     */
    @Override
    public void invokeHeadMap(Map<Integer, String> headMap, AnalysisContext context) {
    }

    /**
     * 读取完成后执行
     *
     * @param analysisContext
     */
    @Override
    public void doAfterAllAnalysed(AnalysisContext analysisContext) {
    }

    /**
     * 获取表格内容
     *
     * @param obj 需要转化的实体
     * @param <T>
     * @return
     */
    public <T> List<T> getList(Class<T> obj) {
        String jsonObj = JSONUtil.toJsonStr(list);
        return JSONUtil.toList(jsonObj, obj);
    }

    @Override
    public boolean hasNext(AnalysisContext analysisContext) {
        if (RowTypeEnum.EMPTY.equals(analysisContext.readRowHolder().getRowType())) {
            doAfterAllAnalysed(analysisContext);
            return true;
        }
        return super.hasNext(analysisContext);
    }

}

