package com.biz.crm.common.ie.local.service.spring.rest.impl;

import com.biz.crm.common.ie.local.service.spring.rest.IMethodParameterTypeParse;
import com.biz.crm.common.ie.sdk.dto.data.domain.PageRequestBo;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.core.MethodParameter;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.web.PageableDefault;
import org.springframework.stereotype.Component;

/**
 * @title MethodParameterTypeParsePageable
 * @date 2023/10/10 13:38
 * @author Kevin
 * @description pageable参数类型
 */
@Slf4j
@Component
public class MethodParameterTypeParsePageable implements IMethodParameterTypeParse<Pageable> {

  @Override
  public boolean supportsParameter(MethodParameter methodParameter) {
    Class<?> parameterType = methodParameter.getParameterType();
    return parameterType == Pageable.class;
  }

  @Override
  public Pageable resolveArgument(MethodParameter methodParameter,
      Map<String, Object> requestParamMap) throws IllegalArgumentException {
    final String parameterName = methodParameter.getParameterName();
    if (StringUtils.isNotBlank(parameterName)){
      Object o = requestParamMap.get(parameterName);
      if (o instanceof Pageable){
        return (Pageable) o;
      }
    }
    Pageable defaultPageable = null;
    final PageableDefault parameterAnnotation = methodParameter.getParameterAnnotation(
        PageableDefault.class);
    if (parameterAnnotation != null){
      final String [] sortString  = parameterAnnotation.sort();
      Sort sort;
      if (sortString.length == 0){
        sort = Sort.unsorted();
      }else {
       final Direction direction = parameterAnnotation.direction();
        sort = Sort.by(direction,sortString);
      }
      int page = parameterAnnotation.page();
      int size = parameterAnnotation.size();
      int value = parameterAnnotation.value();
      if (value != 10 && size == 10){
        //value设置了值  size使用的是默认值
        size = value;
      }
      defaultPageable = new PageRequestBo(page,size,sort);
    }
    Pageable pageable = this.getPageable(requestParamMap,defaultPageable);
    return pageable;
  }

  /**
   * 构建pageable对象
   * @param requestParamMap
   * @param defaultPageable
   * @return
   */
  private Pageable getPageable(Map<String, Object> requestParamMap, Pageable defaultPageable) {
    final Object varPage = requestParamMap.get("page");
    int page = NumberUtils.toInt(varPage == null ? null : "" + varPage,
        defaultPageable.getPageNumber());
    final Object varSize = requestParamMap.get("size");
    int size = NumberUtils.toInt(varSize == null ? null : "" + varSize, defaultPageable.getPageSize());
    Sort sort = null;
    try{
     final Object varSort = requestParamMap.get("sort");
     if (varSort != null){
       String sortParam = "" + varSort;
       final String[] sortSplit = sortParam.split(",");
       switch (sortSplit.length){
         case 1:
           sort = Sort.by(sortParam);
           break;
         case 2:
           Direction direction = Direction.fromString(sortSplit[1]);
           sort = Sort.by(direction,sortSplit[0]);
           break;
         default:
           sort = defaultPageable.getSort();
       }
     }
    }catch (Exception e){
      sort = defaultPageable.getSort();
    }
    return this.getPageable(page,size,sort);
  }

  /**
   * 获取pageable对象
   * @param page 页码
   * @param pageSize 页大小
   * @param sort 排序规则 DESC / ASC
   * @return
   */
  private Pageable getPageable(int page, int pageSize, Sort sort) {
    if (sort == null){
      sort = Sort.unsorted();
    }
    PageRequestBo pageRequestBo = new PageRequestBo(page,pageSize,sort);
    return pageRequestBo;
  }
}
