package com.biz.crm.common.ie.local.service.strategy;

import com.alibaba.excel.write.handler.RowWriteHandler;
import com.alibaba.excel.write.metadata.holder.WriteSheetHolder;
import com.alibaba.excel.write.metadata.holder.WriteTableHolder;
import com.biz.crm.common.ie.local.model.ExportExcelHeadModel;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.imageio.ImageIO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.ClientAnchor;
import org.apache.poi.ss.usermodel.Drawing;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.util.Units;
import org.springframework.core.io.ResourceLoader;

/**
 * @author ning.zhang
 * @description 动态图片处理器
 * @date 2025/3/26
 */
@Slf4j
public class DynamicImageWriteHandler implements RowWriteHandler {

  // 每张图片的显示大小,正方形（像素）
  private final static int SINGLE_IMAGE_SIZE = 100;
  // 图片与格子的边距
  private final static int SINGLE_IMAGE_PADDING = 5;
  // 每行最多3张图片
  private final static int MAX_IMAGES_PER_ROW = 3;

  private static byte[] LOAD_FAIL_IMAGE_BYTES = null;

  static {
    // 静态初始化块
    try (InputStream inputStream = ResourceLoader.class.getClassLoader().getResourceAsStream("load_fail_image.png")) {
      if (Objects.nonNull(inputStream)) {
        LOAD_FAIL_IMAGE_BYTES = IOUtils.toByteArray(inputStream);
      } else {
        log.error("加载导出默认失败图片失败");
      }
    } catch (Exception e) {
      log.error("加载导出默认失败图片异常", e);
    }
  }

  // 导出excel数据模版信息
  private final ExportExcelHeadModel excelHeadModel;
  // 文件服务类
  private final FileHandleService fileHandleService;
  // 图片列的最大列宽映射
  private final Map<Integer, Integer> imageColumnMaxWidthMap = Maps.newHashMap();
  public DynamicImageWriteHandler(ExportExcelHeadModel excelHeadModel, FileHandleService fileHandleService) {
    this.excelHeadModel = excelHeadModel;
    this.fileHandleService = fileHandleService;
  }

  @Override
  public void afterRowDispose(WriteSheetHolder writeSheetHolder, WriteTableHolder writeTableHolder,
      Row row, Integer relativeRowIndex, Boolean isHead) {
    // 表头或者没有需要处理的图片列,则跳过
    if (isHead || CollectionUtils.isEmpty(excelHeadModel.getImageDataIndexList())) {
      return;
    }
    // 当前行最大高度
    int maxRowHeight = SINGLE_IMAGE_SIZE;
    for (Integer imageDataIndex : excelHeadModel.getImageDataIndexList()) {
      Cell imageCell = row.getCell(imageDataIndex);
      // 如果不是字符串类型,则不处理
      if (!CellType.STRING.equals(imageCell.getCellType())) {
        continue;
      }
      String fileCodeStr = imageCell.getStringCellValue();
      // 如果单元格数据为空,则不处理
      if (StringUtils.isBlank(fileCodeStr)) {
        continue;
      }
      // 清除单元格原有数据和样式
      imageCell.setCellValue("");

      // 加载图片信息,如果未加载到图片则使用默认加载失败图片替代
      String[] fileCodes = fileCodeStr.split(",");
      List<OrdinaryFileVo> fileVoList = fileHandleService.findByIds(Lists.newArrayList(fileCodes));
      fileVoList = ObjectUtils.defaultIfNull(fileVoList, Lists.newArrayList());
      Map<String, OrdinaryFileVo> ordinaryFileVoMap = fileVoList.stream()
          .collect(Collectors.toMap(OrdinaryFileVo::getId, Function.identity(), (a, b) -> a));
      List<byte[]> images = Arrays.stream(fileCodes)
          .map(fileCode -> {
            byte[] bytes = null;
            OrdinaryFileVo ordinaryFileVo = ordinaryFileVoMap.get(fileCode);
            if (Objects.nonNull(ordinaryFileVo)) {
              bytes = fileHandleService.findContentByFilePathAndFileRename(ordinaryFileVo.getRelativeLocal(), ordinaryFileVo.getFileName());
            }
            return Objects.isNull(bytes) || bytes.length == 0 ? LOAD_FAIL_IMAGE_BYTES : bytes;
          }).collect(Collectors.toList());

      int magesPerRow = Math.min(images.size(), MAX_IMAGES_PER_ROW);
      // 计算需要的行数
      int rowsNeeded = (int) Math.ceil((double)images.size() / magesPerRow);

      // 设置单元格尺寸
      int totalWidthPx = SINGLE_IMAGE_SIZE * magesPerRow;
      int totalHeightPx = SINGLE_IMAGE_SIZE * rowsNeeded;

      // 记录并设置最大列宽
      int columnWidth = (int) (totalWidthPx * 256 / Units.DEFAULT_CHARACTER_WIDTH);
      int mapColumnWidth = Math.max(imageColumnMaxWidthMap.getOrDefault(imageDataIndex, columnWidth), columnWidth);
      imageColumnMaxWidthMap.put(imageDataIndex, mapColumnWidth);
      imageCell.getSheet().setColumnWidth(imageDataIndex, mapColumnWidth);

      maxRowHeight = Math.max(maxRowHeight, totalHeightPx);

      // 嵌入图片
      embedImages(imageDataIndex, writeSheetHolder.getSheet(), row, images, magesPerRow);

    }
    // 设置当前行的高度
    row.setHeightInPoints((float)(maxRowHeight * 72.0 / 96.0));
  }

  /**
   * 嵌入图片
   *
   * @param imageDataIndex 图片列索引位置
   * @param sheet Sheet
   * @param row 行
   * @param imageBytesList 插入的图片信息
   * @param magesPerRow 当前单元格行数
   */
  private void embedImages(int imageDataIndex, Sheet sheet, Row row, List<byte[]> imageBytesList, int magesPerRow) {
    for (int imageIndex = 0; imageIndex < imageBytesList.size(); imageIndex++) {
      byte[] imageBytes = imageBytesList.get(imageIndex);
      Drawing<?> drawing = sheet.createDrawingPatriarch();

      // 计算位置
      int gridRow = imageIndex / magesPerRow;
      int gridCol = imageIndex % magesPerRow;

      // 读取图片尺寸
      ByteArrayInputStream imageByteArrayInputStream = null;
      ByteArrayInputStream failImageByteArrayInputStream = null;
      try {
        // 如果遇到无法识别的图片则展示默认加载失败的图片
        imageByteArrayInputStream = new ByteArrayInputStream(imageBytes);

        BufferedImage bufferedImage = ImageIO.read(imageByteArrayInputStream);
        if (Objects.isNull(bufferedImage)) {
          imageBytes = LOAD_FAIL_IMAGE_BYTES;
          failImageByteArrayInputStream = new ByteArrayInputStream(imageBytes);
          bufferedImage = ImageIO.read(failImageByteArrayInputStream);
        }
        double aspectRatio = (double) bufferedImage.getWidth() / bufferedImage.getHeight();

        // 计算缩放尺寸
        double scaledWidth, scaledHeight;
        if (aspectRatio > 1) {
          // 留边距
          scaledWidth = SINGLE_IMAGE_SIZE - 2 * SINGLE_IMAGE_PADDING;
          scaledHeight = scaledWidth / aspectRatio;
        } else {
          scaledHeight = SINGLE_IMAGE_SIZE - 2 * SINGLE_IMAGE_PADDING;
          scaledWidth = scaledHeight * aspectRatio;
        }

        // 计算居中位置
        double offsetX = (SINGLE_IMAGE_SIZE - scaledWidth) / 2 + gridCol * SINGLE_IMAGE_SIZE;
        double offsetY = (SINGLE_IMAGE_SIZE - scaledHeight) / 2 + gridRow * SINGLE_IMAGE_SIZE;

        // 创建锚点
        Workbook workbook = sheet.getWorkbook();
        ClientAnchor anchor = workbook.getCreationHelper().createClientAnchor();
        anchor.setDx1((int) (offsetX * Units.EMU_PER_PIXEL));
        anchor.setDy1((int) (offsetY * Units.EMU_PER_PIXEL));
        anchor.setDx2((int) ((offsetX + scaledWidth) * Units.EMU_PER_PIXEL));
        anchor.setDy2((int) ((offsetY + scaledHeight) * Units.EMU_PER_PIXEL));
        anchor.setCol1(imageDataIndex);
        anchor.setRow1(row.getRowNum());
        anchor.setCol2(imageDataIndex);
        anchor.setRow2(row.getRowNum());

        // 插入图片
        int picIndex = workbook.addPicture(imageBytes, Workbook.PICTURE_TYPE_JPEG);
        drawing.createPicture(anchor, picIndex);
      } catch (Exception e) {
        log.error("图片绑定单元格失败...", e);
      } finally {
        try {
          if (Objects.nonNull(imageByteArrayInputStream)) {
            imageByteArrayInputStream.close();
          }
          if (Objects.nonNull(failImageByteArrayInputStream)) {
            failImageByteArrayInputStream.close();
          }
        } catch (Exception e) {
          log.error("释放资源失败...", e);
        }
      }
    }
  }
}