package com.biz.crm.common.ie.local.repository;

import com.baomidou.mybatisplus.extension.conditions.query.LambdaQueryChainWrapper;
import com.baomidou.mybatisplus.extension.conditions.update.LambdaUpdateChainWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.local.entity.ExportTask;
import com.biz.crm.common.ie.local.mapper.ExportTaskMapper;
import com.biz.crm.common.ie.local.model.dto.ExportTaskConditionModelDto;
import com.biz.crm.common.ie.local.model.dto.ExportTaskModelDto;
import com.biz.crm.common.ie.local.model.dto.ExportTaskProcessModelDto;
import com.biz.crm.common.ie.sdk.dto.ExportTaskPaginationDto;
import com.biz.crm.common.ie.sdk.enums.CallbackStatusEnum;
import com.biz.crm.common.ie.sdk.enums.ExecStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.datasource.configuration.MultiDataSourceProperties;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import cn.hutool.core.date.DateUtil;

/**
 * 导出任务(repository)
 *
 * @author sunx
 * @date 2022-05-18 16:31:05
 */
@Component
public class ExportTaskRepository extends ServiceImpl<ExportTaskMapper, ExportTask> {
  @Autowired
  private MultiDataSourceProperties multiDataSourceProperties;

  /**
   * 分页
   *
   * @param page 分页
   * @param dto 查询参数
   * @return
   */
  public Page<ExportTask> findByConditions(Page<ExportTask> page, ExportTaskPaginationDto dto) {
    if (multiDataSourceProperties.getSchemaForTenant()) {
      dto.setAppCode(TenantUtils.getAppCode());
    }
    return this.baseMapper.findByConditions(page, dto);
  }

  public ExportTask findByTaskCode(String taskCode) {
    LambdaQueryChainWrapper<ExportTask> lambdaQuery = this.lambdaQuery();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaQuery.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    return lambdaQuery
        .eq(ExportTask::getTaskCode, taskCode)
        .one();
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids 主键集合
   * @return
   */
  public List<ExportTask> findByIds(List<String> ids) {
    LambdaQueryChainWrapper<ExportTask> lambdaQuery = this.lambdaQuery();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaQuery.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    return lambdaQuery
        .in(ExportTask::getId, ids)
        .list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids 主键集合
   * @param enable 启禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    LambdaUpdateChainWrapper<ExportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    lambdaUpdate
        .in(ExportTask::getId, ids)
        .set(ExportTask::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 根据任务编码集合更新执行状态
   *
   * @param execStatusEnum 执行状态
   * @param taskCodes 任务编码集合
   */
  public void updateExecStatusByTaskCodes(ExecStatusEnum execStatusEnum, Set<String> taskCodes) {
    LambdaUpdateChainWrapper<ExportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    lambdaUpdate
        .in(ExportTask::getTaskCode, taskCodes)
        .set(ExportTask::getExecStatus, execStatusEnum.getDictCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids 主键集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    LambdaUpdateChainWrapper<ExportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    lambdaUpdate
        .in(ExportTask::getId, ids)
        .set(ExportTask::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 更新回调状态
   *
   * @param ids 主键集合
   * @param callbackStatusEnum 回调状态
   */
  public void updateCallBackStatusByIds(List<String> ids, CallbackStatusEnum callbackStatusEnum) {
    LambdaUpdateChainWrapper<ExportTask> lambdaUpdate = this.lambdaUpdate();
    if (multiDataSourceProperties.getSchemaForTenant()) {
      lambdaUpdate.eq(ExportTask::getAppCode, TenantUtils.getAppCode());
    }
    lambdaUpdate
        .in(ExportTask::getId, ids)
        .set(ExportTask::getCallBackStatus, callbackStatusEnum.getDictCode())
        .update();
  }

  /**
   * 任务结束执行更新任务状态信息
   *
   * @param dto 导出任务结束更新主任务信息model实体
   */
  public void updateByExportTaskProcessModelDto(ExportTaskProcessModelDto dto) {
    if (dto == null || StringUtils.isBlank(dto.getTaskCode())) {
      return;
    }
    ExportTask exportTask = this.findByTaskCode(dto.getTaskCode());
    exportTask.setExecStatus(dto.getExecStatus());
    if (StringUtils.isNotBlank(dto.getRemark())) {
      exportTask.setRemark(dto.getRemark());
    }
    if (multiDataSourceProperties.getSchemaForTenant()) {
      exportTask.setAppCode(TenantUtils.getAppCode());
    }
    this.updateById(exportTask);
  }

  public int getTaskCountByNoExe(ExportTask current) {
    if (multiDataSourceProperties.getSchemaForTenant()) {
      current.setAppCode(TenantUtils.getAppCode());
    }
    Date createTime = current.getCreateTime();
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(createTime);
    calendar.add(Calendar.DAY_OF_MONTH, -1);// 减去一天
    Date createTimeForBegin = calendar.getTime();
    int taskCountByNoExe = this.getBaseMapper().getTaskCountByNoExe(current, createTimeForBegin);
    return taskCountByNoExe;
  }

  public List<ExportTask> findExportTaskNoGetTotal(ExportTaskModelDto dto) {
    Calendar calendar = Calendar.getInstance();
    calendar.add(Calendar.DAY_OF_MONTH, -1);// 减去一天
    Date createTimeForBegin = calendar.getTime();
    return this.getBaseMapper().findExportTaskNoGetTotal(dto, createTimeForBegin);
  }

  public List<ExportTask> findByExportTaskConditionModelDto(ExportTaskConditionModelDto dto) {
    return this.getBaseMapper().findByExportTaskConditionModelDto(dto);
  }
}
