package com.biz.crm.common.ie.sdk.excel.process;

import com.biz.crm.common.ie.sdk.constant.ImportExportConstant;
import com.biz.crm.common.ie.sdk.enums.WriteErrorExcelModelEnums;
import com.biz.crm.common.ie.sdk.excel.vo.CrmExcelVo;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.text.CharSequenceUtil;

/**
 * 导入基座
 *
 * @author sunx
 * @date 2022/5/12
 */
public interface ImportProcess<T extends CrmExcelVo> {
  /**
   * 单次缓存的数量
   *
   * @return
   */
  default Integer getBatchCount() {
    return 1;
  }

  /**
   * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
   * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
   *
   * @return
   */
  default boolean importBeforeValidationFlag() {
    return false;
  }

  /**
   * 数据处理
   *
   * @param data 待处理的数据集合，k-流水号，v-excel解析后的对象
   * @param paramsVo 任务公共参数
   * @param params 导入任务自定义参数
   * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
   */
  Map<Integer, String> execute(
      LinkedHashMap<Integer, T> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params);

  /**
   * 数据校验
   *
   * @param data 待处理的数据集合，k-流水号，v-excel解析后的对象
   * @param paramsVo 任务公共参数
   * @param params 导入任务自定义参数
   * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
   */
  default Map<Integer, String> tryVerify(
      LinkedHashMap<Integer, T> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
    return null;
  }

  /**
   * 数据存储
   *
   * @param data 待处理的数据集合，k-流水号，v-excel解析后的对象
   * @param paramsVo 任务公共参数
   * @param params 导入任务自定义参数
   * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
   */
  default Map<Integer, String> tryConfirm(
      LinkedHashMap<Integer, T> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
    // 自行实现保存逻辑 若无 则使用旧逻辑
    return this.execute(data, paramsVo, params);
  }

  /**
   * 获取数据实体
   *
   * @return
   */
  Class<T> findCrmExcelVoClass();

  /**
   * 获取业务编码(一个业务场景多模板情况,不设置默认取模板编码)
   *
   * @return
   */
  default String getBusinessCode() {
    return CharSequenceUtil.EMPTY;
  }

  /**
   * 业务名称(一个业务场景多模板情况,不设置默认取模板名称)
   *
   * @return
   */
  default String getBusinessName() {
    return CharSequenceUtil.EMPTY;
  }

  /**
   * 获取业务对应的模板编码，全局唯一
   *
   * @return
   */
  String getTemplateCode();

  /**
   * 获取业务对应的模板描述
   *
   * @return
   */
  String getTemplateName();

  /**
   * 前端下拉显示顺序(升序)
   *
   * @return
   */
  default int order() {
    return 0;
  }

  /**
   * 获取用户自定义任务扩展参数（在新建导入任务的时候使用，如果实现会将返回数据存入任务数据库）
   *
   * @return
   */
  default Map<String, Object> getGlobalParams() {
    Map<String, Object> map = Maps.newHashMap();
    map.put(
        ImportExportConstant.IE_PARAMS_CREATE_TIME,
        DateUtil.format(new Date(), "yyyy-MM-dd HH:mm:ss"));
    return map;
  }

  /**
   * 写入错误Excel是全文件写入还是仅写入错误的
   */
  default WriteErrorExcelModelEnums getWriteErrorExcelModel() {
    return WriteErrorExcelModelEnums.ONLY_ERROR_ROW;
  }

  /**
   * 获取当前线程的导入错误记录
   */
  static final ThreadLocal<List<String>> errorListLocal = new ThreadLocal<>();

  /**
   * 获取当前线程的导入错误记录,如果为空，自动创建一个
   * 
   * @return
   * @see #errorListLocal
   */
  static List<String> getThreadErrorList() {
    List<String> list = errorListLocal.get();
    if (list == null) {
      list = new ArrayList<>();
      errorListLocal.set(list);
    }
    return list;
  }

  /**
   * 验证如果是true，则将异常消息缓存
   * 
   * @param isTrue isTrue的条件
   * @param errorMsg 如果不是空时的异常消息
   */
  default void validateIsTrue(boolean isTrue, String errorMsg) {
    if (false == isTrue) {
      List<String> errorList = getThreadErrorList();
      errorList.add(errorMsg);
    }
  }

  /**
   * 进行异常缓存检测，如果检测有缓存数据，则弹出异常信息，触发异常处理逻辑<br/>
   * 备注：可选操作，如果校验过程中不触发该动作，系统也会在处理结束后自动触发检测机制
   * 
   * @param separator 多个异常信息之间的分隔符
   * @throws IllegalArgumentException 如果发生异常，则弹出此异常
   */
  default void validateCheck(String separator) throws IllegalArgumentException {
    if (separator == null) {
      separator = " ";
    }
    String errorInfo = this.validateGetErrorInfo(separator);
    Validate.isTrue(errorInfo == null, errorInfo);
  }

  /**
   * 进行异常缓存检测，如果检测有缓存数据，则弹出异常信息，触发异常处理逻辑<br/>
   * 备注：可选操作，如果校验过程中不触发该动作，系统也会在处理结束后自动触发检测机制<br/>
   * <b>异常信息只能获取一次</b>，获取后即清空本线程的异常List对象
   * 
   * @param separator 多个异常信息之间的分隔符
   * @return 如果有异常信息则返回字符串，否则返回null
   * @see #validateIsTrue(boolean, String)
   */
  default String validateGetErrorInfo(String separator) {
    List<String> errorList = errorListLocal.get();
    if (!CollectionUtils.isEmpty(errorList)) {
      errorListLocal.remove();
      return StringUtils.join(errorList, separator);
    } else {
      return null;
    }
  }

  /**
   * 进行异常缓存检测，如果检测有缓存数据，则弹出异常信息，触发异常处理逻辑<br/>
   * 备注：可选操作，如果校验过程中不触发该动作，系统也会在处理结束后自动触发检测机制<br/>
   * <b>异常信息只能获取一次</b>，获取后即清空本线程的异常List对象<br/>
   * <b>多个异常信息之间的分隔符使用“ ”</b><br/>
   * 
   * @param separator 多个异常信息之间的分隔符
   * @return 如果有异常信息则返回字符串，否则返回null
   * @see #validateIsTrue(boolean, String)
   */
  default String validateGetErrorInfo() {
    return this.validateGetErrorInfo(" ");
  }

  /**
   * 进行异常缓存检测，如果检测有缓存数据，则弹出异常信息，触发异常处理逻辑<br/>
   * <b>多个异常信息之间的分隔符使用“ ”</b><br/>
   * 备注：可选操作，如果校验过程中不触发该动作，系统也会在处理结束后自动触发检测机制
   * 
   * @throws IllegalArgumentException 如果记录有异常信息，则抛出此异常
   */
  default void validateCheck() throws IllegalArgumentException {
    this.validateCheck(" ");
  }

  /**
   * 导入最大条数限制(超过该设置导入将会报错)
   *
   * @return 导入最大条数限制
   */
  default Integer getLimitRowNum() {
    return ImportExportConstant.IE_IMPORT_LIMIT_ROW_NUM;
  }
}
