package com.biz.crm.common.itextpdf.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.utils.MultipartFileUtil;
import com.biz.crm.common.itextpdf.local.model.HtmlMergePdfResultModel;
import com.biz.crm.dms.business.itextpdf.sdk.dto.ITextPdfHtmlMergeDto;
import com.biz.crm.dms.business.itextpdf.sdk.dto.WordTemplateDataVo;
import com.biz.crm.dms.business.itextpdf.sdk.service.ITextPdfVoService;
import com.biz.crm.dms.business.itextpdf.sdk.utils.MockMultipartFile;
import com.biz.crm.dms.business.itextpdf.sdk.utils.WordSwitchPdfUtil;
import com.biz.crm.dms.business.itextpdf.sdk.utils.WordUtil;
import com.bizunited.nebula.venus.sdk.dto.Base64UploadDto;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.itextpdf.html2pdf.ConverterProperties;
import com.itextpdf.html2pdf.HtmlConverter;
import com.itextpdf.html2pdf.resolver.font.DefaultFontProvider;
import com.itextpdf.io.font.PdfEncodings;
import com.itextpdf.kernel.font.PdfFont;
import com.itextpdf.kernel.font.PdfFontFactory;
import com.itextpdf.kernel.geom.PageSize;
import com.itextpdf.kernel.pdf.PdfDocument;
import com.itextpdf.kernel.pdf.PdfWriter;
import com.itextpdf.layout.Document;
import com.itextpdf.layout.element.AreaBreak;
import com.itextpdf.layout.element.IBlockElement;
import com.itextpdf.layout.element.IElement;
import com.itextpdf.layout.property.AreaBreakType;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;
import org.springframework.web.multipart.MultipartFile;

/**
 * @description: 文件转pdf
 * @author: rentao
 * @date: 2022/4/25 15:39
 */
@Slf4j
@Service
public class ITextPdfVoServiceImpl implements ITextPdfVoService {

  //subsystem不能有符合
  //@Value("${spring.application.name:}")
  @Value("${server.port:}")
  private String subsystem;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private FileHandleService fileHandleService;

  /**
   * file 转 pdf (直转)
   *
   * @param html
   */
  @Override
  public OrdinaryFileVo createPdfByFile(MultipartFile html) {
    Validate.notNull(html, "请选择文件！");
    OrdinaryFileVo uploadVo = null;
    try {
      uploadVo = this.venusFileUpload(html).get(0);
    } catch (IOException e) {
      log.error(e.getMessage(), e);
      Validate.isTrue(false, "上传错误文件失败");
    }
    return uploadVo;
  }

  /**
   * 该私有方法为venus文件上传接口参数封装
   */
  private List<OrdinaryFileVo> venusFileUpload(MultipartFile multipartFile) throws IOException {
    Base64UploadDto base64UploadDto = new Base64UploadDto();
    base64UploadDto.setCreator(loginUserService.findCurrentAccount());
    base64UploadDto.setFileNanmes(new String[]{multipartFile.getOriginalFilename()});
    base64UploadDto.setBase64Contents(new String[]{Base64Utils.encodeToString(multipartFile.getBytes())});
    List<OrdinaryFileVo> ordinaryFileVoList = this.fileHandleService.fileUpload(subsystem, base64UploadDto);
    Validate.notEmpty(ordinaryFileVoList, "文件上传失败！");
    return ordinaryFileVoList;
  }

  /**
   * html 转 pdf (直转)
   *
   * @param html
   * @return
   */
  @Override
  public OrdinaryFileVo createPdfByHtml(String html) {
    Validate.notNull(html, "入参不能为空！");
    try (InputStream inputStream = new ByteArrayInputStream(this.html2Pdf(html).toByteArray())){
      MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(inputStream, System.currentTimeMillis() + ".pdf");
      return this.venusFileUpload(multipartFile).get(0);
    } catch (IOException e) {
      log.error(e.getMessage(), e);
    }
    return null;
  }

  @Override
  public OrdinaryFileVo createPdfByHtmlMerge(ITextPdfHtmlMergeDto dto) throws IOException {
    Validate.notNull(dto, "入参不能为空！");
    Validate.notBlank(dto.getHeadHtml(), "首部html页面不能为空！");
    Validate.notBlank(dto.getTailHtml(), "尾部html页面不能为空！");
    Validate.notBlank(dto.getMergeType(), "合并方式不能为空！");
    switch (dto.getMergeType()) {
      case "1":
        HtmlMergePdfResultModel appendResultModel = this.htmlMergeWithAppend(dto.getHeadHtml(), dto.getTailHtml());
        try (ByteArrayOutputStream mergeOutputStream = appendResultModel.getMergeOutputStream();
            ByteArrayInputStream inputStream = new ByteArrayInputStream(mergeOutputStream.toByteArray())){
          MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(inputStream, System.currentTimeMillis() + ".pdf");
          return this.venusFileUpload(multipartFile).get(0);
        }
      case "2":
        HtmlMergePdfResultModel newPageMergeResultModel = this.htmlMergeWithNewPage(dto.getHeadHtml(), dto.getTailHtml());
        try (ByteArrayOutputStream mergeOutputStream = newPageMergeResultModel.getMergeOutputStream();
            ByteArrayInputStream inputStream = new ByteArrayInputStream(mergeOutputStream.toByteArray())){
          MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(inputStream, System.currentTimeMillis() + ".pdf");
          return this.venusFileUpload(multipartFile).get(0);
        }
      case "3":
        HtmlMergePdfResultModel appendResult = this.htmlMergeWithAppend(dto.getHeadHtml(), dto.getTailHtml());
        HtmlMergePdfResultModel newPageResult = this.htmlMergeWithNewPage(dto.getHeadHtml(), dto.getTailHtml());
        try (ByteArrayOutputStream appendOutputStream = appendResult.getMergeOutputStream();
            ByteArrayOutputStream newPageOutputStream = newPageResult.getMergeOutputStream();
            // 如果追加方式和新页面方式页数不相等, 则使用追加方式,相等则使用新页面方式
            ByteArrayInputStream inputStream = new ByteArrayInputStream(newPageResult.getPdfPages()
                > appendResult.getPdfPages() ? appendOutputStream.toByteArray() : newPageOutputStream.toByteArray())){
          MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(inputStream, System.currentTimeMillis() + ".pdf");
          return this.venusFileUpload(multipartFile).get(0);
        }
      default:
        throw new IllegalArgumentException("不支持的合并方式！");
    }
  }

  @Override
  public void reviewPdfByHtml(String htmlStr, HttpServletResponse response) {
    try {
      ByteArrayOutputStream stream = html2Pdf(htmlStr);
      ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(stream.toByteArray());
      OutputStream outputStream= response.getOutputStream();
      int count = 0;
      byte[] buffer = new byte[1024 * 1024];
      while ((count =byteArrayInputStream.read(buffer)) != -1){
        outputStream.write(buffer, 0,count);
      }
      response.setContentType( "application/pdf;charset=UTF-8");
      outputStream.flush();
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  @Override
  public OrdinaryFileVo createWordDocxToPdfFile(WordTemplateDataVo dataVo) {
    OrdinaryFileVo ordinaryFileVo = null;
    FileInputStream fileInputStream = null;
    MultipartFile multipartFile = null;
    try {
      //生成word
      String wordPath = WordUtil.createDocx(dataVo.getDataMap(), dataVo.getFileName(), dataVo.getTemplateName(), dataVo.getZipTemplateName());
      //签名图片
      OrdinaryFileVo signFileVo = this.fileHandleService.findById(dataVo.getSignFileCode());
      byte[] bytes = this.fileHandleService.findContentByFilePathAndFileRename(signFileVo.getRelativeLocal(), signFileVo.getFileName());
      String docxSginPath = WordUtil.sealInWord(wordPath, dataVo.getFileName(), bytes, dataVo.getTabText(),
          dataVo.getWidth(), dataVo.getHeight(), dataVo.getLeftOffset(), dataVo.getTopOffset(), dataVo.getBehin());
      //转换为PDF
      String docx2Pdf = WordSwitchPdfUtil.convertDocx2Pdf(docxSginPath, dataVo.getFileName());
      //上传文件
      File pdfFile = new File(docx2Pdf);
      fileInputStream = new FileInputStream(pdfFile);
      multipartFile = new MockMultipartFile(pdfFile.getName(), pdfFile.getName(), "application/octet-stream", fileInputStream);
      ordinaryFileVo = this.venusFileUpload(multipartFile).get(0);
      //删除目录
      WordUtil.deleteTemporaryFile(dataVo.getFileName());
    } catch (FileNotFoundException e) {
      Validate.isTrue(Boolean.FALSE, "文件生成失败！");
    } catch (IOException e) {
      Validate.isTrue(Boolean.FALSE, "IO异常！");
    } finally {
      if (fileInputStream != null) {
        try {
          fileInputStream.close();
        } catch (IOException e) {
          Validate.isTrue(Boolean.FALSE, "IO关闭异常！");
        }
      }
    }
    return ordinaryFileVo;
  }

  /**
   * @param html html页面字符串
   * @return
   * @throws FileNotFoundException
   * @throws IOException
   * @Description 将html转换为pdf文件
   */
  private ByteArrayOutputStream html2Pdf(String html) throws IOException {
    ConverterProperties props = new ConverterProperties();
    //设置字体
    this.setFont(props);
    //设置输出流
    ByteArrayOutputStream bao = new ByteArrayOutputStream();
    PdfWriter writer = new PdfWriter(bao);
    PdfDocument pdf = new PdfDocument(writer);
    pdf.setDefaultPageSize(PageSize.A4);
    //转成document进行操作
    Document document = HtmlConverter.convertToDocument(html, pdf, props);
    document.add(new AreaBreak(AreaBreakType.NEXT_PAGE));
    document.getRenderer().close();
    document.close();
    return bao;
  }

  /**
   * 设置中文编码
   *
   * @param props
   * @throws IOException
   */
  private void setFont(ConverterProperties props) throws IOException {
    DefaultFontProvider defaultFontProvider = new DefaultFontProvider(false, false, false);
    //设置正常字体
    InputStream inputStream = this.getClass().getClassLoader()
        .getResourceAsStream("font/SourceHanSansCN-Regular.ttf");
    byte[] bytes = MultipartFileUtil.toByteArray(inputStream);
    PdfFont regularFont = PdfFontFactory.createFont(bytes, PdfEncodings.IDENTITY_H, false);
    defaultFontProvider.addFont(regularFont.getFontProgram());
    //设置加粗字体
    InputStream boldInputStream = this.getClass().getClassLoader()
        .getResourceAsStream("font/SourceHanSansCN-Bold.ttf");
    byte[] boldBytes = MultipartFileUtil.toByteArray(boldInputStream);
    PdfFont boldFont = PdfFontFactory.createFont(boldBytes, PdfEncodings.IDENTITY_H, false);
    defaultFontProvider.addFont(boldFont.getFontProgram());
    props.setFontProvider(defaultFontProvider);
  }


  /**
   * 以追加方式合并首尾html为pdf文档
   *
   * @param headHtml 首部html页面
   * @param tailHtml 尾部html页面
   * @return pdf文档信息
   * @throws IOException IOException
   */
  private HtmlMergePdfResultModel htmlMergeWithAppend(String headHtml, String tailHtml) throws IOException {
    ConverterProperties props = new ConverterProperties();
    //设置字体
    this.setFont(props);
    HtmlMergePdfResultModel resultModel = new HtmlMergePdfResultModel();
    ByteArrayOutputStream mergeOutputStream = new ByteArrayOutputStream();
    try (PdfWriter writer = new PdfWriter(mergeOutputStream);
        PdfDocument pdf = new PdfDocument(writer);
        Document mergeDocument = HtmlConverter.convertToDocument(headHtml, pdf, props)){
      pdf.setDefaultPageSize(PageSize.A4);
      List<IElement> iElements = HtmlConverter.convertToElements(tailHtml, props);
      for (IElement iElement : iElements) {
        mergeDocument.add((IBlockElement) iElement);
      }
      mergeDocument.flush();
      resultModel.setMergeOutputStream(mergeOutputStream);
      resultModel.setPdfPages(mergeDocument.getPdfDocument().getNumberOfPages());
      return resultModel;
    }
  }

  /**
   * 以新开页面方式合并首尾html为pdf文档
   *
   * @param headHtml 首部html页面
   * @param tailHtml 尾部html页面
   * @return pdf文档信息
   * @throws IOException IOException
   */
  private HtmlMergePdfResultModel htmlMergeWithNewPage(String headHtml, String tailHtml) throws IOException {
    ConverterProperties props = new ConverterProperties();
    //设置字体
    this.setFont(props);
    HtmlMergePdfResultModel resultModel = new HtmlMergePdfResultModel();
    ByteArrayOutputStream mergeOutputStream = new ByteArrayOutputStream();
    try (PdfWriter writer = new PdfWriter(mergeOutputStream);
        PdfDocument pdf = new PdfDocument(writer);
        Document mergeDocument = HtmlConverter.convertToDocument(headHtml, pdf, props)){
      pdf.setDefaultPageSize(PageSize.A4);
      mergeDocument.add(new AreaBreak(AreaBreakType.NEXT_PAGE));
      List<IElement> iElements = HtmlConverter.convertToElements(tailHtml, props);
      for (IElement iElement : iElements) {
        mergeDocument.add((IBlockElement) iElement);
      }
      mergeDocument.flush();
      resultModel.setMergeOutputStream(mergeOutputStream);
      resultModel.setPdfPages(mergeDocument.getPdfDocument().getNumberOfPages());
      return resultModel;
    }
  }
}
