package com.biz.crm.dms.business.itextpdf.sdk.utils;

import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.util.Units;
import org.apache.poi.xwpf.usermodel.Document;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.apache.poi.xwpf.usermodel.XWPFParagraph;
import org.apache.poi.xwpf.usermodel.XWPFRun;
import org.apache.xmlbeans.XmlException;
import org.openxmlformats.schemas.drawingml.x2006.main.CTGraphicalObject;
import org.openxmlformats.schemas.drawingml.x2006.wordprocessingDrawing.CTAnchor;
import org.openxmlformats.schemas.wordprocessingml.x2006.main.CTDrawing;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

/**
 * @Project crm
 * @PackageName com.biz.crm.util
 * @ClassName WordUtil
 * @Author Gavin
 * @Date 2021/3/17 下午3:11
 * @Description word文档工具类
 */
@Slf4j
@Component
public class WordUtil {

    private static String BASE_DIR;

    @Autowired(required = false)
    private FileHandleService fileHandleService;

    @Value("${server.tomcat.basedir:}")
    public void setBaseDir(String baseDir) {
        WordUtil.BASE_DIR = baseDir;
    }

    /**
     * @param dataMap
     * @param wordName
     * @param templateName
     * @return
     */
    public static String createDocx(Map dataMap, String wordName, String templateName, String zipTemplateName) {
        ZipOutputStream zipout = null;
        String wordPath = "";
        //<#list lists as p>${p.activityCode}</#list>
        FileOutputStream outputStream = null;
        InputStream resourceAsStream = null;
        ByteArrayInputStream documentInput = null;
        try {
            File file;
            //内容模板
            documentInput = FreeMarkUtils.getFreemarkerContentInputStream(dataMap, templateName);
            //zip路径
            String path = BASE_DIR + File.separator + wordName;
            File zipFolderPath = new File(path);
            if (!zipFolderPath.exists()) {
                zipFolderPath.mkdirs();
            }
            //ClassPathResource resource = new ClassPathResource("templates" + File.separator + zipTemplateName);
            //File zipFilePath = resource.getFile();
            //读取zip模版文件
            ClassPathResource classPathResource = new ClassPathResource("templates" + File.separator + zipTemplateName);
            resourceAsStream = classPathResource.getInputStream();
            File zipFilePath = new File(path + File.separator + resourceAsStream);
            // commons-io
            FileUtils.copyInputStreamToFile(resourceAsStream, zipFilePath);
            wordPath = path + File.separator + wordName + ".docx";
            outputStream = new FileOutputStream(wordPath);
            ZipFile zipFile = new ZipFile(zipFilePath);
            Enumeration<? extends ZipEntry> zipEntrys = zipFile.entries();
            zipout = new ZipOutputStream(outputStream);
            //开始覆盖文档------------------
            int len = -1;
            byte[] buffer = new byte[1024];
            while (zipEntrys.hasMoreElements()) {
                ZipEntry next = zipEntrys.nextElement();
                InputStream is = zipFile.getInputStream(next);
                if (next.toString().indexOf("media") < 0) {
                    zipout.putNextEntry(new ZipEntry(next.getName()));
                    if ("word/document.xml".equals(next.getName())) {
                        if (documentInput != null) {
                            while ((len = documentInput.read(buffer)) != -1) {
                                zipout.write(buffer, 0, len);
                            }
                            documentInput.close();
                        }
                    } else {
                        while ((len = is.read(buffer)) != -1) {
                            zipout.write(buffer, 0, len);
                        }
                        is.close();
                    }
                }
            }
        } catch (Exception e) {
            log.warn("生成失败{}", e.getMessage());
            return null;
        } finally {
            safeClose(zipout);
            safeClose(outputStream);
            safeClose(resourceAsStream);
            safeClose(documentInput);
        }
        return wordPath;
    }

    /**
     * <b> Word中添加图章
     * </b><br><br><i>Description</i> :
     * String srcPath, 源Word路径
     * String storePath, 添加图章后的路径
     * byte[] signFileBites, 图章文件流
     * tString abText, 在Word中盖图章的标识字符串，如：(签字/盖章)
     * int width, 图章宽度
     * int height, 图章高度
     * int leftOffset, 图章在编辑段落向左偏移量
     * int topOffset, 图章在编辑段落向上偏移量
     * boolean behind，图章是否在文字下面
     *
     * @return void
     * <br><br>Date: 2019/12/26 15:12     <br>Author : dxl
     */
    public static String sealInWord(String srcPath, String wordName, byte[] signFileBites, String tabText,
                                    int width, int height, int leftOffset, int topOffset, boolean behind) {
        //签名
        String docxSginPath = BASE_DIR + File.separator + wordName + File.separator + wordName + "_sgin.docx";
        InputStream srcWordIn = null;
        InputStream imageIn = null;
        FileOutputStream sginWordOut = null;
        try {
            File fileTem = new File(srcPath);
            srcWordIn = new FileInputStream(fileTem);
            XWPFDocument document = new XWPFDocument(srcWordIn);
            List<XWPFParagraph> paragraphs = document.getParagraphs();
            XWPFRun targetRun = null;
            for (XWPFParagraph paragraph : paragraphs) {
                if (!"".equals(paragraph.getText()) && paragraph.getText().contains(tabText)) {
                    List<XWPFRun> runs = paragraph.getRuns();
                    targetRun = runs.get(runs.size() - 1);
                }
            }
            if (targetRun != null) {
                imageIn = new ByteArrayInputStream(signFileBites);
                if (width <= 0) {
                    width = 100;
                }
                if (height <= 0) {
                    height = 100;
                }
                //创建Random类对象
                Random random = new Random();
                //产生随机数
                int number = random.nextInt(999) + 1;
                targetRun.addPicture(imageIn, Document.PICTURE_TYPE_PNG, "Seal" + number, Units.toEMU(width), Units.toEMU(height));
                // 2. 获取到图片数据
                CTDrawing drawing = targetRun.getCTR().getDrawingArray(0);
                CTGraphicalObject graphicalobject = drawing.getInlineArray(0).getGraphic();

                //拿到新插入的图片替换添加CTAnchor 设置浮动属性 删除inline属性
                CTAnchor anchor = getAnchorWithGraphic(graphicalobject, "Seal" + number,
                        Units.toEMU(width), Units.toEMU(height),//图片大小
                        Units.toEMU(leftOffset), Units.toEMU(topOffset), behind);//相对当前段落位置 需要计算段落已有内容的左偏移
                drawing.setAnchorArray(new CTAnchor[]{anchor});//添加浮动属性
                drawing.removeInline(0);//删除行内属性
            }
            sginWordOut = new FileOutputStream(docxSginPath);
            document.write(sginWordOut);
            document.close();
        } catch (InvalidFormatException e) {
            log.error("添加签名异常InvalidFormatException：{}", e.getMessage());
        } catch (FileNotFoundException e) {
            log.error("添加签名异常FileNotFoundException：{}", e.getMessage());
        } catch (MalformedURLException e) {
            log.error("添加签名异常MalformedURLException：{}", e.getMessage());
        } catch (IOException e) {
            log.error("添加签名异常IOException：{}", e.getMessage());
        } finally {
            safeClose(imageIn);
            safeClose(srcWordIn);
            safeClose(sginWordOut);
        }
        return docxSginPath;
    }

    /**
     * @param ctGraphicalObject 图片数据
     * @param deskFileName      图片描述
     * @param width             宽
     * @param height            高
     * @param leftOffset        水平偏移 left
     * @param topOffset         垂直偏移 top
     * @param behind            文字上方，文字下方
     * @return
     * @throws Exception
     */
    public static CTAnchor getAnchorWithGraphic(CTGraphicalObject ctGraphicalObject,
                                                String deskFileName, int width, int height,
                                                int leftOffset, int topOffset, boolean behind) {
        System.out.println(">>width>>" + width + "; >>height>>>>" + height);
        String anchorXML =
                "<wp:anchor xmlns:wp=\"http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing\" "
                        + "simplePos=\"0\" relativeHeight=\"0\" behindDoc=\"" + ((behind) ? 1 : 0) + "\" locked=\"0\" layoutInCell=\"1\" allowOverlap=\"1\">"
                        + "<wp:simplePos x=\"0\" y=\"0\"/>"
                        + "<wp:positionH relativeFrom=\"column\">"
                        + "<wp:posOffset>" + leftOffset + "</wp:posOffset>"
                        + "</wp:positionH>"
                        + "<wp:positionV relativeFrom=\"paragraph\">"
                        + "<wp:posOffset>" + topOffset + "</wp:posOffset>" +
                        "</wp:positionV>"
                        + "<wp:extent cx=\"" + width + "\" cy=\"" + height + "\"/>"
                        + "<wp:effectExtent l=\"0\" t=\"0\" r=\"0\" b=\"0\"/>"
                        + "<wp:wrapNone/>"
                        + "<wp:docPr id=\"1\" name=\"Drawing 0\" descr=\"" + deskFileName + "\"/><wp:cNvGraphicFramePr/>"
                        + "</wp:anchor>";
        CTDrawing drawing = null;
        try {
            drawing = CTDrawing.Factory.parse(anchorXML);
        } catch (XmlException e) {
            log.error("添加签名异常XmlException：{}", e.getMessage());
        }
        CTAnchor anchor = drawing.getAnchorArray(0);
        anchor.setGraphic(ctGraphicalObject);
        return anchor;
    }

    /**
     * 删除临时文件
     *
     * @param wordName
     */
    public static void deleteTemporaryFile(String wordName) {
        File directory = new File(BASE_DIR + File.separator + wordName);
        try {
            FileUtils.deleteDirectory(directory);
        } catch (IOException e) {
            log.info("转换临时文件删除失败!");
        }
    }

    public static void safeClose(InputStream inputStream) {
        if (inputStream != null) {
            try {
                inputStream.close();
            } catch (IOException e) {
            }
        }
    }

    public static void safeClose(OutputStream outputStream) {
        if (outputStream != null) {
            try {
                outputStream.close();
            } catch (IOException e) {
            }
        }
    }
}