package com.biz.crm.common.log.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.common.log.local.entity.CrmBusinessLogEsEntity;
import com.biz.crm.common.log.local.service.CrmBusinessLogService;
import com.biz.crm.common.log.local.utils.LogTransform;
import com.biz.crm.common.log.sdk.dto.CrmBusinessLogQueryDto;
import com.biz.crm.common.log.sdk.strategy.CrmBusinessLogStrategy;
import com.biz.crm.common.log.sdk.utils.EsUtil;
import com.biz.crm.common.log.sdk.vo.CrmBusinessLogEsVo;
import com.biz.crm.common.log.sdk.vo.FieldTransformVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * @description: 业务日志service
 * @author: rentao
 * @date: 2022/5/10 15:06
 */
@Service
public class CrmBusinessLogServiceImpl implements CrmBusinessLogService {

  @Autowired
  private ElasticsearchRestTemplate elasticsearchRestTemplate;
  @Autowired(required = false)
  private List<CrmBusinessLogStrategy> crmBusinessLogStrategyServices;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<CrmBusinessLogEsVo> findByConditions(Pageable pageable,
      CrmBusinessLogQueryDto crmBusinessLogDto) {
    Validate.notNull(crmBusinessLogDto, "查询日志参数不能为空");
    Validate.notBlank(crmBusinessLogDto.getOnlyKey(), "查询日志,业务id不能为空");
    Validate.notBlank(crmBusinessLogDto.getAccessType(), "查询日志,获取方式不能为空");
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    if (!EsUtil.indexExsitNoCreate(this.elasticsearchRestTemplate, CrmBusinessLogEsEntity.class)) {
      return null;
    } else {
      BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
      //onlyKey + 菜单编码确定唯一
      boolQueryBuilder
          .must(QueryBuilders.termQuery("onlyKey.keyword", crmBusinessLogDto.getOnlyKey()));
      boolQueryBuilder.must(QueryBuilders.termQuery("appCode.keyword", TenantUtils.getTenantCode()));
      boolQueryBuilder
          .must(QueryBuilders.termQuery("tenantCode.keyword", TenantUtils.getTenantCode()));
      if (null != crmBusinessLogDto.getOperationType()) {
        boolQueryBuilder.must(
            QueryBuilders.termQuery("operationType.keyword", crmBusinessLogDto.getOperationType()));
      }
      NativeSearchQuery searchQuery = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder)
          .withSorts(SortBuilders.fieldSort("createTimeTimeNum").unmappedType("keyword").order(SortOrder.DESC))
          .withPageable(PageRequest.of(pageable.getPageNumber() - 1, pageable.getPageSize()))
          .build();
      SearchHits<CrmBusinessLogEsEntity> searchs = this.elasticsearchRestTemplate.search(searchQuery, CrmBusinessLogEsEntity.class);
      List<CrmBusinessLogEsEntity> crmLogEntityList = searchs.get().map(SearchHit::getContent).collect(Collectors.toList());
      List<CrmBusinessLogEsVo> crmBusinessLogEsEntities = new ArrayList<>();
      if (!CollectionUtils.isEmpty(crmLogEntityList) && StringUtils.isNotBlank(crmBusinessLogDto.getParentCode())) {
        //需要转换成汉字
        List<CrmBusinessLogEsVo> crmBusinessLogEsVoList = (List<CrmBusinessLogEsVo>) this.nebulaToolkitService.copyCollectionByBlankList(crmLogEntityList, CrmBusinessLogEsEntity.class,CrmBusinessLogEsVo.class, HashSet.class, ArrayList.class);
        crmBusinessLogEsEntities.addAll(this.handleFiledTransformation(crmBusinessLogEsVoList, crmBusinessLogDto.getAccessType(), crmBusinessLogDto.getParentCode()));
      }
      Page<CrmBusinessLogEsVo> page = new Page<>();
      page.setTotal(searchs.getTotalHits());
      page.setRecords(crmBusinessLogEsEntities);
      return page;
    }
  }

  private List<CrmBusinessLogEsVo> handleFiledTransformation(List<CrmBusinessLogEsVo> crmBusinessLogEsVos, String accessType, String parentCode) {
    List<FieldTransformVo> list = new ArrayList<>();
    if (CollectionUtils.isEmpty(crmBusinessLogStrategyServices)) {
      return crmBusinessLogEsVos;
    }
    for (CrmBusinessLogStrategy crmBusinessLogStrategyService : crmBusinessLogStrategyServices) {
      String group = crmBusinessLogStrategyService.getTransformationCode();
      if (StringUtils.equals(group, accessType)) {
        list.addAll(crmBusinessLogStrategyService.handleTransformation(parentCode));
        break;
      }
    }
    if (!CollectionUtils.isEmpty(list)) {
      for (CrmBusinessLogEsVo crmBusinessLogEsVo : crmBusinessLogEsVos) {
        String s = LogTransform.fieldTransform(crmBusinessLogEsVo.getCompareResult(), list);
        if (StringUtils.isNotBlank(s)) {
          JSONObject jsonObject = JSON.parseObject(s);
          crmBusinessLogEsVo.setCompareResultJson(jsonObject);
        }
      }
    }
    return crmBusinessLogEsVos;
  }

}


