package com.biz.crm.common.log.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.common.log.local.entity.CrmBusinessLogEsEntity;
import com.biz.crm.common.log.local.repository.TmLogRepository;
import com.biz.crm.common.log.local.service.CrmBusinessLogService;
import com.biz.crm.common.log.local.utils.LogTransform;
import com.biz.crm.common.log.local.vo.TmLogVo;
import com.biz.crm.common.log.sdk.dto.CrmBusinessLogQueryDto;
import com.biz.crm.common.log.sdk.strategy.CrmBusinessLogStrategy;
import com.biz.crm.common.log.sdk.utils.EsUtil;
import com.biz.crm.common.log.sdk.vo.CrmBusinessLogEsVo;
import com.biz.crm.common.log.sdk.vo.FieldTransformVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @description: 业务日志service
 * @author: rentao
 * @date: 2022/5/10 15:06
 */
@Service
public class CrmBusinessLogServiceImpl implements CrmBusinessLogService {

    @Autowired
    private ElasticsearchRestTemplate elasticsearchRestTemplate;
    @Autowired(required = false)
    private List<CrmBusinessLogStrategy> crmBusinessLogStrategyServices;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private TmLogRepository tmLogRepository;
    private final static int MAX_TOTAL = 10000;

    @Override
    public Page<CrmBusinessLogEsVo> findByConditions(Pageable pageable, CrmBusinessLogQueryDto queryDto) {
        queryDto = Optional.ofNullable(queryDto).orElse(new CrmBusinessLogQueryDto());
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        if (!EsUtil.indexExsitNoCreate(this.elasticsearchRestTemplate, CrmBusinessLogEsEntity.class)) {
            return new Page<>();
        } else {
            int pageNumber = pageable.getPageNumber();
            int pageSize = pageable.getPageSize();
            if (pageNumber > 0) {
                pageNumber = pageNumber - 1;
            }
            Assert.isTrue((pageNumber + 1) * pageSize < MAX_TOTAL,
                    "分页查询,单次查询总数不可超过[" + MAX_TOTAL + "]条,请输入更多查询条件!");
            Pageable pageEs = PageRequest.of(pageNumber, pageable.getPageSize());

            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            //onlyKey + 菜单编码确定唯一
            if (StringUtils.isNotBlank(queryDto.getOnlyKey())) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(queryDto.getOnlyKey()).field("onlyKey"));
            }
            if (StringUtils.isNotBlank(queryDto.getAppCode())) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(queryDto.getAppCode()).field("appCode"));
            }
            if (StringUtils.isNotBlank(queryDto.getTenantCode())) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(queryDto.getTenantCode()).field("tenantCode"));
            }
            if (StringUtils.isNotBlank(queryDto.getAppName())) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(queryDto.getAppName()).field("appName"));
            }
//            if (StringUtils.isNotBlank(queryDto.getMenuCode())
//                    || StringUtils.isNotBlank(queryDto.getParentCode())) {
//                String menuCode = queryDto.getMenuCode();
//                if (StringUtils.isBlank(menuCode) && StringUtils.isNotBlank(queryDto.getParentCode())) {
//                    menuCode = queryDto.getParentCode();
//                }
//                boolQueryBuilder.must(QueryBuilders.queryStringQuery(menuCode).field("menuCode"));
//            }
            if (StringUtils.isNotBlank(queryDto.getMenuName())) {
                boolQueryBuilder.must(QueryBuilders.wildcardQuery("menuName", this.getLikeString(queryDto.getMenuName())));
            }
            if (StringUtils.isNotBlank(queryDto.getMenuAllName())) {
                boolQueryBuilder.must(QueryBuilders.wildcardQuery("menuAllName", this.getLikeString(queryDto.getMenuAllName())));
            }
            if (StringUtils.isNotBlank(queryDto.getOperationType())) {
                boolQueryBuilder.must(QueryBuilders.queryStringQuery(queryDto.getOperationType()).field("operationType"));
            }
            NativeSearchQuery searchQuery = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder)
                    .withSorts(SortBuilders.fieldSort("createTimeTimeNum").unmappedType("keyword").order(SortOrder.DESC))
                    .withPageable(pageEs)
                    .build();
            SearchHits<CrmBusinessLogEsEntity> searchs = this.elasticsearchRestTemplate.search(searchQuery, CrmBusinessLogEsEntity.class);
            List<CrmBusinessLogEsEntity> crmLogEntityList = searchs.get().map(SearchHit::getContent).collect(Collectors.toList());
            List<CrmBusinessLogEsVo> crmBusinessLogEsEntities = new ArrayList<>();
            List<CrmBusinessLogEsVo> crmBusinessLogEsVoList = (List<CrmBusinessLogEsVo>) this.nebulaToolkitService.copyCollectionByBlankList(crmLogEntityList, CrmBusinessLogEsEntity.class, CrmBusinessLogEsVo.class, HashSet.class, ArrayList.class);
            if (!CollectionUtils.isEmpty(crmLogEntityList) && StringUtils.isNotBlank(queryDto.getParentCode())) {
                //需要转换成汉字
                crmBusinessLogEsEntities.addAll(this.handleFiledTransformation(crmBusinessLogEsVoList, queryDto.getAccessType(), queryDto.getParentCode()));
            } else {
                crmBusinessLogEsEntities.addAll(crmBusinessLogEsVoList);
            }
            Page<CrmBusinessLogEsVo> page = new Page<>();
            page.setTotal(searchs.getTotalHits());
            page.setRecords(crmBusinessLogEsEntities);
            page.setCurrent(pageable.getPageNumber());
            return page;
        }
    }

    @Override
    public Page<TmLogVo> findByConditionsHistory(Pageable pageable, TmLogVo dto) {
        return tmLogRepository.findByConditionsHistory(pageable, dto);
    }

    /**
     * es like查询
     *
     * @param queryString
     * @return
     */
    private String getLikeString(String queryString) {
        if (StringUtils.isEmpty(queryString)) {
            return "";
        }
        return "*" + queryString + "*";
    }


    private List<CrmBusinessLogEsVo> handleFiledTransformation(List<CrmBusinessLogEsVo> crmBusinessLogEsVos, String accessType, String parentCode) {
        List<FieldTransformVo> list = new ArrayList<>();
        if (CollectionUtils.isEmpty(crmBusinessLogStrategyServices)) {
            return crmBusinessLogEsVos;
        }
        for (CrmBusinessLogStrategy crmBusinessLogStrategyService : crmBusinessLogStrategyServices) {
            String group = crmBusinessLogStrategyService.getTransformationCode();
            if (StringUtils.equals(group, accessType)) {
                list.addAll(crmBusinessLogStrategyService.handleTransformation(parentCode));
                break;
            }
        }
        if (!CollectionUtils.isEmpty(list)) {
            for (CrmBusinessLogEsVo crmBusinessLogEsVo : crmBusinessLogEsVos) {
                String s = LogTransform.fieldTransform(crmBusinessLogEsVo.getCompareResult(), list);
                if (StringUtils.isNotBlank(s)) {
                    JSONObject jsonObject = JSON.parseObject(s);
                    crmBusinessLogEsVo.setCompareResultJson(jsonObject);
                }
            }
        }
        return crmBusinessLogEsVos;
    }

}


