package com.biz.crm.common.log.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.log.local.repository.CrmBusinessLogEsRepository;
import com.biz.crm.common.log.local.utils.CrmBusinessLogUtil;
import com.biz.crm.common.log.local.utils.LogCompare;
import com.biz.crm.common.log.sdk.dto.CrmBusinessLogDto;
import com.biz.crm.common.log.sdk.enums.OperationTypeEunm;
import com.biz.crm.common.log.sdk.service.CrmBusinessLogVoService;
import com.bizunited.nebula.common.util.SpringBeanUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.competence.sdk.constant.Constants;
import com.bizunited.nebula.competence.sdk.service.CompetenceCacheVoService;
import com.bizunited.nebula.competence.sdk.vo.CompetenceVo;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @description: 业务日志service
 * @author: rentao
 * @date: 2022/5/13 20:17
 */
@Service
@Slf4j
public class CrmBusinessLogVoServiceImpl implements CrmBusinessLogVoService {

    @Autowired(required = false)
    private CrmBusinessLogEsRepository crmBusinessLogRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Value("${spring.application.name:}")
    private String appName;

    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");


    @Override
    @Transactional
    public void handleSave(CrmBusinessLogDto crmBusinessLogDto) {
        Validate.notNull(crmBusinessLogDto, "传入比对参数不存在！");
        Validate.notBlank(crmBusinessLogDto.getOnlyKey(), "记录业务日志时，业务数据id不能为空！");
        Validate.notBlank(crmBusinessLogDto.getOperationType(), "记录业务日志时，操作类型不能为空！");
        if (StringUtils.isEmpty(crmBusinessLogDto.getAppCode())) {
            crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
        }
        if (StringUtils.isEmpty(crmBusinessLogDto.getTenantCode())) {
            crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
        }
        String operationType = crmBusinessLogDto.getOperationType();
        crmBusinessLogDto.setAppName(appName);
        setBusinessLogBaseInfo(crmBusinessLogDto);
        if (StringUtils.isNotEmpty(crmBusinessLogDto.getCreateAccount())) {
            if (OperationTypeEunm.CREATE.getDictCode().equals(operationType)) {
                this.crmBusinessLogRepository.saveAll(CrmBusinessLogUtil.voToEntityForAdd(crmBusinessLogDto));
            } else if (OperationTypeEunm.DELETE.getDictCode().equals(operationType)) {
                this.crmBusinessLogRepository.saveAll(CrmBusinessLogUtil.voToEntityForDel(crmBusinessLogDto));
            } else if (OperationTypeEunm.ENABLE.getDictCode().equals(operationType)) {
                this.crmBusinessLogRepository.saveAll(CrmBusinessLogUtil.voToEntityForEnableAndDisable(crmBusinessLogDto, OperationTypeEunm.ENABLE));
            } else if (OperationTypeEunm.DISABLE.getDictCode().equals(operationType)) {
                this.crmBusinessLogRepository.saveAll(CrmBusinessLogUtil.voToEntityForEnableAndDisable(crmBusinessLogDto, OperationTypeEunm.DISABLE));
            } else {
                //对date 字段格式化 并赋值
                Map<String, String> oldDateMap = this.transformDateFormat(crmBusinessLogDto.getOldObject());
                Map<String, String> newDateMap = this.transformDateFormat(crmBusinessLogDto.getNewObject());
                Object result = LogCompare.compareObject(JSON.toJSONString(crmBusinessLogDto.getOldObject(), SerializerFeature.WriteMapNullValue), JSON.toJSONString(crmBusinessLogDto.getNewObject(), SerializerFeature.WriteMapNullValue), oldDateMap, newDateMap,crmBusinessLogDto.getAppendMsg());
                this.crmBusinessLogRepository.save(CrmBusinessLogUtil.voToEntityForUpdate(crmBusinessLogDto, result));
            }
        }
    }

    /**
     * 日期格式化
     */
    private Map<String, String> transformDateFormat(Object crmBusinessLogDto) {
        Map<String, String> stringMap = new HashMap<>();
        if (Objects.isNull(crmBusinessLogDto)) {
            return stringMap;
        }
        List<Class<?>> classes = getAll(crmBusinessLogDto);
        for (Class<?> aClass : classes) {
            Field[] declaredFields = aClass.getDeclaredFields();
            for (int i = 0; i < declaredFields.length; i++) {
                Field declaredField = declaredFields[i];
                declaredField.setAccessible(true);
                if (declaredField.getType() == Date.class) {
                    try {
                        String name = declaredField.getName();
                        Object date = declaredField.get(crmBusinessLogDto);
                        if (Objects.nonNull(date)) {
                            stringMap.put(name, DATE_FORMAT.format(date));
                        }
                    } catch (Exception e) {
                        log.info("格式化日期错误，跳过");
                    }
                }
            }
        }
        return stringMap;
    }


    /**
     * 获取所有父类
     */
    public static List<Class<?>> getAll(Object o) {
        boolean s = true;
        List<Class<?>> classes = new ArrayList<>();
        classes.add(o.getClass());
        Class<?> superclass = o.getClass();
        while (s) {
            superclass = superclass.getSuperclass();
            if (!"java.lang.Object".equals(superclass.getName())) {
                classes.add(superclass);
            } else {
                s = false;
            }
        }
        return classes;
    }

    /**
     * 设置菜单信息
     *
     * @return
     */

    private void setBusinessLogBaseInfo(CrmBusinessLogDto dto) {
        if (Objects.nonNull(dto.getLoginUser())) {
            loginUserService.refreshAuthentication(dto.getLoginUser());
        }

        if (StringUtils.isEmpty(dto.getCreateAccount())) {
            AbstractCrmUserIdentity context = loginUserService.getAbstractLoginUser();
            dto.setCreateAccount(context.getAccount());
            dto.setCreateName(context.getRealName());
        }
        if (StringUtils.isEmpty(dto.getMenuCode())) {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            if (requestAttributes != null) {
                HttpServletRequest request = requestAttributes.getRequest();
                String menuCode = request.getHeader(Constants.MENU_CODE);
                if (StringUtils.isNotEmpty(menuCode)) {
                    CompetenceCacheVoService competenceCacheVoService = SpringBeanUtils.getBean(CompetenceCacheVoService.class);
                    CompetenceVo competenceVo = competenceCacheVoService.findCacheByCompetenceCode(menuCode);
                    if (Objects.nonNull(competenceVo)) {
                        dto.setMenuCode(competenceVo.getCode());
                        dto.setMenuName(competenceVo.getComment());
                        dto.setMenuAllName(competenceVo.getParentCodeAllComment());
                    }
                }
            }
        }
    }
}
