package com.biz.crm.common.log.local.task;

import com.biz.crm.common.log.local.config.CleanCrmBusinessLogConfig;
import com.biz.crm.common.log.local.entity.CrmBusinessLogEsEntity;
import com.biz.crm.common.log.sdk.constant.BusinessLogConstant;
import com.biz.crm.common.log.sdk.service.CleanBusinessLogExtendService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.ByQueryResponse;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 清理业务操作ES日志
 *
 * @author huxmld
 * @version v1.0.0
 * @describe 简述
 * @date 2024.03.01 18:01
 */
@EnableScheduling
@Component
@Slf4j
public class DeleteCrmBusinessLogCornTask implements SchedulingConfigurer {

    private static final String DEFAULT_CRON = "0 10 0 * * ?";

    private String cron = DEFAULT_CRON;
    private boolean openClean;

    @Resource
    private ElasticsearchRestTemplate elasticsearchRestTemplate;
    @Resource
    private CleanCrmBusinessLogConfig cleanCrmBusinessLogConfig;
    @Autowired(required = false)
    private CleanBusinessLogExtendService cleanBusinessLogExtendService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    /**
     * 清理对外日志应用-必须是 crm-mdm
     */
    private final static String APPLICATION_NAME = "crm-mdm";

    /**
     * 根据配置文件参数,系统启动完成后,自动设置自定义定时任务执行计划
     */
    @PostConstruct
    private void setDeleteLogCron() {
        this.setCron(cleanCrmBusinessLogConfig.getCleanCron());
        openClean = cleanCrmBusinessLogConfig.isOpenClean();
    }

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.addTriggerTask(this::deleteData, triggerContext -> {
            CronTrigger trigger = new CronTrigger(cron);
            return trigger.nextExecutionTime(triggerContext);
        });
    }

    public void setCron(String cron) {
        log.info("====>     清理业务操作ES日志:原来的cron：[{}]]更新后的cron：[{}]     <=====", this.cron, cron);
        this.cron = cron;
    }

    /**
     * 清理日志
     */
    private void deleteData() {
        if (!APPLICATION_NAME.equalsIgnoreCase(cleanCrmBusinessLogConfig.getApplicationName())
                || !openClean) {
            return;
        }
        String lockKey = BusinessLogConstant.CRM_BUSINESS_LOG_DEL;
        if (redisMutexService != null) {
            boolean lock = redisMutexService.tryLock(lockKey, TimeUnit.MINUTES, 30);
            if (!lock) {
                return;
            }
        }
        if (cleanBusinessLogExtendService == null) {
            try {
                Integer saveDay = cleanCrmBusinessLogConfig.getSaveDay();
                if (saveDay == null) {
                    saveDay = 90;
                }
                Long minNum = cleanCrmBusinessLogConfig.getMinNum();
                if (minNum == null) {
                    minNum = 10000L;
                }
                log.info("清理业务操作ES日志清理[{}]天及其以前的;最小存放数量[{}]条", saveDay, minNum);
                long totalCount = elasticsearchRestTemplate.count(new NativeSearchQuery(QueryBuilders.boolQuery()), CrmBusinessLogEsEntity.class);
                if (totalCount > minNum) {
                    log.info("清理业务操作ES日志当前总[{}]条,大于日志保有最小量[{}],开始清理日志。", totalCount, minNum);
                    long saveDayMillis = 1000L * 60 * 60 * 24 * saveDay;
                    long cleanMillis = System.currentTimeMillis() - saveDayMillis;
                    SimpleDateFormat dateYyyyMMDdHHMmSs = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    String cleanDate = dateYyyyMMDdHHMmSs.format(new Date(cleanMillis));
                    log.info("清理业务操作ES日志清理[{}][{}]及其以前的", cleanDate, cleanMillis);
                    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
                    boolQueryBuilder.must(QueryBuilders.rangeQuery("createTimeTimeNum").lte(cleanMillis));
                    NativeSearchQuery deleteQuery = new NativeSearchQuery(boolQueryBuilder);
                    ByQueryResponse queryResponse = elasticsearchRestTemplate.delete(deleteQuery, CrmBusinessLogEsEntity.class);
                    log.info("清理业务操作ES日志清理数量:[{}]条", queryResponse.getDeleted());
                } else {
                    log.info("清理业务操作ES日志当前总[{}]条,小于日志保有最小量[{}],不清理日志。", totalCount, minNum);
                }
            } catch (Exception e) {
                log.error("清理业务操作ES日志 异常");
                log.error(e.getMessage(), e);
            }
        } else {
            cleanBusinessLogExtendService.cleanBusinessLog();
        }
        if (redisMutexService != null) {
            redisMutexService.unlock(lockKey);
        }
    }

}
