package com.biz.crm.common.message.local.service.internal;

import com.biz.crm.common.message.sdk.dto.SystemMessageTypeQueryDto;
import com.biz.crm.common.message.sdk.register.SystemMessageHandlerRegister;
import com.biz.crm.common.message.sdk.register.SystemMessageRegister;
import com.biz.crm.common.message.sdk.service.SystemMessageTypeService;
import com.biz.crm.common.message.sdk.vo.SystemMessageTypeItemVo;
import com.biz.crm.common.message.sdk.vo.SystemMessageTypeVo;
import com.google.common.collect.Lists;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 系统消息通知类型接口实现
 *
 * @author sunx
 * @date 2022/7/4
 */
@Service
public class SystemMessageTypeServiceImpl implements SystemMessageTypeService {

  @Autowired(required = false)
  private List<SystemMessageRegister> registers;

  @Autowired(required = false)
  private List<SystemMessageHandlerRegister> handlerRegisters;

  @Override
  public List<SystemMessageTypeVo> findBySystemMessageTypeQueryDto(SystemMessageTypeQueryDto dto) {
    if (CollectionUtils.isEmpty(registers)) {
      return Lists.newLinkedList();
    }
    List<SystemMessageTypeVo> list = Lists.newArrayList();
    for (SystemMessageRegister register : registers) {
      SystemMessageTypeVo cur = this.findByRegister(register);
      if (Objects.nonNull(cur)) {
        list.add(cur);
      }
    }
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }

    dto.setDesc(Optional.ofNullable(dto.getDesc()).orElse(StringUtils.EMPTY));
    dto.setItemName(Optional.ofNullable(dto.getItemName()).orElse(StringUtils.EMPTY));
    dto.setName(Optional.ofNullable(dto.getName()).orElse(StringUtils.EMPTY));
    dto.setNoticeDesc(Optional.ofNullable(dto.getNoticeDesc()).orElse(StringUtils.EMPTY));

    return list.stream()
        .filter(a -> a.getName().contains(dto.getName()))
        .filter(a -> a.getDesc().contains(dto.getDesc()))
        .filter(a -> a.getNoticeDesc().contains(dto.getNoticeDesc()))
        .filter(a -> a.getItemName().contains(dto.getItemName()))
        .sorted(Comparator.comparing(SystemMessageTypeVo::getOrder))
        .collect(Collectors.toList());
  }

  @Override
  public SystemMessageRegister findByCode(String code) {
    if (CollectionUtils.isEmpty(registers)) {
      return null;
    }
    return this.registers.stream().filter(a -> a.getCode().equals(code)).findFirst().orElse(null);
  }

  @Override
  public List<SystemMessageHandlerRegister> findHandlerByCodes(Set<String> codes) {
    if (CollectionUtils.isEmpty(handlerRegisters) || CollectionUtils.isEmpty(codes)) {
      return Lists.newLinkedList();
    }
    return this.handlerRegisters.stream()
        .filter(a -> codes.contains(a.getCode()))
        .sorted(Comparator.comparing(SystemMessageHandlerRegister::getOrder))
        .collect(Collectors.toList());
  }

  /**
   * 根据注册器获取实体信息
   *
   * @param register
   * @return
   */
  private SystemMessageTypeVo findByRegister(SystemMessageRegister register) {
    if (Objects.isNull(register)) {
      return null;
    }
    final SystemMessageTypeVo vo = new SystemMessageTypeVo();
    vo.setCode(register.getCode());
    vo.setName(register.getName());
    vo.setDesc(register.getDesc());
    vo.setOrder(register.getOrder());
    if (!CollectionUtils.isEmpty(register.getHandlerRegisterList())) {
      List<SystemMessageTypeItemVo> list = Lists.newArrayList();
      for (SystemMessageHandlerRegister item : register.getHandlerRegisterList()) {
        final SystemMessageTypeItemVo cur = new SystemMessageTypeItemVo();
        cur.setCode(item.getCode());
        cur.setName(item.getName());
        cur.setNoticeDesc(item.getNoticeDesc());
        cur.setContent(item.getContent());
        cur.setOrder(item.getOrder());
        list.add(cur);
      }
      vo.setItems(
          list.stream()
              .sorted(Comparator.comparing(SystemMessageTypeItemVo::getOrder))
              .collect(Collectors.toList()));
      vo.setItemName(
          vo.getItems().stream()
              .map(SystemMessageTypeItemVo::getName)
              .collect(Collectors.joining(",")));
      vo.setNoticeDesc(
          vo.getItems().stream()
              .map(SystemMessageTypeItemVo::getNoticeDesc)
              .collect(Collectors.joining(",")));
    }
    return vo;
  }
}
