package com.biz.crm.common.pay.business.local.service.internal;



import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.pay.business.local.repository.PaymentRecordRepository;
import com.biz.crm.common.pay.business.local.entity.PaymentRecord;
import com.biz.crm.common.pay.business.local.service.PaymentRecordService;
import com.biz.crm.common.pay.business.sdk.dto.PaymentRecordDto;
import com.biz.crm.common.pay.business.sdk.enums.PayStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

/**
 * 支付流水表(PaymentRecord)表服务实现类
 *
 * @author HanJiaJun
 * @since 2022-08-12 14:02:56
 */
@Service("PaymentRecordService")
@Slf4j
public class PaymentRecordServiceImpl implements PaymentRecordService {

  @Autowired
  private PaymentRecordRepository paymentRecordRepository;

  /**
   * 分页查询数据
   * @param pageable 分页对象
   * @param dto 实体对象
   * @return
   */
  @Override
  public Page<PaymentRecord> findByConditions(Pageable pageable, PaymentRecordDto dto) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(dto)) {
      dto = new PaymentRecordDto();
    }
    return this.paymentRecordRepository.findByConditions(pageable, dto);
  }
  
  /**
   * 通过主键查询单条数据
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public PaymentRecord findById(String id) {
    if (StringUtils.isBlank(id)) {
	  return null;
	}
    return this.paymentRecordRepository.getById(id);
  }
  
  /**
   * 新增数据
   * @param paymentRecord 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public PaymentRecord create(PaymentRecord paymentRecord) {
    this.createValidate(paymentRecord);
    this.paymentRecordRepository.saveOrUpdate(paymentRecord);
    return paymentRecord;
  }
  
  /**
   * 修改新据
   * @param paymentRecord 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public PaymentRecord update(PaymentRecord paymentRecord) {
    this.updateValidate(paymentRecord);
    this.paymentRecordRepository.saveOrUpdate(paymentRecord);
    return paymentRecord;
  }
  
  /**
   * 删除数据
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    this.paymentRecordRepository.removeByIds(idList);
  }

  @Override
  public void updateStatusByTxSn(String txSn, String status) {
    Validate.notBlank(txSn,"交易流水号不能为空");
    Validate.notBlank(status,"状态不能为空");
    this.paymentRecordRepository.updateStatusByTxSn(txSn,status);
  }

  @Override
  public PaymentRecord findByTxSn(String txSN) {
    if (StringUtils.isBlank(txSN)){
      return null;
    }
    return this.paymentRecordRepository.findByTxSn(txSN);
  }

  @Override
  public PaymentRecord findByOrderNo(String orderNo) {
    if (StringUtils.isBlank(orderNo)){
      return null;
    }
    return this.paymentRecordRepository.findByOrderNo(orderNo);
  }

  @Override
  public void deleteQRMsgByTxSn(String txSN) {
    Validate.notBlank(txSN,"删除QR信息时，交易流水号不能为空");
    PaymentRecord byTxSn = this.paymentRecordRepository.findByTxSn(txSN);
    Validate.notNull(byTxSn,"删除QR信息时,未查询到相关流水记录");
    this.paymentRecordRepository.deleteQRMsgByTxSn(txSN);
  }

  /**
   * 创建验证
   * @param paymentRecord
   */
  private void createValidate(PaymentRecord paymentRecord) {
    Validate.notNull(paymentRecord, "新增时，对象信息不能为空！");
    paymentRecord.setId(null);
    paymentRecord.setTenantCode(TenantUtils.getTenantCode());
    paymentRecord.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    paymentRecord.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //默认状态 处理中
    paymentRecord.setStatus(PayStatusEnum.PROCESSING.getDictCode());
    Validate.notBlank(paymentRecord.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notNull(paymentRecord.getAmount(), "新增数据时，金额不能为空！");
    Validate.notBlank(paymentRecord.getCustomerCode(), "新增数据时，客户编码不能为空！");
    Validate.notNull(paymentRecord.getOperationTime(), "新增数据时，操作时间不能为空！");
    Validate.notBlank(paymentRecord.getOperationType(), "新增数据时，操作类型不能为空！");
    Validate.notBlank(paymentRecord.getRelationshipDocNumber(), "新增数据时，关联单据号不能为空！");
  }
  
   /**
   * 修改验证
   * @param paymentRecord
   */
  private void updateValidate(PaymentRecord paymentRecord) {
    Validate.notNull(paymentRecord, "修改时，对象信息不能为空！");
    Validate.notBlank(paymentRecord.getId(), "修改数据时，id不能为空！");
    Validate.notBlank(paymentRecord.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notNull(paymentRecord.getAmount(), "修改数据时，金额不能为空！");
    Validate.notBlank(paymentRecord.getCustomerCode(), "修改数据时，客户编码不能为空！");
    Validate.notNull(paymentRecord.getOperationTime(), "修改数据时，操作时间不能为空！");
    Validate.notBlank(paymentRecord.getOperationType(), "修改数据时，操作类型不能为空！");
    Validate.notBlank(paymentRecord.getRelationshipDocNumber(), "修改数据时，关联单据号不能为空！");
  }
}

