package com.biz.crm.common.pay.business.local.service.notifier;

import com.biz.crm.common.pay.business.sdk.dto.PayEventDto;
import com.biz.crm.common.pay.business.sdk.enums.PayStatusEnum;
import com.biz.crm.common.pay.business.sdk.event.PayEventListener;
import com.biz.crm.common.pay.business.sdk.service.PaymentRecordVoService;
import com.biz.crm.common.pay.business.sdk.vo.PaymentRecordVo;
import com.biz.crm.common.pay.support.sdk.event.CustomerRechargeListener;
import com.biz.crm.common.pay.support.sdk.vo.transfer.notice.RechargeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * @author HanJiaJun
 * @describe: 客户支付操作监听器
 * @createTime 2022年08月30日 16:56:00
 */
@Component
@Slf4j
public class CustomerRechargeListenerImpl implements CustomerRechargeListener {

  @Autowired
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private PaymentRecordVoService paymentRecordVoService;

  /**
   * 未支付
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onUnpaid(RechargeVo vo) {
    this.generalValidate(vo);
    //改变在线支付流水记录状态
    PaymentRecordVo byTxSn = paymentRecordVoService.findByTxSn(vo.getTxSN());
    Validate.notNull(byTxSn, "不存在流水记录");
    //更新状态
    paymentRecordVoService.updateStatusByTxSn(vo.getTxSN(), PayStatusEnum.UNPAID.getDictCode());
    //发送事件
    PayEventDto payEventDto = this.nebulaToolkitService.copyObjectByWhiteList(byTxSn, PayEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<PayEventListener, PayEventDto> onUnpaid = PayEventListener::onUnpaid;
    this.nebulaNetEventClient.publish(payEventDto,PayEventListener.class,onUnpaid);

  }

  /**
   * 在处理
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onProcessing(RechargeVo vo) {
    this.generalValidate(vo);
    PaymentRecordVo byTxSn = paymentRecordVoService.findByTxSn(vo.getTxSN());
    Validate.notNull(byTxSn, "不存在流水记录");
    //改变在线支付流水记录状态
    paymentRecordVoService.updateStatusByTxSn(vo.getTxSN(), PayStatusEnum.PROCESSING.getDictCode());
    //发送事件
    PayEventDto payEventDto = this.nebulaToolkitService.copyObjectByWhiteList(byTxSn, PayEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<PayEventListener, PayEventDto> onProcessing = PayEventListener::onProcessing;
    this.nebulaNetEventClient.publish(payEventDto,PayEventListener.class,onProcessing);
  }

  /**
   * 成功
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onSuccess(RechargeVo vo) {
    this.generalValidate(vo);
    PaymentRecordVo byTxSn = paymentRecordVoService.findByTxSn(vo.getTxSN());
    Validate.notNull(byTxSn, "不存在流水记录");
    //改变在线支付流水记录状态
    paymentRecordVoService.updateStatusByTxSn(vo.getTxSN(), PayStatusEnum.SUCCESS.getDictCode());
    //删除QR信息
    paymentRecordVoService.deleteQRMsgByTxSn(vo.getTxSN());
    //发送事件
    PayEventDto payEventDto = this.nebulaToolkitService.copyObjectByWhiteList(byTxSn, PayEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<PayEventListener, PayEventDto> onSuccess = PayEventListener::onSuccess;
    this.nebulaNetEventClient.publish(payEventDto,PayEventListener.class,onSuccess);
  }

  /**
   * 失败
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onFailure(RechargeVo vo) {
    this.generalValidate(vo);
    PaymentRecordVo byTxSn = paymentRecordVoService.findByTxSn(vo.getTxSN());
    Validate.notNull(byTxSn, "不存在流水记录");
    //改变在线支付流水记录状态
    paymentRecordVoService.updateStatusByTxSn(vo.getTxSN(), PayStatusEnum.FAILURE.getDictCode());
    //删除QR信息
    paymentRecordVoService.deleteQRMsgByTxSn(vo.getTxSN());
    //发送事件
    PayEventDto payEventDto = this.nebulaToolkitService.copyObjectByWhiteList(byTxSn, PayEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<PayEventListener, PayEventDto> onFailure = PayEventListener::onFailure;
    this.nebulaNetEventClient.publish(payEventDto,PayEventListener.class,onFailure);
  }

  /**
   * 关闭
   *
   * @param vo
   */
  @Override
  @Transactional
  public void onClose(RechargeVo vo) {
    this.generalValidate(vo);
    PaymentRecordVo byTxSn = paymentRecordVoService.findByTxSn(vo.getTxSN());
    Validate.notNull(byTxSn, "不存在流水记录");
    //改变在线支付流水记录状态
    paymentRecordVoService.updateStatusByTxSn(vo.getTxSN(), PayStatusEnum.CLOSE.getDictCode());
    //删除QR信息
    paymentRecordVoService.deleteQRMsgByTxSn(vo.getTxSN());
    //发送事件
    PayEventDto payEventDto = this.nebulaToolkitService.copyObjectByWhiteList(byTxSn, PayEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<PayEventListener, PayEventDto> onClose = PayEventListener::onClose;
    this.nebulaNetEventClient.publish(payEventDto,PayEventListener.class,onClose);
  }

  /**
   * 通用校验
   *
   * @param vo
   */
  private void generalValidate(RechargeVo vo) {
    Validate.notNull(vo, "不存在监听实例");
    String txSN = vo.getTxSN();
    Validate.notNull(txSN, "监听实例不存在交易流水号");
  }

}
