package com.biz.crm.common.pay.support.cpcn.base.cpcn.configuration;

import cfca.sadk.x509.certificate.X509Cert;
import com.biz.crm.common.pay.support.cpcn.base.common.http.HttpsConnection;
import com.biz.crm.common.pay.support.cpcn.base.common.http.SecurityContext;
import com.biz.crm.common.pay.support.cpcn.base.common.http.internal.HttpsClientConnection;
import com.biz.crm.common.pay.support.cpcn.base.common.http.internal.SimpleHttpsConnection;
import com.biz.crm.common.pay.support.cpcn.base.common.security.CertificateVerifier;
import com.biz.crm.common.pay.support.cpcn.base.common.security.PfxSigner;
import com.biz.crm.common.pay.support.cpcn.base.common.security.SMSigner;
import com.biz.crm.common.pay.support.cpcn.base.common.security.SMVerifier;
import com.biz.crm.common.pay.support.cpcn.base.common.security.Signer;
import com.biz.crm.common.pay.support.cpcn.base.common.security.Verifier;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.enums.CpcnEncryptType;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.config.CpcnConfig;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.apache.http.client.HttpClient;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.ssl.SSLContexts;
import org.apache.http.ssl.TrustStrategy;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ResourceUtils;

import javax.annotation.Resource;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.validation.Validation;
import javax.validation.ValidatorFactory;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 中金支付自动装配类
 *
 * @author Keller
 */
@ComponentScan(basePackages = {"com.biz.crm.common.pay.support.cpcn"})
@Configuration
@EnableConfigurationProperties(CpcnProperties.class)
@ConditionalOnProperty(prefix = CpcnProperties.PREFIX, name = "enabled", havingValue = "true")
@Slf4j
public class CpcnAutoConfiguration {

  @Resource
  private CpcnProperties cpcnProperties;

  /**
   * 通过配置信息初始化中金支付的相关证书以及签名、验证工具
   */
  @Bean
  @ConditionalOnMissingBean(CpcnConfig.class)
  public CpcnConfig getCpcnConfig() throws Exception {
    /*
     * 初始化支付的相关功能与参数
     * 1、验证参数
     * 2、根据加密类型初始化对应的签名与验证方法
     * 3、初始化其他参数
     */
    CpcnConfig cpcnConfig = new CpcnConfig();
    Validate.notBlank(cpcnProperties.getEncryptType(), "加密类型配置不能为空，请检查!");
    Validate.notBlank(cpcnProperties.getPriKey(), "平台私钥参数为空,请检查!");
    Validate.notBlank(cpcnProperties.getPubKey(), "平台公钥参数为空,请检查!");
    Validate.notBlank(cpcnProperties.getPaymentUrl(), "支付平台支付接口地址为空，请检查!");
    Validate.notBlank(cpcnProperties.getTxUrl(), "支付平台交易接口地址为空，请检查!");
    Validate.notBlank(cpcnProperties.getGateway4fileUrl(), "Gateway4File接口地址为空，请检查!");
    Validate.notBlank(cpcnProperties.getAlgorithm(), "支付平台加密算法名为空，请检查!");
    Validate.notBlank(cpcnProperties.getInstitutionID(), "支付平台编号为空，请检查!");
    Validate.notBlank(cpcnProperties.getPassword(), "支付平台私钥密码为空，请检查!");
    Validate.notBlank(cpcnProperties.getNoticeUrl(),"支付平台回调地址为空，请检查!");
    Validate.notBlank(cpcnProperties.getTrustKey(),"SSL请求证书为空，请检查!");
    Validate.notBlank(cpcnProperties.getTrustPassword(),"SSL请求证书密码为空，请检查!");
    Validate.notBlank(cpcnProperties.getPlatformName(),"平台名称为空，请检查!");
    Validate.notNull(cpcnProperties.getIsDgEnv(),"是否使用数字信封不能为空，请检查");

    // 初始化SSL请求证书
    SecurityContext.initSSLSocketFactory(getFilePath(cpcnProperties.getTrustKey()),cpcnProperties.getTrustPassword());

    Signer signer = null;
    Verifier verifier = null;
    // 国际加密
    if (CpcnEncryptType.INTERNATIONAL_ENCRYPT.getCode().equals(cpcnProperties.getEncryptType())) {
      signer = new PfxSigner(getFilePath(cpcnProperties.getPriKey()), cpcnProperties.getPassword(),cpcnProperties.getAlgorithm());
      verifier = new CertificateVerifier(getFilePath(cpcnProperties.getPubKey()), cpcnProperties.getAlgorithm());
    } else if (CpcnEncryptType.NATIONAL_ENCRYPT.getCode().equals(cpcnProperties.getEncryptType())) {
      signer = new SMSigner(getFilePath(cpcnProperties.getPriKey()), cpcnProperties.getPassword());
      //X509Cert x509Cert = CertUtil.getCertFromSM2(getFilePath(cpcnProperties.getPubKey()));
      X509Cert x509Cert = new X509Cert(getFilePath(cpcnProperties.getPubKey()));
      verifier = new SMVerifier(x509Cert);
    }
    // 签名与验证方法
    cpcnConfig.setSigner(signer);
    cpcnConfig.setVerifier(verifier);
    // 其他参数
    cpcnConfig.setAlgorithm(cpcnProperties.getAlgorithm());
    cpcnConfig.setInstitutionID(cpcnProperties.getInstitutionID());
    cpcnConfig.setPaymentUrl(cpcnProperties.getPaymentUrl());
    cpcnConfig.setTxUrl(cpcnProperties.getTxUrl());
    cpcnConfig.setPassword(cpcnProperties.getPassword());
    cpcnConfig.setNoticeUrl(cpcnProperties.getNoticeUrl());
    cpcnConfig.setGateway4fileUrl(cpcnProperties.getGateway4fileUrl());
    cpcnConfig.setGateway4aggregatePaymentUrl(cpcnProperties.getGateway4aggregatePaymentUrl());
    cpcnConfig.setGateway4aggregateTxUrl(cpcnProperties.getGateway4aggregateTxUrl());
    cpcnConfig.setPlatformName(cpcnProperties.getPlatformName());
    cpcnConfig.setIsDgEnv(cpcnProperties.getIsDgEnv());
    cpcnConfig.setDebug(cpcnProperties.isDebug());

    log.debug("***中金支付配置信息加载完成***");
    return cpcnConfig;
  }

  /**
   * 获取证书文件的绝对路径
   *
   * @param path
   * @throws FileNotFoundException
   */
  private String getFilePath(String path) throws FileNotFoundException {
    File file = ResourceUtils.getFile(path);
    Validate.notNull(file, String.format("加载文件【%s】为空，请检查！", path));
    return file.getAbsolutePath();
  }

  /**
   * 网络请求工具获取
   */
  @Bean
  @ConditionalOnMissingBean(HttpsConnection.class)
  @ConditionalOnClass(HttpClient.class)
  public HttpsConnection getHttpsClientConnection(){
    HttpsConnection httpsConnection = new HttpsClientConnection();
    return httpsConnection;
  }

  /**
   * 网络请求工具获取
   */
  @Bean
  @ConditionalOnMissingBean(HttpsConnection.class)
  public HttpsConnection getHttpsConnection(){
    HttpsConnection httpsConnection = new SimpleHttpsConnection();
    return httpsConnection;
  }

  /**
   * 属性验证Validator
   */
  @Bean
  @ConditionalOnMissingBean(ValidatorFactory.class)
  public ValidatorFactory getLocalValidatorFactoryBean(){
    ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();
    return validatorFactory;
  }


  @Bean
  public CloseableHttpClient httpClient()
          throws KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
    SSLContextBuilder builder = SSLContexts.custom();
    builder.loadTrustMaterial(null, new TrustStrategy() {
      @Override
      public boolean isTrusted(X509Certificate[] chain, String authType)
              throws CertificateException {
        return true;
      }
    });
    SSLContext sslContext = builder.build();
    SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(
            sslContext, new X509HostnameVerifier() {
      @Override
      public void verify(String host, SSLSocket ssl)
              throws IOException {
      }
      @Override
      public void verify(String host, X509Certificate cert)
              throws SSLException {
      }
      @Override
      public void verify(String host, String[] cns,
                         String[] subjectAlts) throws SSLException {
      }
      @Override
      public boolean verify(String s, SSLSession sslSession) {
        return true;
      }
    });
    Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder
            .<ConnectionSocketFactory> create().
                    register("https", sslsf).
                    register("http", PlainConnectionSocketFactory.INSTANCE)
                    .build();
    PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager(
            socketFactoryRegistry);
    CloseableHttpClient httpclient = HttpClients.custom()
            .setConnectionManager(cm).build();
    return httpclient;
  }

  /**
   * 创建远程通知线程池
   */
  private class DefaultRemoteNoticeThreadFactory implements ThreadFactory{
    private AtomicInteger count = new AtomicInteger(0);
    @Override
    public Thread newThread(Runnable r) {
      String threadName = "defaultRemoteNotice-thread-" + count.incrementAndGet();
      Thread defaultRemoteNoticeThread = new Thread(r, threadName);
      return defaultRemoteNoticeThread;
    }
  }

  /**
   * 该线程池，专门由于进行远程应用通知
   * @return
   */
  @Bean("defaultRemoteNoticeExecutor")
  @ConditionalOnMissingBean(name = "defaultRemoteNoticeExecutor")
  public ThreadPoolExecutor getDefaultDatasourceExecutor() {
    ThreadPoolExecutor threadPoolExecutor = new ThreadPoolExecutor(2, 5, 1, TimeUnit.MINUTES, new LinkedBlockingDeque<>(), new DefaultRemoteNoticeThreadFactory());
    return threadPoolExecutor;
  }
}
