package com.biz.crm.common.pay.support.cpcn.base.cpcn.strategy.tx;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.pay.support.cpcn.base.common.http.HttpsConnection;
import com.biz.crm.common.pay.support.cpcn.base.common.http.NameValuePair;
import com.biz.crm.common.pay.support.cpcn.base.common.http.RequestDgtEnvlp;
import com.biz.crm.common.pay.support.cpcn.base.common.http.ResponseDgtEnvlp;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.enums.CpcnRequestType;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.config.CpcnConfig;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.TxBaseRequest;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.TxBaseResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Base64Utils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.validation.ValidatorFactory;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 中金接口调用抽象类
 * 该抽象类实现spring InitializingBean接口，在类属性初始化完成后执行afterPropertiesSet方法
 *
 * @author Keller
 */
@Slf4j
public abstract class AbstractTxStrategy<T extends TxBaseRequest, R extends TxBaseResponse> {
  @Autowired
  protected HttpsConnection httpsConnection;
  @Autowired
  protected CpcnConfig cpcnConfig;
  @Autowired
  protected ValidatorFactory validatorFactory;

  /**
   * 策略统一执行处理方法
   *
   * @param request 请求封装报文
   * @return TxBaseResponse 返回响应报文
   */
  public R handler(T request) {
    request.setCpcnConfig(cpcnConfig);
    request.setValidator(validatorFactory.getValidator());
    // 执行属性验证
    request.validate();
    try {
      request.process();
    } catch (Exception e) {
      log.error("数据封装异常", e);
      throw new RuntimeException("数据封装异常", e);
    }
    // 继续接口请求业务
    R response = null;
    log.info("请求中金请求参数为{}", JSONObject.toJSONString(request));

    RequestDgtEnvlp requestDgtEnvlp = null;
    if(cpcnConfig.getIsDgEnv()){
      requestDgtEnvlp = new RequestDgtEnvlp();
      requestDgtEnvlp.setDigitalEnvelope(request.getDgtlEnvlp());
      requestDgtEnvlp.setEncryptSN(request.getEncryptSN());
      requestDgtEnvlp.setSignAlgorithm(request.getSignAlgorithm());
      requestDgtEnvlp.setSignSN(request.getSignSN());
      requestDgtEnvlp.setIsDgEnv("YES");
      requestDgtEnvlp.setInstitutionID(this.cpcnConfig.getInstitutionID());
    }
    // 调用接口请求
    String[] resp = this.send(request.getRequestMessage(), request.getRequestSignature(), request.getRequestPlainText(),requestDgtEnvlp);
    if (ObjectUtils.isEmpty(resp) || resp.length <= 1) {
      throw new IllegalArgumentException("返回数据不符合格式");
    }
    if("YES".equals(resp[2])) {
      //如果响应是数字信封，对消息进行对称解密
      try {
        log.info("开始对响应消息做对称解密。。。。。。");
        resp[0] = cpcnConfig.getSigner().decrypt(resp[0],resp[3]);
        resp[0] = new String(Base64Utils.encode(resp[0].getBytes("UTF-8")));
        log.info("响应消息做对称解密完成。。。。。。");
      }catch (Exception e) {
        throw new IllegalArgumentException("对称解密异常");
      }
    }else {
      // 1:signature
//      plainText = new String(cpcn.institution.tools.util.Base64.decode(respMsg[0]), "UTF-8");
      if(!StringUtils.isEmpty(resp[5])) {
        resp[0] = resp[0]+","+resp[5]+","+resp[4]+","+resp[2];
      }
    }
    // 检查策略处理器
    Validate.notBlank(this.getCode(), "接口编号不能为空");
    Validate.notNull(this.getResponse(), "响应报文类不能为空");
    Class<R> responseType = this.getResponse();
    Validate.notNull(responseType, "responseType获取类型为空");
    try {
      Constructor<R> constructor = responseType.getConstructor(String.class, String.class, CpcnConfig.class);
      response = constructor.newInstance(resp[0], resp[1], cpcnConfig);
      log.info("请求中金返回参数为{}", JSONObject.toJSONString(response));
    } catch (Exception e) {
      log.error("处理数据出现错误", e);
    }
    Validate.notNull(response, "获取数据为空");
    return response;
  }

  /**
   * 请求中金接口
   *
   * @param requestMessage   请求的信息
   * @param requestSignature 签名信息
   * @param xml              组成的xml
   * @return 结果
   */
  protected String[] send(String requestMessage, String requestSignature, String xml,RequestDgtEnvlp requestDgtEnvlp) {
    String responseText;
    String[] response = new String[7];
    // 请求参数初始化
    String url = getRequestUrl();
    List<NameValuePair> list = new ArrayList<>();
    NameValuePair message = new NameValuePair("message", requestMessage);
    NameValuePair signature = new NameValuePair("signature", requestSignature);
    if(requestDgtEnvlp != null){
      NameValuePair isDgEnv = new NameValuePair("isDgEnv", requestDgtEnvlp.getIsDgEnv());
      NameValuePair digitalEnvelope = new NameValuePair("digitalEnvelope", StringUtils.isEmpty(requestDgtEnvlp.getDigitalEnvelope()) ? "" : requestDgtEnvlp.getDigitalEnvelope());
      NameValuePair signAlgorithm = new NameValuePair("signAlgorithm", requestDgtEnvlp.getSignAlgorithm());
      NameValuePair signSN = new NameValuePair("signSN", requestDgtEnvlp.getSignSN());
      NameValuePair encryptSN = new NameValuePair("encryptSN", StringUtils.isEmpty(requestDgtEnvlp.getEncryptSN()) ? "" : requestDgtEnvlp.getEncryptSN());
      NameValuePair institutionID = new NameValuePair("institutionID", requestDgtEnvlp.getInstitutionID());

      list.add(isDgEnv);
      list.add(digitalEnvelope);
      list.add(signAlgorithm);
      list.add(signSN);
      list.add(encryptSN);
      list.add(institutionID);
    }
    list.add(message);
    list.add(signature);
    // 调试信息
    log.debug("请求的地址=" + url + "]");
    log.debug("message=[" + requestMessage + "]");
    log.debug("signature=[" + requestSignature + "]");
    log.debug(String.format("请求中金数据:[%s],xml为:[%s],url:[%s]", requestMessage, xml, url));

    ResponseDgtEnvlp responseDgtEnvlp;
    try {
      responseDgtEnvlp = httpsConnection.executeRequest(url, list);
      responseText = responseDgtEnvlp.getResponsetext();
    } catch (Exception e) {
      log.error(String.format("请求中金出现错误,请求数据为%s", xml), e);
      throw new RuntimeException("支付接口请求错误，请联系平台管理员！");
    }
    int index = responseText.indexOf(',');
    if (index > 0) {
      response[0] = responseText.substring(0, index);
      response[1] = responseText.substring(index + 1, responseText.length());
      response[2] = responseDgtEnvlp.getIsDgEnv();
      response[3] = responseDgtEnvlp.getDgtlEnvlp();
      response[4] = responseDgtEnvlp.getSignAlgorithm();
      if (responseDgtEnvlp.getSignSN() != null) {
        response[5] = responseDgtEnvlp.getSignSN().toUpperCase();
      }
      if (responseDgtEnvlp.getEncryptSN() != null) {
        response[6] = responseDgtEnvlp.getEncryptSN().toUpperCase();
      }
    } else {
      log.error("错误的请求返回：{}", responseText);
      String errorMessage = "响应数据格式不正确";
      throw new RuntimeException(errorMessage);
    }
    log.debug(String.format("请求中金返回原始数据为:[%s]", Arrays.toString(response)));
    return response;
  }

  /**
   * 获取请求url地址
   */
  protected String getRequestUrl() {
    CpcnRequestType type = CpcnRequestType.getByCode(this.getCode());
    switch (type.getUrlType()) {
      case PAYMENT_TX:
        return cpcnConfig.getTxUrl();
      case PAYMENT_PAYMENT:
        return cpcnConfig.getPaymentUrl();
      case GATEWAY4FILE:
        return cpcnConfig.getGateway4fileUrl();
      case GATEWAY4AGGREGATE_PAYMENT:
        return cpcnConfig.getGateway4aggregatePaymentUrl();
      case GATEWAY4AGGREGATE_TX:
        return cpcnConfig.getGateway4aggregateTxUrl();
      default:
        throw new RuntimeException("无匹配的URL地址，请检查配置信息");
    }
  }

  /**
   * 返回处理的编号
   */
  public abstract String getCode();

  /**
   * 返回响应的信息类
   */
  public abstract Class<R> getResponse();

}
