package com.biz.crm.common.pay.support.cpcn.service.internal;

import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.constant.Constants;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.enums.CpcnRequestType;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.service.TxService;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.foundationaccount.Tx4693Request;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.foundationaccount.Tx4693Response;
import com.biz.crm.common.pay.support.sdk.model.ResponseModel;
import com.biz.crm.common.pay.support.sdk.service.BankAccountSupportVoService;
import com.biz.crm.common.pay.support.sdk.vo.BankAccountSupportVo;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 银行卡服务
 *
 * @author hefan
 * @date 2022/08/10
 */
@Service
public class BankAccountSupportVoServiceImpl implements BankAccountSupportVoService {

  @Autowired(required = false)
  private TxService<Tx4693Response> tx4693Service;

  /**
   * 查询银行卡绑定情况
   */
  @Override
  public Collection<BankAccountSupportVo> findAll(String mainUserID, String userID) {
    /**
     * 4693-绑定银行账户查询
     * - 可以获取，绑定流水号
     */
    Tx4693Request tx4693Request = this.getTx4693Request(mainUserID, userID);
    // 提交银行账户查询请求
    Tx4693Response tx4693Response = tx4693Service.handlerTx(CpcnRequestType.TX_4693_REQUEST.getCode(), tx4693Request);
    // 返回处理
    return this.parseResponse(tx4693Response);
  }

  /**
   * 解析响应
   *
   * @param response 响应
   * @return {@link ResponseModel}<{@link Collection}<{@link BankAccountSupportVo}>>
   */
  private Collection<BankAccountSupportVo> parseResponse(Tx4693Response response) {
    String code = response.getCode();
    boolean isSuccess = Constants.SUCCESS_CODE.equals(code);
    if (!isSuccess) {
      throw new RuntimeException(response.getMessage());
    }
    Collection<Tx4693Response.BankAccount> bankAccounts = response.getBankAccounts();
    List<BankAccountSupportVo> bankAccountVos = bankAccounts.stream()
        .map(bankAccount -> {
          BankAccountSupportVo vo = new BankAccountSupportVo();
          vo.setBindingTxSN(bankAccount.getBindingTxSN());
          vo.setBankID(bankAccount.getBankID());
          vo.setBankAccountType(bankAccount.getBankAccountType());
          vo.setBankAccountName(bankAccount.getBankAccountName());
          vo.setBankAccountNumber(bankAccount.getBankAccountNumber());
          vo.setBranchName(bankAccount.getBranchName());
          vo.setBankPhoneNumber(bankAccount.getBankPhoneNumber());
          vo.setCardBusinessType(bankAccount.getCardBusinessType());
          vo.setTransferChargeFlag(bankAccount.getTransferChargeFlag());
          vo.setPayeeUserID(bankAccount.getPayeeUserID());
          vo.setPayeeUserName(bankAccount.getPayeeUserName());
          return vo;
        })
        .collect(Collectors.toList());
    return bankAccountVos;
  }

  /**
   * 得到tx4693请求
   *
   * @return {@link Tx4693Request}
   */
  private Tx4693Request getTx4693Request(String mainUserID, String userID) {
    Tx4693Request tx4693Request = new Tx4693Request();
    // R 必填
    tx4693Request.setTxCode(CpcnRequestType.TX_4693_REQUEST.getCode());
    tx4693Request.setMainUserID(mainUserID);
    tx4693Request.setUserID(userID);
    return tx4693Request;
  }

  @Override
  public BankAccountSupportVo findByBankAccountNumber(String bankAccountNumber, String mainUserID, String userID) {
    Collection<BankAccountSupportVo> bankAccountSupportVos = this.findAll(mainUserID, userID);
    Validate.isTrue(!CollectionUtils.isEmpty(bankAccountSupportVos), "没有查询到可用的银行账户");
    Set<BankAccountSupportVo> bankAccountSet = bankAccountSupportVos.stream()
        .filter(account -> !account.getBankAccountNumber().equals(bankAccountNumber))
        .collect(Collectors.toSet());
    Validate.isTrue(!CollectionUtils.isEmpty(bankAccountSet), "该银行账户不在中金账户的绑定列表中");
    Validate.isTrue(bankAccountSet.size() == 1, "查询到多个银行账户，不是一个唯一确定的银行账户是无法提现的");
    Optional<BankAccountSupportVo> first = bankAccountSet.stream().findFirst();
    BankAccountSupportVo bankAccountSupportVo = first.get();
    return bankAccountSupportVo;
  }


}
