package com.biz.crm.common.pay.support.cpcn.strategy.transfer.convertor;

import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.constant.Constants;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.enums.CpcnBillTransactionStatus;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.config.CpcnConfig;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.aggregate.Tx5011Request;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.tx.aggregate.Tx5011Response;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.vo.SplitItem;
import com.biz.crm.common.pay.support.cpcn.model.Tx5011ScanPayModel;
import com.biz.crm.common.pay.support.sdk.dto.transfer.PayDto;
import com.biz.crm.common.pay.support.sdk.model.ResponseModel;
import com.biz.crm.common.pay.support.sdk.strategy.transfer.carrier.Scan7Aggregate7Forward;
import com.biz.crm.common.pay.support.sdk.strategy.transfer.carrier.TransferCarrier;
import com.biz.crm.common.pay.support.sdk.strategy.transfer.convertor.TransferConvertor;
import com.biz.crm.common.pay.support.sdk.vo.transfer.QRCodeVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;

/**
 * 转账转换器
 * scan - 条码支付
 * aggregate - 聚合码
 * forward - 正向
 * 数字 7 - 因为&符号在7上面，代表and的意思
 *
 * @author hefan
 * @date 2022/08/25
 */
public class Scan7Aggregate7ForwardConvertor implements TransferConvertor<Tx5011Request, Tx5011Response> {

  @Autowired(required = false)
  private CpcnConfig cpcnConfig;

  @Override
  public Boolean isConvertible(TransferCarrier carrier) {
    return carrier instanceof Scan7Aggregate7Forward;
  }

  @Override
  public Tx5011Request convert(TransferCarrier carrier) {
    Validate.isTrue(this.isConvertible(carrier), StringUtils.join("条码支付", "转换器", "不可转换"));
    Scan7Aggregate7Forward carrierImpl = (Scan7Aggregate7Forward) carrier;
    // 载体全量数据
    PayDto transferData = carrierImpl.getTransferData();
    // 设置中金参数匹配具体的转账行为
    Tx5011ScanPayModel model = this.getScanPayModel(transferData);
    // 转换为中金base模块可识别的业务对象
    Tx5011Request tx5011Request = this.getTx5011Request(model);
    return tx5011Request;
  }

  @Override
  public ResponseModel<QRCodeVo> parseResponse(Tx5011Response response) {
    String code = response.getCode();
    boolean isSuccess = Constants.SUCCESS_CODE.equals(code);
    if (!isSuccess) {
      throw new RuntimeException(response.getMessage());
    }
    ResponseModel responseModel = new ResponseModel();
    responseModel.setSuccess(true);
    String status = response.getStatus();
    responseModel.setMessage(response.getResponseMessage());
    boolean haveUrl = CpcnBillTransactionStatus.PROCESSING.getCode().equals(status);
    if (haveUrl) {
      String qrCodeURL = response.getQRCodeURL();
      String qRImageURL = response.getQRImageURL();
      QRCodeVo QRCodeVo = new QRCodeVo();
      responseModel.setResponse(QRCodeVo);
      QRCodeVo.setQRCodeURL(qrCodeURL);
      QRCodeVo.setQRImageURL(qRImageURL);
    }
    return responseModel;

  }

  /**
   * 5011-支付（API）
   * - 条码支付&聚合码&正扫 需要的传参
   *
   * @param dto 传输数据
   * @return {@link Tx5011ScanPayModel}
   */
  private Tx5011ScanPayModel getScanPayModel(PayDto dto) {

    Tx5011ScanPayModel model = new Tx5011ScanPayModel();
    // 42=条码支付
    model.setPaymentWay("42");
    // 40=聚合码
    model.setScanPaymentType("40");
    // 41=正扫
    model.setScanPaymentWay("41");
    // 信用卡限制标识
    if (!dto.getQRLimitPay()) {
      model.setScanPaymentLimitFlag("20");
    }
    model.setGoodsName(dto.getGoodsName());
    model.setCancelPayRedirectPageURL(dto.getCancelPayRedirectPageURL());
    model.setScanPageUrlType(dto.getQRPageUrlType());
    model.setRedirectPageURL(dto.getRedirectPageURL());
    model.setTxSN(dto.getTxSN());
    model.setOrderNo(dto.getOrderNo());
    model.setPayerUserID(dto.getPayerUserID());
    model.setPayeeAccountNumber(dto.getPayeeAccountNumber());
    model.setPayeeUserID(dto.getPayeeUserID());
    model.setAmount(dto.getAmount());
    model.setExpirePeriod(dto.getExpirePeriod());
    model.setPageURL(cpcnConfig.getNoticeUrl());
    model.setHasSubsequentSplit("2");
    model.setNoticeURL(cpcnConfig.getNoticeUrl());
    model.setRemark(dto.getRemark());
    return model;
  }


  /**
   * 条码支付的请求
   * - 按照中金文档，条码支付该做的传值校验都做了。
   * - 这是以paymentWay为维度转参
   *
   * @param model 条码支付的请求
   * @return {@link Tx5011Request}
   */
  private Tx5011Request getTx5011Request(Tx5011ScanPayModel model) {
    Tx5011Request tx5011Request = new Tx5011Request();
    // R
    tx5011Request.setTxSN(model.getTxSN());
    // 付款userID：经销商的userID
    tx5011Request.setPayerUserID(model.getPayerUserID());
    // 收款userID：品牌商的userID
    tx5011Request.setPayeeUserID(model.getPayeeUserID());
    tx5011Request.setPaymentWay(model.getPaymentWay());
    tx5011Request.setAmount(model.getAmount());
    tx5011Request.setExpirePeriod(model.getExpirePeriod());
    tx5011Request.setPageURL(model.getPageURL());
    tx5011Request.setHasSubsequentSplit(model.getHasSubsequentSplit());
    // O
    tx5011Request.setOrderNo(model.getOrderNo());
    tx5011Request.setNoticeURL(model.getNoticeURL());
    tx5011Request.setRemark(model.getRemark());
    // C
    // 因为我们都是机构内转账，所以这里也要填
    tx5011Request.setPayeeAccountNumber(model.getPayeeAccountNumber());
    // 条码支付（必填）商品名称
    tx5011Request.setGoodsName(model.getGoodsName());

    // 后续分账：没有时（HasSubsequentSplit=1）必填 ： SplitItems {
    if ("1".equals(model.getHasSubsequentSplit())) {
      List<SplitItem> splitItems = new ArrayList<>();
      // fixme: 需要的人去实现
      tx5011Request.setSplitItemsList(splitItems);
    }
    // }

    // O
    //tx5011Request.setTerminalUserID();
    //tx5011Request.setExtension();

    // C
    //tx5011Request.setSourceTxTime();
    //tx5011Request.setPlatformName();
    //tx5011Request.setClientIP();
    //tx5011Request.setDeductionSettlementFlag();

    // 支付方式：条码支付 ：ScanPay {
    // - R
    tx5011Request.setScanPaymentType(model.getScanPaymentType());
    tx5011Request.setScanPaymentWay(model.getScanPaymentWay());
    // - C
    tx5011Request.setPaymentScene(model.getPaymentScene());
    // 刷脸支付 必填
    if ("90".equals(model.getPaymentScene())) {
      tx5011Request.setPreTxSN(model.getPreTxSN());
    }
    // 反扫 必填
    if ("42".equals(model.getScanPaymentWay())) {
      tx5011Request.setScanPaymentCode(model.getScanPaymentCode());
    }
    tx5011Request.setScanPaymentLimitFlag(model.getScanPaymentLimitFlag());
    tx5011Request.setInstallmentType(model.getInstallmentType());
    tx5011Request.setFeeMode(model.getFeeMode());
    tx5011Request.setNumberOfInstallments(model.getNumberOfInstallments());
    tx5011Request.setTerminalID(model.getTerminalID());
    // - O
    tx5011Request.setAccountType(model.getAccountType());
    tx5011Request.setCancelPayRedirectPageURL(model.getCancelPayRedirectPageURL());
    tx5011Request.setScanPageUrlType(model.getScanPageUrlType());
    tx5011Request.setRedirectPageURL(model.getRedirectPageURL());
    tx5011Request.setIdentityInfo(model.getIdentityInfo());
    tx5011Request.setSubAppID(model.getSubAppID());
    tx5011Request.setBankID(model.getBankID());
    // }
    return tx5011Request;
  }

}
