package com.biz.crm.common.pay.support.cpcn.base.common.http.internal;

import com.biz.crm.common.pay.support.cpcn.base.common.http.HttpData;
import com.biz.crm.common.pay.support.cpcn.base.common.http.HttpsConnection;
import com.biz.crm.common.pay.support.cpcn.base.common.http.IgnoreHostnameVerifier;
import com.biz.crm.common.pay.support.cpcn.base.common.http.NameValuePair;
import com.biz.crm.common.pay.support.cpcn.base.common.http.ResponseDgtEnvlp;
import com.biz.crm.common.pay.support.cpcn.base.common.http.SecurityContext;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import javax.net.ssl.HttpsURLConnection;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ProtocolException;
import java.net.URL;
import java.util.List;

/**
 * java默认网络库实现网络https请求
 *
 * @author Keller
 */
@Slf4j
@Setter
public class SimpleHttpsConnection implements HttpsConnection {
  /**
   * 网络请求方法
   */
  private String method = "POST";
  /**
   * 网络请求连接超时
   */
  private int connectTimeout = 50000;
  /**
   * 网络请求读取数据超时
   */
  private int readTimeout = 50000;
  /**
   * 输出字符编码
   */
  private String outputCharset = "UTF-8";
  /**
   * 输入字符编码
   */
  private String inputCharset = "UTF-8";
  /**
   * 是否使用默认的ssl厂库方法
   */
  private boolean useDefaultSSLSocketFactory = true;
  /**
   * 请求头媒体类型
   */
  private String contentType = "application/x-www-form-urlencoded";
  /**
   * 忽略
   */
  private boolean ignoreHostname = true;

  /**
   * 通过list形式参数执行请求
   *
   * @param list
   * @return
   */
  @Override
  public String execute(String uri, List<NameValuePair> list) {
    HttpData httpData = new HttpData(list, this.outputCharset);
    String request = httpData.getData();
    String response = execute(uri, request);
    return response;
  }

  @Override
  public ResponseDgtEnvlp executeRequest(String uri, List<NameValuePair> list) {
    return null;
  }

  @Override
  public String execute(String uri, String request) {
    log.info("请求uri: '{}', 请求参数'{}'", uri, request);
    HttpsURLConnection httpsUrlConnection;
    try {
      URL url = new URL(uri);
      httpsUrlConnection = (HttpsURLConnection) url.openConnection();
    } catch (IOException e) {
      throw new RuntimeException("初始化网络连接错误", e);
    }
    if (!this.useDefaultSSLSocketFactory) {
      httpsUrlConnection.setSSLSocketFactory(SecurityContext.sslSocketFactory);
    }
    if (this.ignoreHostname) {
      httpsUrlConnection.setHostnameVerifier(new IgnoreHostnameVerifier());
    }
    httpsUrlConnection.setRequestProperty("Content-Type", this.contentType);
    httpsUrlConnection.setRequestProperty("Accept-Charset", "UTF-8");
    httpsUrlConnection.setDoOutput(true);
    httpsUrlConnection.setDoInput(true);
    try {
      httpsUrlConnection.setRequestMethod(this.method);
    } catch (ProtocolException e) {
      throw new IllegalArgumentException(String.format("不支持的请求类型[%s]", this.method), e);
    }
    httpsUrlConnection.setUseCaches(false);
    httpsUrlConnection.setConnectTimeout(this.connectTimeout);
    httpsUrlConnection.setReadTimeout(this.readTimeout);
    InputStream inputStream = null;
    OutputStream outputStream = null;
    String response;
    try{
      outputStream = httpsUrlConnection.getOutputStream();
      outputStream.write(request.getBytes(this.outputCharset));
      outputStream.flush();
      outputStream.close();

      inputStream = httpsUrlConnection.getInputStream();
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      byte[] buffer = new byte[1024];
      int num = inputStream.read(buffer);
      while (num != -1) {
        baos.write(buffer, 0, num);
        num = inputStream.read(buffer);
      }
      baos.flush();
      response = (baos.toString(this.inputCharset)).trim();
      inputStream.close();
    } catch (IOException e) {
      log.error("发送请求出错", e);
      throw new RuntimeException("发送请求出错", e);
    } finally {
      if(outputStream !=null){
        try {
          outputStream.close();
        } catch (IOException e) {
          throw new RuntimeException(e.getMessage());
        }
      }
      if(inputStream !=null){
        try {
          inputStream.close();
        } catch (IOException e) {
          throw new RuntimeException(e.getMessage());
        }
      }
      if(httpsUrlConnection != null){
        httpsUrlConnection.disconnect();
      }
    }
    return response;
  }

  public boolean isUseDefaultSSLSocketFactory() {
    return useDefaultSSLSocketFactory;
  }

  public void setUseDefaultSSLSocketFactory(boolean useDefaultSSLSocketFactory) {
    this.useDefaultSSLSocketFactory = useDefaultSSLSocketFactory;
  }
}
