package com.biz.crm.common.pay.support.cpcn.base.cpcn.strategy.remote.internal;

import com.biz.crm.common.pay.support.cpcn.base.common.http.HttpData;
import com.biz.crm.common.pay.support.cpcn.base.common.http.NameValuePair;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.common.enums.CpcnNoticeType;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.configuration.CpcnProperties;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.strategy.remote.RemoteStrategy;
import com.biz.crm.common.pay.support.cpcn.base.cpcn.vo.NoticeInfo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.Validate;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.List;

/**
 * 4618-开户/绑卡结果通知
 * <pre>
 *   支付平台收到机构系统发送的4601-开户和4611-绑卡/4613-绑卡确认的交易请求后会进行处理，
 *   如果机构系统实时收到4601或4613接口的响应中交易状态不是“30=成功”或“40=失败”，
 *   那么后续当交易状态变更为“18=被动 已打款待验证”、“30=成功”、“40=失败”三种状态之一时，
 *   支付平台会给机构系统发送该笔交易的结果通知(优先向 4601 或 4611 接口上送的后台通知地址发送，
 *   4601 或 4611 接口没有上送则向机构系统在上线申请时提供的后台通知地址发送)。
 *   机构系统收到该通知后需要给支付平台返回特定的响应，
 *   如果支付平台在5 秒内未收到机构系统返回的通知响应或接收到的响应内容与规定内容不符，
 *   支付平台会认为机构系统未成功收到该通知，这时支付平台会向机构系统补发通知，最多补发7次，共发8次，
 *   各次发送时间间隔为: 0s/30s/30s/3m/10m/20m/30m/1h。
 * </pre>
 * @author dy
 * @date 2022/06/28
 */
@Slf4j
@Component
public class Remote4618Strategy implements RemoteStrategy {

  @Autowired
  private CpcnProperties cpcnProperties;
  @Autowired
  private CloseableHttpClient client;

  @Override
  public String getCode() {
    return CpcnNoticeType.NOTICE_4618.getCode();
  }

  @Override
  @Async("defaultRemoteNoticeExecutor")
  public void handler(NoticeInfo info) {
    if (ArrayUtils.isNotEmpty(cpcnProperties.getRemoteUrl())) {
      NameValuePair message = new NameValuePair("message", info.getMessage());
      NameValuePair signature = new NameValuePair("signature", info.getSignature());
      List<NameValuePair> params = Lists.newArrayList(message, signature);
      String[] urls = cpcnProperties.getRemoteUrl();
      for (String remoteUrl : urls) {
        try {
          String response = execute(remoteUrl, params);
          log.info("调用应用接口 :'{}',返回结果为:'{}'", remoteUrl, response);
        } catch (Exception e) {
          log.error(e.getMessage(), e);
          continue;
        }
      }
    } else {
      log.info("未配置转发地址，4658响应请求未转发");
    }
  }

  /**
   * 执行回调信息通知
   *
   * @param uri
   * @param list
   * @return
   */
  private String execute(String uri, List<NameValuePair> list) {
    HttpData httpData = new HttpData(list, "UTF-8");
    String request = httpData.getData();
    HttpPost httpPost = new HttpPost(uri);
    BasicHeader basicHeader = new BasicHeader("Content-Type", "application/x-www-form-urlencoded");
    httpPost.addHeader(basicHeader);
    String response = null;
    HttpEntity httpEntity = null;
    try {
      StringEntity stringEntity = new StringEntity(request, Charset.forName("UTF-8"));
      httpPost.setEntity(stringEntity);
      CloseableHttpResponse closeableHttpResponse = client.execute(httpPost);
      httpEntity = closeableHttpResponse.getEntity();
      Validate.notNull(httpEntity, "httpclient请求返回为空，请检查请求地址！");
      response = EntityUtils.toString(httpEntity, Charset.forName("UTF-8"));
    } catch (IOException e) {
      log.error("httpclient 请求错误", e);
      throw new RuntimeException(e);
    }
    return response;
  }
}
