package com.biz.crm.common.pay.support.cpcn.base.common.utils;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

/**
 * xml工具类 该类主要
 *
 * @author Keller
 */
@Slf4j
@UtilityClass
public class XmlUtils {

  /**
   * xml schema创建
   *
   * @param schemaFile
   * @return
   * @throws Exception
   */
  public Schema createSchema(File schemaFile) throws Exception {
    SchemaFactory schemaFactory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
    return schemaFactory.newSchema(schemaFile);
  }

  /**
   * 验证xml文档的schema dom解析
   *
   * @param document
   * @param schema
   * @throws Exception
   */
  public static void validateViaDOM(Document document, Schema schema) throws Exception {
    DOMSource source = new DOMSource(document);
    Validator validator = schema.newValidator();
    validator.validate(source);
  }

  /**
   * 验证xml文档的schema dom解析
   *
   * @param xmlString
   * @param schema
   * @throws Exception
   */
  public static void validateViaDOM(String xmlString, Schema schema) throws Exception {
    DOMSource source = new DOMSource(createDocument(xmlString));
    Validator validator = schema.newValidator();
    validator.validate(source);
  }

  /**
   * 验证xml文档的schema sax解析
   *
   * @param xmlString
   * @param schema
   * @throws Exception
   */
  public static void validateViaSAX(String xmlString, Schema schema) throws Exception {
    SAXSource source = new SAXSource(new InputSource(new InputStreamReader(new ByteArrayInputStream(xmlString.getBytes(StandardCharsets.UTF_8.name())))));
    Validator validator = schema.newValidator();
    validator.validate(source);
  }

  public static void validateViaStreamSource(String xmlString, Schema schema) throws Exception {
    StreamSource source = new StreamSource(new ByteArrayInputStream(xmlString.getBytes(StandardCharsets.UTF_8)));
    Validator validator = schema.newValidator();
    validator.validate(source);
  }

  /**
   * 通过xml字符串形式创建xml的document对象
   *
   * @param xmlString
   * @return
   * @throws Exception
   */
  public static Document createDocument(String xmlString) throws Exception {
    DocumentBuilder documentBuilder = createDocumentBuilder();
    return documentBuilder.parse(new ByteArrayInputStream(xmlString.getBytes(StandardCharsets.UTF_8)));
  }

  /**
   * 通过xml字符串形式创建xml的document对象
   *
   * @param xmlString
   * @param charset
   * @return
   * @throws Exception
   */
  public static Document createDocument(String xmlString, String charset) throws Exception {
    DocumentBuilder documentBuilder = createDocumentBuilder();
    return documentBuilder.parse(new ByteArrayInputStream(xmlString.getBytes(charset)));
  }

  /**
   * 通过xml文件创建xml的document对象
   *
   * @param xmlFile
   * @return
   * @throws Exception
   */
  public static Document createDocument(File xmlFile) throws Exception {
    DocumentBuilder documentBuilder = createDocumentBuilder();
    return documentBuilder.parse(xmlFile);
  }

  /**
   * 创建DocumentBuilder
   *
   * @return
   * @throws Exception
   */
  public static DocumentBuilder createDocumentBuilder() throws Exception {
    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
    return documentBuilderFactory.newDocumentBuilder();
  }

  /**
   * 获取xml的document节点文本信息
   *
   * @param document
   * @param nodeName
   * @return
   * @throws Exception
   */
  public static String getNodeText(Document document, String nodeName) throws Exception {
    return getNodeText(document, nodeName, 0);
  }

  /**
   * 按照节点名称获取节点文本信息
   *
   * @param document
   * @param nodeName
   * @param index
   * @return
   * @throws Exception
   */
  public static String getNodeText(Document document, String nodeName, int index) throws Exception {
    NodeList nodeList = document.getElementsByTagName(nodeName);
    if (nodeList == null || index >= nodeList.getLength()) {
      return null;
    }
    return nodeList.item(index).getTextContent();
  }

  /**
   * 按照节点名称获取节点文本信息
   *
   * @param xmlString
   * @param nodeName
   * @return
   * @throws Exception
   */
  public static String getNodeText(String xmlString, String nodeName) throws Exception {
    Document document = createDocument(xmlString);
    return getNodeText(document, nodeName);
  }

  /**
   * 按照节点名称获取子节点文本信息
   *
   * @param node
   * @param childName
   * @return
   */
  public static String getChildNodeText(Node node, String childName) {
    NodeList children = node.getChildNodes();
    int len = children.getLength();
    for (int i = 0; i < len; i++) {
      Node child = children.item(i);
      if (child.getNodeType() == 1 && child.getNodeName() == childName)
        return child.getTextContent();
    }
    return null;
  }

  /**
   * 按照节点名称获取节点信息
   *
   * @param document
   * @param nodeName
   * @return
   * @throws Exception
   */
  public static List<String> getNodeList(Document document, String nodeName) throws Exception {
    NodeList nodeList = document.getElementsByTagName(nodeName);
    List<String> textList = new ArrayList<String>();
    if (nodeList == null) {
      return null;
    }
    for (int i = 0; i < nodeList.getLength(); i++) {
      textList.add(nodeList.item(i).getTextContent());
    }
    return textList;
  }

  /**
   * 按照节点名称以及节点属性名称获取属性值
   *
   * @param document
   * @param nodeName
   * @param attributeName
   * @return
   */
  public static String getNodeAttributeValue(Document document, String nodeName, String attributeName) {
    NodeList nodeList = document.getElementsByTagName(nodeName);
    if (nodeList != null && nodeList.getLength() > 0) {
      Element element = (Element) nodeList.item(0);
      return element.getAttribute(attributeName);
    }
    return null;
  }

  /**
   * 按照节点名称获取子节点文本信息
   *
   * @param node
   * @param childName
   * @return
   */
  public static String getNodeText(Node node, String childName) {
    NodeList children = node.getChildNodes();
    int len = children.getLength();
    for (int i = 0; i < len; i++) {
      Node child = children.item(i);
      if (child.getNodeType() == 1 && child.getNodeName().equals(childName)) {
        return child.getTextContent();
      }
    }
    return null;
  }

  /**
   * 按照节点名称获取子节点对象息
   *
   * @param node
   * @param childName
   * @return
   */
  public static Node getChildNode(Node node, String childName) {
    NodeList children = node.getChildNodes();
    int len = children.getLength();
    for (int i = 0; i < len; i++) {
      Node child = children.item(i);
      if (child.getNodeType() == 1 && child.getNodeName().equals(childName)) {
        return child;
      }
    }
    return null;
  }

  /**
   * 转换xml的document对象至字符串（格式化后） 默认utf-8
   *
   * @param document
   * @return
   */
  public static String createPrettyFormat(Document document) {
    return createPrettyFormat(document, StandardCharsets.UTF_8.toString());
  }

  /**
   * 转换xml的document对象至字符串,支持定义输出的字符编码格式
   *
   * @param document
   * @param outputCharset
   * @return
   */
  public static String createPrettyFormat(Document document, String outputCharset) {
    String result;
    Source source = new DOMSource(document);
    TransformerFactory transformerFactory = TransformerFactory.newInstance();
    Transformer transformer = null;
    try {
      transformer = transformerFactory.newTransformer();
      transformer.setOutputProperty("indent", "yes");
      transformer.setOutputProperty("encoding", outputCharset);
      Writer writer = new StringWriter();
      transformer.transform(source, new StreamResult(writer));
      result = writer.toString();
    } catch (TransformerException e) {
      log.error("XML Doc对象转换字符串错误", e);
      throw new RuntimeException("XML Doc对象转换字符串错误", e);
    }
    return result;
  }

  /**
   * 转换xml的document对象至字符串 默认utf-8
   *
   * @param document
   * @return
   */
  public static String createCompactFormat(Document document) {
    return createCompactFormat(document, StandardCharsets.UTF_8.toString());
  }

  /**
   * 转换xml的document对象至字符串,支持定义输出的字符编码格式
   *
   * @param document
   * @param outputCharset
   * @return
   */
  public static String createCompactFormat(Document document, String outputCharset) {
    String result;
    Source source = new DOMSource(document);
    TransformerFactory transformerFactory = TransformerFactory.newInstance();
    Transformer transformer = null;
    try {
      transformer = transformerFactory.newTransformer();
      transformer.setOutputProperty("encoding", outputCharset);
      Writer writer = new StringWriter();
      transformer.transform(source, new StreamResult(writer));
      result = writer.toString();
    } catch (TransformerException e) {
      log.error("XML Doc对象转换字符串错误", e);
      throw new RuntimeException("XML Doc对象转换字符串错误", e);
    }
    return result;
  }
}
