package com.biz.crm.common.rulecode.local.service.internal;


import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.common.rulecode.sdk.constant.GenerateCodeRuleConstant;
import com.biz.crm.common.rulecode.sdk.enums.RuleCodeRedisEnum;
import com.biz.crm.common.rulecode.sdk.service.GenerateCodeRuleVoService;
import com.biz.crm.common.rulecode.sdk.vo.GenerateCodeRuleVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 编码规则接口实现
 *
 * @author XXLsansui
 * @date 2024-01-24 10:14
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "GenerateCodeRuleVoServiceExpandImpl")
public class GenerateCodeRuleVoServiceImpl implements GenerateCodeRuleVoService {


    @Value("${spring.application.name:}")
    private String applicationName;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private RedisService redisService;

    /**
     * 生成多个编码  不要动,不要动,不要动
     *
     * @param ruleCode 编码规则
     * @param number   生成数量
     * @return
     */
    @Override
    public List<String> generateCode(String ruleCode, long number) {
        Assert.hasLength(ruleCode, "编码规则不能为空");
        Assert.isTrue(number >= 1, "至少生成一个编码");
        String redisKey = String.format(GenerateCodeRuleConstant.GENERATE_CODE_RULE_LOCK_PREFIX, TenantUtils.getTenantCode(), ruleCode);
        boolean locked = false;
        try {
            locked = redisMutexService.tryLock(redisKey, 15, 15, TimeUnit.SECONDS);
            Validate.isTrue(locked, "生成编码规则繁忙请稍后再试！");
            List<String> list = new ArrayList<>();
            Object obj = redisService.hGet(RuleCodeRedisEnum.GEN_VO.getVal(), ruleCode);
            Assert.notNull(obj, "编码规则[" + ruleCode + "]不存在,或未同步到缓存,请联系管理员!");
            GenerateCodeRuleVo ruleVo = (GenerateCodeRuleVo) obj;
            Assert.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(ruleVo.getDelFlag()), "编码规则[" + ruleCode + "]已被逻辑删除!");
            Assert.isTrue(EnableStatusEnum.ENABLE.getCode().equals(ruleVo.getEnableStatus()), "编码规则[" + ruleCode + "]已被禁用");

            //当前时间
            Date now = new Date();
            String prefix = StringUtils.stripToEmpty(ruleVo.getPrefix());
            String dateStr = "";
            if (StringUtils.isNotEmpty(ruleVo.getDateFormat())) {
                SimpleDateFormat sdf = new SimpleDateFormat(ruleVo.getDateFormat());
                dateStr = sdf.format(now);
            }
            int expirationTimes = 0;
            TimeUnit timeUnit = TimeUnit.DAYS;
            boolean restCurrentValueFormat = false;
            if (StringUtils.isNotEmpty(dateStr) && BooleanEnum.TRUE.getCapital().equals(ruleVo.getRestCurrentValueFormat())) {
                Assert.isTrue(GenerateCodeRuleConstant.DATE_FORMAT_SET.contains(ruleVo.getDateFormat()), "编码规则[" + ruleCode + "]的时间格式不正确!");
                restCurrentValueFormat = true;
                switch (ruleVo.getDateFormat()) {
                    case "yyyyMMdd":
                        expirationTimes = 25;
                        timeUnit = TimeUnit.HOURS;
                        break;
                    case "yyyyMM":
                        expirationTimes = 32;
                        break;
                    case "yyyy":
                        expirationTimes = 370;
                        break;
                    default:
                        throw new IllegalArgumentException("编码规则[" + ruleCode + "]的时间格式不正确!");

                }
            }
            Long currentValue = null;
            if (restCurrentValueFormat) {
                String ruleKey = String.format(RuleCodeRedisEnum.RULE_CODE_DATE_FORMAT.getVal(), ruleCode, dateStr);
                currentValue = redisService.incr(ruleKey, number);
                redisService.expire(ruleKey, expirationTimes, timeUnit);
            } else {
                currentValue = redisService.hIncr(RuleCodeRedisEnum.GEN_KEY.getVal(), ruleCode, number);
            }
            int numberLength = ruleVo.getCodeLength() - prefix.length() - dateStr.length();
            long begin = currentValue - number;
            for (int i = 1; i <= number; i++) {
                begin++;
                list.add(prefix + dateStr + String.format("%0" + numberLength + "d", begin));
            }
            ruleVo.setCurrentValue(currentValue);
            //记录最后一次生成时间
            ruleVo.setGenerateDate(now);
            ruleVo.setRuleModule(applicationName);
            redisService.hSet(RuleCodeRedisEnum.GEN_VO.getVal(), ruleVo.getRuleCode(), ruleVo);
            return list;
        } catch (Exception e) {
            log.error("生成编码规则出错：{}", ruleCode, e);
            throw new IllegalArgumentException(e.getMessage());
        } finally {
            if (locked && redisMutexService.islock(redisKey)) {
                redisMutexService.unlock(redisKey);
            }
        }
    }

    /**
     * 根据编码规则获取一个编码
     *
     * @param ruleCode
     * @return
     */
    @Override
    public String generateCode(String ruleCode) {
        Validate.isTrue(StringUtils.isNotEmpty(ruleCode), "编码规则不能为空");
        List<String> list = this.generateCode(ruleCode, 1L);
        if (ObjectUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return "";
    }
}
