package com.biz.crm.common.rulecode.local.service.internal;


import com.biz.crm.common.rulecode.local.entity.GenerateCodeRuleEntity;
import com.biz.crm.common.rulecode.local.repository.GenerateCodeRuleRepository;
import com.biz.crm.common.rulecode.sdk.constant.GenerateCodeRuleConstant;
import com.biz.crm.common.rulecode.sdk.service.GenerateCodeRuleVoService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

/**
 * 编码规则接口实现
 *
 * @author XXLsansui
 * @date 2024-01-24 10:14
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "GenerateCodeRuleVoServiceExpandImpl")
public class GenerateCodeRuleVoServiceImpl implements GenerateCodeRuleVoService {

  @Autowired(required = false)
  private GenerateCodeRuleRepository generateCodeRuleRepository;
  @Autowired
  private RedisMutexService redisMutexService;

  /**
   * 生成多个编码
   *
   * @param ruleCode 编码规则
   * @param number   生成数量
   * @return
   */
  @Override
  public List<String> generateCodeList(String ruleCode, Integer number) {
    Validate.isTrue(StringUtils.isNotEmpty(ruleCode), "编码规则不能为空");
    Validate.isTrue(ObjectUtils.isNotEmpty(number), "编码个数不能为空");
    Validate.isTrue(number >= 1, "至少生成一个编码");
    String redisKey = String.format(GenerateCodeRuleConstant.GENERATE_CODE_RULE_LOCK_PREFIX, TenantUtils.getTenantCode(), ruleCode);
    boolean locked = false;
    try {
      locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
      Validate.isTrue(locked, "生成编码规则繁忙请稍后再试！");
      List<String> list = new ArrayList<>();
      GenerateCodeRuleEntity rule = this.generateCodeRuleRepository.findByCode(ruleCode);
      Validate.isTrue(ObjectUtils.isNotEmpty(rule), "未找到编码规则：" + ruleCode);

      //当前时间
      Date now = new Date();
      String prefix = rule.getPrefix() == null ? "" : rule.getPrefix();
      String dateStr = "";
      if (StringUtils.isNotEmpty(rule.getDateFormat())) {
        SimpleDateFormat sdf = new SimpleDateFormat(rule.getDateFormat());
        dateStr = sdf.format(now);
      }
      Long currentValue = rule.getCurrentValue();
      int numberLength = rule.getCodeLength() - prefix.length() - dateStr.length();
      for (int i = 1; i <= number; i++) {
        currentValue++;
        list.add(prefix + dateStr + String.format("%0" + numberLength + "d", currentValue));
      }
      rule.setCurrentValue(currentValue);

      //记录最后一次生成时间
      rule.setGenerateDate(now);
      this.generateCodeRuleRepository.updateById(rule);
      return list;
    } catch (Exception e) {
      log.error("生成编码规则出错：{}, {}", ruleCode, e);
      throw new IllegalArgumentException(e.getMessage());
    } finally {
      if (locked && redisMutexService.islock(redisKey)) {
        redisMutexService.unlock(redisKey);
      }
    }
  }

  /**
   * 根据编码规则获取一个编码
   *
   * @param ruleCode
   * @return
   */
  @Override
  public String generateCode(String ruleCode) {
    Validate.isTrue(StringUtils.isNotEmpty(ruleCode), "编码规则不能为空");
    List<String> list = this.generateCodeList(ruleCode, 1);
    if (ObjectUtils.isNotEmpty(list)) {
      return list.get(0);
    }
    return "";
  }
}
