package com.biz.crm.common.rulecode.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.rulecode.local.entity.GenerateCodeRuleEntity;
import com.biz.crm.common.rulecode.local.repository.GenerateCodeRuleRepository;
import com.biz.crm.common.rulecode.local.service.GenerateCodeRuleService;
import com.biz.crm.common.rulecode.sdk.dto.GenerateCodeRuleDto;
import com.biz.crm.common.rulecode.sdk.vo.GenerateCodeRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 编码规则接口实现
 *
 * @author XXLsansui
 * @date 2024-01-24 10:14
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "GenerateCodeRuleServiceExpandImpl")
public class GenerateCodeRuleServiceImpl implements GenerateCodeRuleService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeRuleRepository generateCodeRuleRepository;

  /**
   * 分页条件查询
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<GenerateCodeRuleEntity> findByConditions(Pageable pageable, GenerateCodeRuleDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new GenerateCodeRuleDto());
    Page<GenerateCodeRuleEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    return this.generateCodeRuleRepository.findByConditions(page, dto);
  }

  /**
   * 查看规则详情
   *
   * @param id ID
   * @return
   */
  @Override
  public GenerateCodeRuleVo findById(String id) {
    Validate.isTrue(StringUtils.isNotEmpty(id), "请选择规则编码详情id!");
    GenerateCodeRuleEntity entity = this.generateCodeRuleRepository.findById(id);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 创建
   *
   * @param dto 请求参数dto
   * @return
   */
  @Override
  @Transactional
  public GenerateCodeRuleVo create(GenerateCodeRuleDto dto) {
    Validate.isTrue(ObjectUtils.isNotEmpty(dto), "新增编码规则参数不能为空!");
    //校验新增参数
    this.createVerify(dto);
    GenerateCodeRuleEntity oldEntity = this.generateCodeRuleRepository.findByCode(dto.getRuleCode());
    Validate.isTrue(ObjectUtils.isEmpty(oldEntity), "编码规则key已存在");
    GenerateCodeRuleEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, GenerateCodeRuleEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.generateCodeRuleRepository.save(entity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 更新
   *
   * @param dto 请求参数dto
   * @return
   */
  @Override
  @Transactional
  public GenerateCodeRuleVo update(GenerateCodeRuleDto dto) {
    Validate.isTrue(ObjectUtils.isNotEmpty(dto), "编辑编码规则参数不能为空!");
    Validate.isTrue(StringUtils.isNotEmpty(dto.getId()), "id不能为空!");
    this.createVerify(dto);
    GenerateCodeRuleEntity oldEntity = this.generateCodeRuleRepository.findById(dto.getId());
    Validate.isTrue(ObjectUtils.isNotEmpty(oldEntity), "当前id对应编码规则不存在!");
    if (ObjectUtils.isNotEmpty(dto.getCurrentValue())) {
      Validate.isTrue(dto.getCurrentValue() >= oldEntity.getCurrentValue(), "当前值不能改小!");
    }
    Validate.isTrue(dto.getRuleCode().equals(oldEntity.getRuleCode()), "编码规则key不能修改");
    GenerateCodeRuleEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, GenerateCodeRuleEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.generateCodeRuleRepository.saveOrUpdate(entity);
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 批量删除编码规则
   *
   * @param ids 编码规则id集合
   */
  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(ObjectUtils.isNotEmpty(ids), "请选择要删除的编码规则");
    this.generateCodeRuleRepository.deleteBatch(ids);
  }

  /**
   * 重置
   *
   * @param ids 编码规则集合
   */
  @Override
  @Transactional
  public void reset(List<String> ids) {
    Validate.isTrue(ObjectUtils.isNotEmpty(ids), "请选择要重置的数据");
    Validate.isTrue(ids.size() == 1, "该操作影响较大，暂不支持批量重置");
    GenerateCodeRuleEntity entity = this.generateCodeRuleRepository.findById(ids.get(0));
    Validate.isTrue(ObjectUtils.isNotEmpty(entity), "重置的编码规则对象不存在!");
    entity.setCurrentValue(entity.getCurrentValue() - 1);
    this.generateCodeRuleRepository.saveOrUpdate(entity);
  }

  /**
   * 新增编辑数据校验
   *
   * @param dto
   */
  private void createVerify(GenerateCodeRuleDto dto) {
    Validate.isTrue(StringUtils.isNotEmpty(dto.getRuleCode()), "编码规则key不能为空");
    Validate.isTrue(ObjectUtils.isNotEmpty(dto.getCodeLength()), "长度不能为空");
    Validate.isTrue(dto.getCodeLength() <= 32, "长度不能超过32");
    Validate.isTrue(ObjectUtils.isNotEmpty(dto.getInitialValue()), "起始值不能为空");
    Validate.isTrue(dto.getInitialValue() > 0, "起始值必须大于0");
    int lengthWithoutNumber = (StringUtils.isEmpty(dto.getPrefix()) ? 0 : dto.getPrefix().length()) + (StringUtils.isEmpty(dto.getDateFormat()) ? 0 : dto.getDateFormat().length());
    Validate.isTrue(dto.getCodeLength() > lengthWithoutNumber, "编码长度必须大于前缀和时间戳长度之和");
    if (StringUtils.isNotEmpty(dto.getDateFormat())) {
      try {
        SimpleDateFormat sdf = new SimpleDateFormat(dto.getDateFormat());
      } catch (Exception e) {
        Validate.isTrue(false, "时间戳格式" + dto.getDateFormat() + "不合法");
      }
    }
  }
}
