package com.biz.crm.common.rulecode.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.common.log.sdk.dto.BusinessLogEventDto;
import com.biz.crm.common.log.sdk.utils.BusinessLogUtil;
import com.biz.crm.common.rulecode.local.entity.GenerateCodeRuleEntity;
import com.biz.crm.common.rulecode.local.repository.GenerateCodeRuleRepository;
import com.biz.crm.common.rulecode.local.service.GenerateCodeRuleService;
import com.biz.crm.common.rulecode.sdk.constant.GenerateCodeRuleConstant;
import com.biz.crm.common.rulecode.sdk.dto.GenerateCodeRuleDto;
import com.biz.crm.common.rulecode.sdk.enums.RuleCodeRedisEnum;
import com.biz.crm.common.rulecode.sdk.vo.GenerateCodeRuleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 编码规则接口实现
 *
 * @author XXLsansui
 * @date 2024-01-24 10:14
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "GenerateCodeRuleServiceExpandImpl")
public class GenerateCodeRuleServiceImpl implements GenerateCodeRuleService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private GenerateCodeRuleRepository generateCodeRuleRepository;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private RedisService redisService;

    /**
     * 分页条件查询
     *
     * @param pageable
     * @param dto
     * @return
     */
    @Override
    public Page<GenerateCodeRuleEntity> findByConditions(Pageable pageable, GenerateCodeRuleDto dto) {
        return this.generateCodeRuleRepository.findByConditions(pageable, dto);
    }

    /**
     * 查看规则详情
     *
     * @param id ID
     * @return
     */
    @Override
    public GenerateCodeRuleVo findById(String id) {
        Validate.isTrue(StringUtils.isNotEmpty(id), "请选择规则编码详情id!");
        GenerateCodeRuleEntity entity = this.generateCodeRuleRepository.findById(id);
        return this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 创建
     *
     * @param dto 请求参数dto
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public GenerateCodeRuleVo create(GenerateCodeRuleDto dto) {
        Assert.notNull(dto, "新增编码规则参数不能为空!");
        //校验新增参数
        this.baseVerify(dto);
        GenerateCodeRuleEntity oldEntity = this.generateCodeRuleRepository.findByCode(dto.getRuleCode());
        Assert.isNull(oldEntity, "编码规则[" + dto.getRuleCode() + "]已存在");

        GenerateCodeRuleEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, GenerateCodeRuleEntity.class, HashSet.class, ArrayList.class);
        entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        entity.setTenantCode(TenantUtils.getTenantCode());
        entity.setAllInfoNull();
        this.generateCodeRuleRepository.save(entity);
        GenerateCodeRuleVo newRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        this.syncToRedisVo(newRuleVo);
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewest(newRuleVo);
        BusinessLogUtil.onCreate(logDto);
        return newRuleVo;
    }

    /**
     * 更新
     *
     * @param dto 请求参数dto
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public GenerateCodeRuleVo update(GenerateCodeRuleDto dto) {
        Assert.notNull(dto, "新增编码规则参数不能为空!");
        Assert.notNull(dto.getId(), "id不能为空!");
        this.baseVerify(dto);
        GenerateCodeRuleEntity oldEntity = this.generateCodeRuleRepository.findById(dto.getId());
        Validate.isTrue(ObjectUtils.isNotEmpty(oldEntity), "当前id对应编码规则不存在!");
        if (ObjectUtils.isNotEmpty(dto.getCurrentValue())) {
            Validate.isTrue(dto.getCurrentValue() >= oldEntity.getCurrentValue(), "当前值不能改小!");
        }
        GenerateCodeRuleVo oldRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        Validate.isTrue(dto.getRuleCode().equals(oldEntity.getRuleCode()), "编码规则key不能修改");
        GenerateCodeRuleEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, GenerateCodeRuleEntity.class, HashSet.class, ArrayList.class);
        entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        entity.setTenantCode(TenantUtils.getTenantCode());
        entity.setUpdateInfoToNull();
        this.generateCodeRuleRepository.saveOrUpdate(entity);
        GenerateCodeRuleVo newRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);

        if (StringUtil.isNotEmpty(newRuleVo.getDateFormat())
                && Objects.equals(newRuleVo.getRestCurrentValueFormat(), BooleanEnum.TRUE.getCapital())) {
            String ruleCode = newRuleVo.getRuleCode();
            SimpleDateFormat sdf = new SimpleDateFormat(newRuleVo.getDateFormat());
            String dateStr = sdf.format(new Date());
            String ruleKey = String.format(RuleCodeRedisEnum.RULE_CODE_DATE_FORMAT.getVal(), ruleCode, dateStr);
            switch (newRuleVo.getDateFormat()) {
                case "yyyyMMdd":
                case "yyyyMM":
                case "yyyy":
                    break;
                default:
                    Validate.isTrue(false, "时间戳格式" + dto.getDateFormat() + "不合法");

            }
            Object currentObj = redisService.get(ruleKey);
            if (Objects.nonNull(currentObj)) {
                newRuleVo.setCurrentValue(Long.valueOf(currentObj.toString()));
            }
        }
        this.syncToRedisVo(newRuleVo);
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewest(newRuleVo);
        logDto.setOriginal(oldRuleVo);
        BusinessLogUtil.onUpdate(logDto);
        return newRuleVo;

    }


    /**
     * 批量删除编码规则
     *
     * @param ids 编码规则id集合
     */
    @Override
    @Transactional
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择要删除的编码规则");
        List<GenerateCodeRuleEntity> entityList = this.generateCodeRuleRepository.findByIds(ids);
        Assert.notEmpty(entityList, "编码规则不存在");
        List<String> ruleCodes = entityList.stream().map(GenerateCodeRuleEntity::getRuleCode).collect(Collectors.toList());
        entityList.forEach(item -> {
            Assert.isTrue(EnableStatusEnum.DISABLE.getCode().equals(item.getEnableStatus()),
                    "编码规则[" + item.getRuleCode() + "]需要先禁用,才能删除!");
            item.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            item.setRemark(item.getRuleCode());
            item.setRuleCode(item.getId());
            item.setUpdateInfoToNull();
        });
        this.generateCodeRuleRepository.updateBatchById(entityList);
        redisService.hDel(RuleCodeRedisEnum.RULE_CODE_KEY.getVal(), ruleCodes.toArray());
        redisService.hDel(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), ruleCodes.toArray());
        List<GenerateCodeRuleVo> newRuleVoList = (List<GenerateCodeRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, GenerateCodeRuleEntity.class, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewestList(newRuleVoList);
        BusinessLogUtil.onDelete(logDto);

    }

    /**
     * 批量启用
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择要启用的编码规则");
        List<GenerateCodeRuleEntity> entityList = this.generateCodeRuleRepository.findByIds(ids);
        Assert.notEmpty(entityList, "编码规则不存在");
        entityList.forEach(item -> {
            Assert.isTrue(!EnableStatusEnum.ENABLE.getCode().equals(item.getEnableStatus()), "编码规则[" + item.getRuleCode() + "]已启用");
            item.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            item.setUpdateInfoToNull();
        });
        this.generateCodeRuleRepository.updateBatchById(entityList);
        List<GenerateCodeRuleVo> newRuleVoList = (List<GenerateCodeRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, GenerateCodeRuleEntity.class, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        newRuleVoList.forEach(item -> {
            redisService.hSet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), item.getRuleCode(), item);
        });
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewestList(newRuleVoList);
        BusinessLogUtil.onEnable(logDto);

    }

    /**
     * 批量禁用
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择要禁用的编码规则");
        List<GenerateCodeRuleEntity> entityList = this.generateCodeRuleRepository.findByIds(ids);
        Assert.notEmpty(entityList, "编码规则不存在");
        entityList.forEach(item -> {
            Assert.isTrue(!EnableStatusEnum.DISABLE.getCode().equals(item.getEnableStatus()), "编码规则[" + item.getRuleCode() + "]已禁用");
            item.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            item.setUpdateInfoToNull();
        });
        this.generateCodeRuleRepository.updateBatchById(entityList);
        List<GenerateCodeRuleVo> newRuleVoList = (List<GenerateCodeRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, GenerateCodeRuleEntity.class, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        newRuleVoList.forEach(item -> {
            redisService.hSet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), item.getRuleCode(), item);
        });
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewestList(newRuleVoList);
        BusinessLogUtil.onDisable(logDto);


    }

    /**
     * 重置
     *
     * @param ids 编码规则集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void reset(List<String> ids) {
        Validate.isTrue(ObjectUtils.isNotEmpty(ids), "请选择要重置的数据");
        Validate.isTrue(ids.size() == 1, "该操作影响较大，暂不支持批量重置");
        GenerateCodeRuleEntity entity = this.generateCodeRuleRepository.findById(ids.get(0));
        Validate.isTrue(ObjectUtils.isNotEmpty(entity), "重置的编码规则对象不存在!");
        GenerateCodeRuleVo oldRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        entity.setCurrentValue(entity.getInitialValue() - 1L);
        entity.setUpdateInfoToNull();
        if (StringUtils.isNotEmpty(entity.getDateFormat())
                && BooleanEnum.TRUE.getCapital().equals(entity.getRestCurrentValueFormat())) {
            SimpleDateFormat sdf = new SimpleDateFormat(entity.getDateFormat());
            String dateStr = sdf.format(new Date());
            boolean hasKey = redisService.hasKey(RuleCodeRedisEnum.RULE_CODE_KEY.getVal() + entity.getRuleCode() + ":" + dateStr);
            Validate.isTrue(!hasKey, "时间戳格式不为空且是否按时间维度重置为是，无法重置!");
        }

        this.generateCodeRuleRepository.updateById(entity);
        GenerateCodeRuleVo newRuleVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
        this.syncToRedisVo(newRuleVo);
        BusinessLogEventDto<GenerateCodeRuleVo> logDto = new BusinessLogEventDto<>();
        logDto.setNewest(newRuleVo);
        logDto.setOriginal(oldRuleVo);
        BusinessLogUtil.onDelete(logDto);

    }

    private void syncToRedisVo(GenerateCodeRuleVo vo) {
        if (Objects.isNull(vo)) {
            return;
        }
        String ruleCode = vo.getRuleCode();
        if (StringUtils.isBlank(ruleCode)) {
            return;
        }
        if (StringUtil.isNotEmpty(vo.getDateFormat())
                && Objects.equals(vo.getRestCurrentValueFormat(), BooleanEnum.TRUE.getCapital())) {
            SimpleDateFormat sdf = new SimpleDateFormat(vo.getDateFormat());
            String dateStr = sdf.format(new Date());
            String ruleKey = String.format(RuleCodeRedisEnum.RULE_CODE_DATE_FORMAT.getVal(), ruleCode, dateStr);
            long expirationTimes;
            TimeUnit timeUnit = TimeUnit.DAYS;
            switch (vo.getDateFormat()) {
                case "yyyyMMdd":
                    expirationTimes = 25;
                    timeUnit = TimeUnit.HOURS;
                    break;
                case "yyyyMM":
                    expirationTimes = 32;
                    break;
                case "yyyy":
                    expirationTimes = 370;
                    break;
                default:
                    throw new IllegalArgumentException("编码规则[" + ruleCode + "]的时间格式不正确!");

            }
            redisService.set(ruleKey, vo.getCurrentValue(), expirationTimes, timeUnit);
            redisService.expire(ruleKey, expirationTimes, timeUnit);
        } else {
            redisService.hSet(RuleCodeRedisEnum.RULE_CODE_KEY.getVal(), ruleCode, vo.getCurrentValue());
        }
        redisService.hSet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), ruleCode, vo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void syncNumKey(GenerateCodeRuleDto dto) {
        Pageable pageable = PageRequest.of(1, 400);
        int number = 1;
        dto = Optional.ofNullable(dto).orElse(new GenerateCodeRuleDto());
        Page<GenerateCodeRuleEntity> page = null;
        String redisLock = GenerateCodeRuleConstant.RULE_CODE_SYNC_LOCK;
        boolean isLock = redisMutexService.tryLock(redisLock, 1, 1, TimeUnit.MINUTES);
        Assert.isTrue(isLock, "同步数据时加锁失败，请稍后重试");
        try {
            do {
                page = this.findByConditions(pageable, dto);
                number++;
                pageable = pageable.next();
                if (CollectionUtils.isEmpty(page.getRecords())) {
                    return;
                }
                Map<String, GenerateCodeRuleEntity> entityMap = page.getRecords().stream().collect(Collectors.toMap(GenerateCodeRuleEntity::getRuleCode, v -> v, (v1, v2) -> v1));
                List<GenerateCodeRuleVo> list = (List<GenerateCodeRuleVo>) this.nebulaToolkitService.copyCollectionByWhiteList(page.getRecords(), GenerateCodeRuleEntity.class, GenerateCodeRuleVo.class, HashSet.class, ArrayList.class);
                list.forEach(vo -> {
                    String ruleCode = vo.getRuleCode();
                    Object obj = redisService.hGet(RuleCodeRedisEnum.RULE_CODE_KEY.getVal(), ruleCode);
                    Long currentValue = vo.getCurrentValue();
                    if (obj != null) {
                        Integer redisValue = (Integer) obj;
                        if (currentValue < redisValue) {
                            currentValue = Long.valueOf(redisValue);
                            GenerateCodeRuleEntity entity = entityMap.get(ruleCode);
                            if (Objects.nonNull(entity)) {
                                entity.setCurrentValue(currentValue);
                                entity.setUpdateInfoToNull();
                                entityMap.put(ruleCode, entity);
                            }
                        }
                    }
                    redisService.hSet(RuleCodeRedisEnum.RULE_CODE_KEY.getVal(), ruleCode, currentValue);
                    vo.setCurrentValue(currentValue);

                    Object genVo = redisService.hGet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), ruleCode);
                    if (genVo == null) {
                        redisService.hSet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), ruleCode, vo);
                    } else {
                        try {
                            GenerateCodeRuleVo oldVo = (GenerateCodeRuleVo) genVo;
                            GenerateCodeRuleEntity entity = entityMap.get(ruleCode);
                            if (Objects.nonNull(entity)) {
                                entity.setGenerateDate(oldVo.getGenerateDate());
                                if (StringUtil.isEmpty(entity.getRuleModule())) {
                                    entity.setRuleModule(oldVo.getRuleModule());
                                } else if (!entity.getRuleModule().equals(oldVo.getRuleModule())) {
                                    entity.setRuleModule("crm-sys");
                                }
                                vo.setGenerateDate(entity.getGenerateDate());
                                vo.setRuleModule(entity.getRuleModule());

                                entity.setUpdateInfoToNull();
                                entityMap.put(ruleCode, entity);
                            }
                        } catch (Exception e) {
                            //结构变化
                            log.error(e.getMessage(), e);
                        }
                        redisService.hSet(RuleCodeRedisEnum.RULE_CODE_VO.getVal(), ruleCode, vo);
                    }
                });
                this.generateCodeRuleRepository.updateBatchById(new ArrayList<>(entityMap.values()));
            } while (page.hasNext() && number <= 50);
        } finally {
            redisMutexService.unlock(redisLock);
        }

    }

    /**
     * 新增编辑数据校验
     *
     * @param dto
     */
    private void baseVerify(GenerateCodeRuleDto dto) {
        Validate.isTrue(StringUtils.isNotEmpty(dto.getRuleCode()), "编码规则key不能为空");
        Validate.isTrue(ObjectUtils.isNotEmpty(dto.getCodeLength()), "长度不能为空");
        if (StringUtil.isNotEmpty(dto.getPrefix())) {
            String oldPrefix = dto.getPrefix();
            dto.setPrefix(oldPrefix.replaceAll("\\s+", ""));
            Assert.isTrue(oldPrefix.equals(dto.getPrefix()), "前缀[" + oldPrefix + "]不合法");
        }
        if (StringUtil.isNotEmpty(dto.getDateFormat())) {
            switch (dto.getDateFormat()) {
                case "yyyyMMdd":
                case "yyyyMM":
                case "yyyy":
                    break;
                default:
                    Validate.isTrue(false, "时间戳格式" + dto.getDateFormat() + "不合法");

            }
            try {
                new SimpleDateFormat(dto.getDateFormat());
            } catch (Exception e) {
                Validate.isTrue(false, "时间戳格式" + dto.getDateFormat() + "不合法");
            }
            String oldDateFormat = dto.getDateFormat();
            dto.setDateFormat(oldDateFormat.replaceAll("[^a-zA-Z]", ""));
            Assert.isTrue(oldDateFormat.equals(dto.getDateFormat()), "时间戳格式[" + oldDateFormat + "]不合法");
            Assert.hasLength(dto.getRestCurrentValueFormat(), "是否按时间维度重置当前值[是/否]不能为空");
            Assert.isTrue(BooleanEnum.FALSE.getCapital().equals(dto.getRestCurrentValueFormat())
                            || BooleanEnum.TRUE.getCapital().equals(dto.getRestCurrentValueFormat()),
                    "是否按时间维度重置当前值不合法");
            String dateFormat = dto.getDateFormat();
            Assert.isTrue(GenerateCodeRuleConstant.DATE_FORMAT_SET.contains(dateFormat), "时间戳格式目前只支持" + GenerateCodeRuleConstant.DATE_FORMAT_SET);
        } else {
            Assert.isTrue(BooleanEnum.FALSE.getCapital().equals(dto.getRestCurrentValueFormat()),
                    "时间戳格式为空时,是否按时间维度重置必须是[否]");
            dto.setRestCurrentValueFormat(BooleanEnum.FALSE.getCapital());
        }
        Validate.isTrue(dto.getCodeLength() <= 32, "长度不能超过32");
        Validate.isTrue(ObjectUtils.isNotEmpty(dto.getInitialValue()), "                  ");
        Validate.isTrue(dto.getInitialValue() > 0, "起始值必须大于0");
        int lengthWithoutNumber = (StringUtils.isEmpty(dto.getPrefix()) ? 0 : dto.getPrefix().length()) + (StringUtils.isEmpty(dto.getDateFormat()) ? 0 : dto.getDateFormat().length());
        Validate.isTrue(dto.getCodeLength() > lengthWithoutNumber, "编码长度必须大于前缀和时间戳长度之和");

    }
}
