package com.biz.crm.common.weixinsign.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.weixinsign.local.config.WXPlatformAccountProperties;
import com.biz.crm.common.weixinsign.local.config.WXAccountProperties;
import com.biz.crm.common.weixinsign.sdk.common.constant.WXOpenConstant;
import com.biz.crm.common.weixinsign.sdk.common.enums.WXPlatformTypeEnum;
import com.biz.crm.common.weixinsign.sdk.service.WXCacheVoService;
import com.biz.crm.common.weixinsign.sdk.service.WXOpenVoService;
import com.biz.crm.common.weixinsign.sdk.vo.WXJsConfigVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxAccessTokenRespVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxCacheVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxJsApiTicketRespVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxUserInfoVo;
import com.biz.crm.common.weixinsign.sdk.vo.WxWebViewAccessTokenRespVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

/**
 * @author hecheng
 * @description: 微信公众号 接
 * @date 2021/9/8 下午2:59
 */
@Service
@Slf4j
public class WXOpenVoServiceImpl implements WXOpenVoService {

  @Autowired
  private WXAccountProperties wxAccountProperties;

  @Autowired
  @Qualifier("restTemplateRemote")
  private RestTemplate restTemplate;

  @Autowired
  private WXCacheVoService wxCache;

  @Override
  public String findAccessToken(String appType) {
    /**
     * 1从缓存中获取是否
     * 2如果缓存中没有直接查询
     * 3存入缓存
     */
    if (Objects.isNull(this.wxAccountProperties) || CollectionUtils.isEmpty(this.wxAccountProperties.getAccounts())) {
      return null;
    }
    if (StringUtils.isBlank(appType)) {
      appType = WXPlatformTypeEnum.OFFICIAL_ACCOUNT.getCode();
    }
    String finalAppType = appType;
    WXPlatformAccountProperties wxProperties = this.wxAccountProperties.getAccounts().stream().filter(item -> Objects.equals(item.getAppType(), finalAppType)).findFirst().orElse(null);
    if (Objects.isNull(wxProperties)) {
      return null;
    }
    log.info("wx:accessToken:wxProperties:{}", wxProperties.getAppId());
    String cacheKey = String.format(WXOpenConstant.OPEN_CACHE_ACCESS_TOKEN_FORMAT, wxProperties.getAppId());
    String accessToken = this.wxCache.get(cacheKey);
    if (StringUtils.isNotBlank(accessToken)) {
      log.debug("accessToken:{}", accessToken);
      return accessToken;
    }
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("appid", wxProperties.getAppId());
    uriVariables.put("secret", wxProperties.getAppSecret());
    WxAccessTokenRespVo accessTokenRespVo = this.restTemplate
            .getForObject(WXOpenConstant.ACCESS_TOKEN_URL + "?grant_type=client_credential&appid={appid}&secret={secret}", WxAccessTokenRespVo.class, uriVariables);
    if (Objects.isNull(accessTokenRespVo) || StringUtils.isNotBlank(accessTokenRespVo.getErrcode())) {
      log.warn("获取微信全局ACCESS_TOKEN失败！{}", accessTokenRespVo.getErrmsg());
      return null;
    }
    accessToken = accessTokenRespVo.getAccessToken();
    WxCacheVo cacheVo = new WxCacheVo();
    cacheVo.setCreateTimestamp(new Date().getTime());
    cacheVo.setValue(accessToken);
    cacheVo.setExpiresIn(accessTokenRespVo.getExpiresIn());
    this.wxCache.set(cacheKey, JSONObject.toJSONString(cacheVo));
    log.debug("accessToken:{}", accessToken);
    return accessToken;
  }

  @Override
  public String findJsApiTicket(String appType) {
    /**
     * 1从缓存中获取是否
     * 2如果缓存中没有直接查询
     * 3存入缓存
     */
    if (Objects.isNull(this.wxAccountProperties)) {
      return null;
    }
    if (StringUtils.isBlank(appType)) {
      appType = WXPlatformTypeEnum.OFFICIAL_ACCOUNT.getCode();
    }
    String finalAppType = appType;
    WXPlatformAccountProperties wxProperties = this.wxAccountProperties.getAccounts().stream().filter(item -> Objects.equals(item.getAppType(), finalAppType)).findFirst().orElse(null);
    if (Objects.isNull(wxProperties)) {
      return null;
    }
    log.info("wx:jsApiTicket:wxProperties:{}", wxProperties.getAppId());
    String cacheKey = String.format(WXOpenConstant.OPEN_CACHE_JSAPI_TICKET_FORMAT, wxProperties.getAppId());
    String jsApiTicket = this.wxCache.get(cacheKey);
    if (StringUtils.isNotBlank(jsApiTicket)) {
      log.debug("jsApiTicket:{}", jsApiTicket);
      return jsApiTicket;
    }
    String accessToken = this.findAccessToken(appType);
    if (StringUtils.isBlank(accessToken)) {
      return null;
    }
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("access_token", accessToken);
    uriVariables.put("type", "jsapi");
    WxJsApiTicketRespVo jsApiTicketRespVo = this.restTemplate.getForObject(WXOpenConstant.JSAPI_TICKET_URL + "?access_token={access_token}&type={type}", WxJsApiTicketRespVo.class, uriVariables);
    if (Objects.isNull(jsApiTicketRespVo) || !Objects.equals(jsApiTicketRespVo.getErrcode(), WXOpenConstant.SUCCESS)) {
      log.warn("获取微信全局JSAPI_TICKET失败！{}", jsApiTicketRespVo.getErrmsg());
      return null;
    }
    jsApiTicket = jsApiTicketRespVo.getTicket();
    WxCacheVo cacheVo = new WxCacheVo();
    cacheVo.setCreateTimestamp(new Date().getTime());
    cacheVo.setValue(jsApiTicket);
    cacheVo.setExpiresIn(jsApiTicketRespVo.getExpiresIn());
    this.wxCache.set(cacheKey, JSONObject.toJSONString(cacheVo));
    log.debug("jsApiTicket:{}", jsApiTicket);
    return jsApiTicket;
  }

  @Override
  public WXJsConfigVo findWXJsConfig(String url, String appType) {
    if (Objects.isNull(this.wxAccountProperties)) {
      return null;
    }
    if (StringUtils.isBlank(appType)) {
      appType = WXPlatformTypeEnum.OFFICIAL_ACCOUNT.getCode();
    }
    String finalAppType = appType;
    WXPlatformAccountProperties wxProperties = this.wxAccountProperties.getAccounts().stream().filter(item -> Objects.equals(item.getAppType(), finalAppType)).findFirst().orElse(null);
    if (Objects.isNull(wxProperties)) {
      return null;
    }
    if (StringUtils.isBlank(url)) {
      url = wxProperties.getDefaultPageUrl();
    }
    Validate.notBlank(url, "获取jssdk的url不能为空");
    String jsapiTicket = this.findJsApiTicket(appType);
    if (StringUtils.isBlank(jsapiTicket)) {
      return null;
    }
    //
    String nonceStr = UUID.randomUUID().toString();
    long timestamp = System.currentTimeMillis() / 1000;
    String signOriginal = String.format(WXOpenConstant.SIGN_FORMAT, jsapiTicket, nonceStr, timestamp, url);
    log.info("wx:signOriginal:{}", signOriginal);
    String signature = null;
    try {
      MessageDigest crypt = MessageDigest.getInstance("SHA-1");
      crypt.reset();
      crypt.update(signOriginal.getBytes(StandardCharsets.UTF_8.name()));
      signature = this.byteToHex(crypt.digest());
    } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
      log.error(e.getMessage(), e);
    }
      Validate.notBlank(signature, "获取jssdk时签名失败");
    WXJsConfigVo vo = new WXJsConfigVo();
    vo.setAppId(wxProperties.getAppId());
    vo.setNonceStr(nonceStr);
    vo.setSignature(signature);
    vo.setTimestamp(timestamp);
    return vo;
  }

  @Override
  public String getAuthorizeUrl(String redirectUri, String scope, String appType) {
    if (Objects.isNull(this.wxAccountProperties) || CollectionUtils.isEmpty(this.wxAccountProperties.getAccounts())) {
      return null;
    }
    if (StringUtils.isBlank(appType)) {
      appType = WXPlatformTypeEnum.OFFICIAL_ACCOUNT.getCode();
    }
    String finalAppType = appType;
    WXPlatformAccountProperties wxProperties = this.wxAccountProperties.getAccounts().stream().filter(item -> Objects.equals(item.getAppType(), finalAppType)).findFirst().orElse(null);
    if (Objects.isNull(wxProperties)) {
      return null;
    }
    String appid = wxProperties.getAppId();
    String urlEncodeRedirectUri = null;
    try {
      urlEncodeRedirectUri = URLEncoder.encode(redirectUri, StandardCharsets.UTF_8.name());
    } catch (UnsupportedEncodingException e) {
      throw new IllegalArgumentException("redirectUri不能为空！");
    }
    String authorizeUrl = String.format(WXOpenConstant.AUTHORIZE_URL_FORMAT, appid, urlEncodeRedirectUri, scope);
    return authorizeUrl;
  }

  @Override
  public WxWebViewAccessTokenRespVo findWebViewAccessToken(String code, String appType) {
    if (Objects.isNull(this.wxAccountProperties) || CollectionUtils.isEmpty(this.wxAccountProperties.getAccounts())) {
      return null;
    }
    if (StringUtils.isBlank(appType)) {
      appType = WXPlatformTypeEnum.OFFICIAL_ACCOUNT.getCode();
    }
    String finalAppType = appType;
    WXPlatformAccountProperties wxProperties = this.wxAccountProperties.getAccounts().stream().filter(item -> Objects.equals(item.getAppType(), finalAppType)).findFirst().orElse(null);
    if (Objects.isNull(wxProperties)) {
      return null;
    }
    if(StringUtils.isBlank(code)){
      log.warn("获取网页授权access_token失败!code为空");
      return null;
    }
    Map<String, String> uriVariables = new HashMap<>();
    uriVariables.put("appid", wxProperties.getAppId());
    uriVariables.put("secret", wxProperties.getAppSecret());
    uriVariables.put("code", code);
    String jsonStr = this.restTemplate
            .getForObject(WXOpenConstant.WEB_VIEW_ACCESS_TOKEN_URL + "?appid={appid}&secret={secret}&code={code}&grant_type=authorization_code", String.class, uriVariables);
    WxWebViewAccessTokenRespVo accessTokenRespVo = JSONObject.parseObject(jsonStr,WxWebViewAccessTokenRespVo.class);
    if (Objects.isNull(accessTokenRespVo) || StringUtils.isNotBlank(accessTokenRespVo.getErrcode())) {
      log.warn("获取网页授权access_token失败！{}", accessTokenRespVo.getErrmsg());
      return null;
    }
    log.info("获取网页授权access_token！{}", JSONObject.toJSONString(accessTokenRespVo));
    return accessTokenRespVo;
  }

  /**
   * 获取微信用户信息
   *
   * @param openId 用户的唯一标识
   * @param accessToken 网页授权接口调用凭证,注意：此access_token与基础支持的access_token不同
   * @param lang 返回国家地区语言版本，zh_CN 简体，zh_TW 繁体，en 英语
   * @return
   */
  @Override
  public WxUserInfoVo findWxUserInfo(String openId, String accessToken, String lang) {
    if (StringUtils.isAnyBlank(openId, accessToken)) {
      return null;
    }
    if (StringUtils.isBlank(lang)) {
      lang = "zh_CN";
    }
    String jsonStr= this.restTemplate.getForObject(String.format(WXOpenConstant.FIND_USER_INFO, accessToken, openId, lang), String.class);
    log.info("获取微信用户信息jsonStr：{}", jsonStr);
    WxUserInfoVo wxUserInfo = JSONObject.parseObject(jsonStr, WxUserInfoVo.class);
    if (Objects.isNull(wxUserInfo) || StringUtils.isNotBlank(wxUserInfo.getErrcode())) {
      log.warn("获取微信用户信息失败！{}", wxUserInfo.getErrmsg());
      return null;
    }
    return wxUserInfo;
  }

  /**
   * 进制转换
   *
   * @param hash
   * @return
   */
  private String byteToHex(final byte[] hash) {
    Formatter formatter = new Formatter();
    for (byte b : hash) {
      formatter.format("%02x", b);
    }
    String result = formatter.toString();
    formatter.close();
    return result;
  }
}
